/*
 * Copyright (C) 2011 OgakiSoft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ogakisoft.android.gestureime;

import java.util.HashMap;
import java.util.Map;

import ogakisoft.android.gestureime.R;
import ogakisoft.android.gesture.reform.GestureView;
import ogakisoft.android.util.LOG;
import ogakisoft.android.util.Utils;
import android.content.SharedPreferences;
import android.content.SharedPreferences.OnSharedPreferenceChangeListener;
import android.content.res.Resources;
import android.os.Bundle;
import android.preference.CheckBoxPreference;
import android.preference.PreferenceActivity;
import android.preference.PreferenceManager;

/**
 * Preference Screen
 * 
 * @author noritoshi ogaki
 * @version 1.0
 */
public class Settings extends PreferenceActivity implements
		OnSharedPreferenceChangeListener {
	private static final String TAG = "Settings";
	private static final Float DEFAULT_GESTURE_STROKE_WIDTH = 7.0f;
	private static final Float DEFAULT_GESTURE_STROKE_LENGTH = 50.0f;
	private static final Boolean DEFAULT_GESTURE_FADE_ENABLED = false;
	private static final Float DEFAULT_GESTURE_FADE_OFFSET = 10000.0f; // 10s
	private static final Boolean DEFAULT_GESTURE_MESH_ENABLED = true;
	private static Map<String, Object> DefaultValues;

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		final SharedPreferences sharedPreferences = PreferenceManager
				.getDefaultSharedPreferences(Settings.this);

		super.onCreate(savedInstanceState);
		final Resources resources = getResources();
		final String ID_GESTURE_STROKE_WIDTH = resources
				.getString(R.string.pref_gesture_stroke_width_id);
		final String ID_GESTURE_STROKE_LENGTH = resources
				.getString(R.string.pref_gesture_stroke_lengththreshold_id);
		final String ID_GESTURE_FADE_ENABLED = resources
				.getString(R.string.pref_fadeenabled_id);
		final String ID_GESTURE_FADE_OFFSET = resources
				.getString(R.string.pref_fadeoffset_id);
		final String ID_GESTURE_MESH_ENABLED = resources
				.getString(R.string.pref_meshenabled_id);

		DefaultValues = new HashMap<String, Object>();
		DefaultValues
				.put(ID_GESTURE_STROKE_WIDTH, DEFAULT_GESTURE_STROKE_WIDTH);
		DefaultValues.put(ID_GESTURE_STROKE_LENGTH,
				DEFAULT_GESTURE_STROKE_LENGTH);
		DefaultValues
				.put(ID_GESTURE_FADE_ENABLED, DEFAULT_GESTURE_FADE_ENABLED);
		DefaultValues.put(ID_GESTURE_FADE_OFFSET, DEFAULT_GESTURE_FADE_OFFSET);
		DefaultValues
				.put(ID_GESTURE_MESH_ENABLED, DEFAULT_GESTURE_MESH_ENABLED);
		addPreferencesFromResource(R.xml.preferences);
		onSharedPreferenceChanged(sharedPreferences, null);
	}

	@Override
	protected void onResume() {
		super.onResume();
		getPreferenceScreen().getSharedPreferences()
				.registerOnSharedPreferenceChangeListener(this);
	}

	@Override
	protected void onPause() {
		super.onPause();
		getPreferenceScreen().getSharedPreferences()
				.unregisterOnSharedPreferenceChangeListener(this);
	}

	public void onSharedPreferenceChanged(SharedPreferences sp, String key) {
		String str = null;
		String entryKey = null;
		Object entryVal = null;
		// Update the current value of the summary-text & check-box state
		for (Map.Entry<String, Object> entry : DefaultValues.entrySet()) {
			entryKey = entry.getKey();
			entryVal = entry.getValue();
			// LOG.d(TAG, "onSharedPreferenceChanged: key={0}, val={1}",
			// entryKey,
			// entryVal);
			if (null == key || key.equals(entryKey)) {
				if (entryVal instanceof Float) {
					final float value = getFloatValue(sp, entryKey,
							(Float) entryVal);
					str = String.valueOf(value);
					final EditDigitPreference ed = (EditDigitPreference) findPreference(entryKey);
					ed.setDefaultValue(str);
				} else if (entry.getValue() instanceof Boolean) {
					final boolean value = getBooleanValue(sp, entryKey,
							(Boolean) entryVal);
					if (value) {
						str = getResources().getString(
								R.string.pref_boolean_true);
					} else {
						str = getResources().getString(
								R.string.pref_boolean_false);
					}
					final CheckBoxPreference cb = (CheckBoxPreference) findPreference(entryKey);
					cb.setChecked(value);
				}
				findPreference(entryKey).setSummary(str);
			}
		}
	}

	public static void setFromPreferences(Resources r, SharedPreferences sp,
			GestureView view) {
		final String ID_GESTURE_STROKE_WIDTH = r
				.getString(R.string.pref_gesture_stroke_width_id);
		final String ID_GESTURE_STROKE_LENGTH = r
				.getString(R.string.pref_gesture_stroke_lengththreshold_id);
		final String ID_GESTURE_FADE_ENABLED = r
				.getString(R.string.pref_fadeenabled_id);
		final String ID_GESTURE_FADE_OFFSET = r
				.getString(R.string.pref_fadeoffset_id);
		final String ID_GESTURE_MESH_ENABLED = r
				.getString(R.string.pref_meshenabled_id);

		view.setGestureStrokeWidth(getFloatValue(sp, ID_GESTURE_STROKE_WIDTH,
				DEFAULT_GESTURE_STROKE_WIDTH));
		view.setGestureStrokeLengthThreshold(getFloatValue(sp,
				ID_GESTURE_STROKE_LENGTH, DEFAULT_GESTURE_STROKE_LENGTH));
		view.setFadeEnabled(sp.getBoolean(ID_GESTURE_FADE_ENABLED,
				DEFAULT_GESTURE_FADE_ENABLED));
		view.setFadeOffset(Utils.castLong(getFloatValue(sp,
				ID_GESTURE_FADE_OFFSET, DEFAULT_GESTURE_FADE_OFFSET)));
		view.setMeshEnabled(sp.getBoolean(ID_GESTURE_MESH_ENABLED,
				DEFAULT_GESTURE_MESH_ENABLED));
	}

	private static float getFloatValue(SharedPreferences sp, String key,
			Float defaultValue) {
		final String str = sp.getString(key,
				String.valueOf(defaultValue.floatValue()));
		float value = 0f;
		try {
			value = Float.parseFloat(str);
			// LOG.d(TAG, "getFloatValue: key={0}, val={1,number,#}", key,
			// value);
		} catch (NumberFormatException e) {
			LOG.e(TAG, "getFloatValue: {0}", e.getMessage());
		}
		return value;
	}

	private static boolean getBooleanValue(SharedPreferences sp, String key,
			Boolean defaultValue) {
		final boolean value = sp.getBoolean(key, defaultValue);
		// LOG.d(TAG, "getBooleanValue: key={0}, val={1}", key, value);
		return value;
	}
}
