#ifndef GINTENLIB_INCLUDED_CAST_HPP_
#define GINTENLIB_INCLUDED_CAST_HPP_

/*

      <gintenlib/cast.hpp>

  cast ： 明示的な暗黙キャスト

  宣言＆定義：
    // 引数をそのまま返す
    template<typename T>
    inline T cast( typename call_traits<T>::param_type x ) { return x; }

  機能：
    暗黙変換を明示的に行うためのキャストです。 cast<void*>( 0 ) のように使います。
    主に多重定義された関数の呼び出しを制御するときに使います。
    Boost にも implicit_cast という同機能のキャストがありますが、
    こちらのほうが文字数が少ないという点で利点があります。

  使用例：
    void hoge( int ){ cout << "int 版です\n"; }
    void hoge( void* ){ cout << "void* 版です\n"; }
    
    // クラスに対して
    // noncopyable なクラスにしておく
    class base : boost::noncopyable {};
    struct derived : base {};
    
    void fuga( base& ){ cout << "base& 版です\n"; }
    void fuga( derived& ){ cout << "derived& 版です\n"; }

    int main()
    {
      // 予め using 宣言すれば四文字で使えて幸せ
      using gintenlib::cast;
      
      hoge( 0 ); // int 版
      hoge( cast<void*>(0) ); // void* 版。static_cast<void*>(0) でもＯＫだが長い
      
      derived x;  // あるいは base x;
      fuga( x );  // デフォルト呼び出し
      fuga( cast<base&>(x)    );  // base& 版。
      fuga( cast<derived&>(x) );  // derived& 版。base x; の場合はコンパイルエラー
                                  // static_cast だと、コンパイルエラーにならない
    }

  補足：
    ・暗黙変換を明示的にやって何が楽しいかといえば・・・使用例を見てください。これが全てです。
    ・このように、関数の多重定義やテンプレートに対し、正しいものを指定するときに使えます。
    ・static_cast との違いは、 Source の型が暗黙的に Target に変換できる時だけキャストできる点。
    ・まぁ普通は static_cast でも十分なんですが、クラスの継承とかを使うと、やっぱり危険なので。
    ・少し前までの実装は boost::enable_if を使っていましたが、修正されました。
    ・boost::call_traits さえ動けば動きます。

*/

#include "call_traits.hpp"

namespace gintenlib
{
  // 暗黙キャスト関数
  template<typename T>
  inline T cast( typename call_traits<T>::param_type x )
  {
    return x;
  }

}   // namespace gintenlib


#endif  // #ifndef GINTENLIB_INCLUDED_CAST_HPP_
