#ifndef GINTENLIB_INCLUDED_D_ENABLE_IF_HPP_
#define GINTENLIB_INCLUDED_D_ENABLE_IF_HPP_

/*

      <gintenlib/d_enable_if.hpp>

  d_enable_if ： 引数用に boost::enable_if の使い勝手を少しだけ改善

  宣言：
    struct dummy_arg_t {};
    
    template<typename Cond>
    struct  d_enable_if  : enable_if<Cond, dummy_arg_t> {};
    template<typename Cond>
    struct d_disable_if : disable_if<Cond, dummy_arg_t> {};
    
  使用例：
    // gintenlib::cast の実装がいい使用例なのでそれで（注：昔の実装、今は enable_if は使ってません）
    using namespace gintenlib;  // gintenlib::d_enable_if とか書くの正直めんどい
    
    // 引数を Target 型へ暗黙変換する
    // 通常版、Target が参照型でなければこっちが呼ばれる
    template< typename Target, typename Source >
    inline Target cast( const Source& src,
      typename d_disable_if< boost::is_reference<Target> >::type = dummy_arg_t() )
    {
      return src;
    }
    
    // 参照版
    template< typename Target, typename Source >
    inline Target cast( Source& src,
      typename d_enable_if<  boost::is_reference<Target> >::type = dummy_arg_t() )
    {
      return src;
    }
    
    // なぜこう書くか、ちょっとだけ補足。
    // 見たとおり第二引数はダミーで、こう書かなくても常套手段として
    //   typename boost::disable_if< boost::is_reference<Target> >::type* = 0
    // と記述することも出来るのだが、この場合、うっかり
    //   cast<なんたら>( x, 0 );
    // とか呼び出した場合にも（エラーになるべきはずが）コンパイルが通ってしまう。
    // コンパイル通るだけならまだしも、もし cast が２引数バージョンで多重定義されていた場合、
    // （たとえば cast<Src>( Target x, int param ) とか）
    // cast<Src>( x, 1 ) ならコンパイル通るけど cast<Src>( x, 0 ) だと通らないよ、
    // とかいう意味不明な事態になって混乱する恐れがある。
    // d_disable_if があるのはそのため、専用のクラスを用意してるので
    // うっかり cast<Target>( x, 0 ) とか呼んでもエラーになるので安心

  機能：
    上記で書いたように、関数に対し enable_if を使う場合に、
    ダミーとなる型が存在していると楽なので、 dummy_arg_t を予め用意しておきます
    また、その都度ダミー型を enable_if の第二引数で指定するのは面倒なので、その手間も省きます。
    ちなみに d_enable_if の d は dummy の略。

*/

#include "enable_if.hpp"

namespace gintenlib
{
  // enable_if 用ダミー引数クラス
  struct dummy_arg_t {};
  
  // ダミー引数版 enable_if
  template<typename Cond>
  struct  d_enable_if  : enable_if<Cond, dummy_arg_t> {};
  template<typename Cond>
  struct d_disable_if : disable_if<Cond, dummy_arg_t> {};

}   // namespace gintenlib


#endif  // #ifndef GINTENLIB_INCLUDED_D_ENABLE_IF_HPP_
