#ifndef GINTENLIB_INCLUDED_INTRUSIVE_TO_SHARED_HPP_
#define GINTENLIB_INCLUDED_INTRUSIVE_TO_SHARED_HPP_

/*

      <gintenlib/intrusive_to_shared.hpp>

  intrusive_to_shared ： intrusive_ptr を shared_ptr に変換

  宣言：
    template<typename T>
    shared_ptr<T> intrusive_to_shared( const boost::intrusive_ptr<T>& p );
    
    template<typename T>
    shared_ptr<T> intrusive_to_shared( T* p, bool add_ref = true );
    
    template<typename T>
    shared_ptr<T> to_shared( const boost::intrusive_ptr<T>& p );

  機能：
    intrusive_ptr を shared_ptr に変換します。
    intrusive_ptr_add_ref や intrusive_ptr_release のコストが高い場合や、
    あるいは単に shared_ptr としての値が欲しい場合などに使えます。
    なおヘッダ名は intrusive_to_shared ですが、省略名である to_shared でも使えます。
    むしろ <gintenlib/to_shared.hpp> との兼ね合いから、こちらを使うことを推奨しています。

*/

#include "shared_ptr.hpp"

#include <cassert>
#include <boost/intrusive_ptr.hpp>

namespace gintenlib
{
  // deleter として使う補助ファンクタ
  struct intrusive_ptr_releaser
  {
    typedef void result_type;
    
    // ptr をリリースする
    template<typename T>
    void operator()( T* ptr ) const
    {
      using namespace std;
      using namespace boost;  // for intrusive_ptr_release
      
      assert( ptr != 0 );
      intrusive_ptr_release(ptr);
    }
  
  };  // class intrusive_ptr_releaser
  
  // 本体
  // intrusive_ptr 版
  template<typename T>
  inline shared_ptr<T> intrusive_to_shared( const boost::intrusive_ptr<T>& p )
  {
    // 生ポインタ版に単純に転送するだけ
    return intrusive_to_shared( p.get(), true );
  }
  // 生ポインタ版
  template<typename T>
  inline shared_ptr<T> intrusive_to_shared( T* p, bool add_ref = true )
  {
    using namespace boost;  // for intrusive_ptr_add_ref
    
    if(p)
    {
      // 参照カウント増やしてから shared_ptr に突っ込む
      // boost::shared_ptr はコンストラクタで例外を投げうるが、その場合でも
      // ちゃんと削除ファンクタを呼び出してくれるので問題ない
      if( add_ref ){ intrusive_ptr_add_ref( p ); }
      return shared_ptr<T>( p, intrusive_ptr_releaser() );
    }
    
    // NULL の場合は空の shared_ptr を返す
    return shared_ptr<T>();
  }
  
  // 省略名称
  // 他の場所で to_shared の一般的なスマートポインタに対する定義もあるが、
  // intrusive_ptr に関してはこちらの方が効率が良い。
  template<typename T>
  inline shared_ptr<T> to_shared( const boost::intrusive_ptr<T>& p )
  {
    // 単純に転送するだけ
    return intrusive_to_shared( p );
  }

}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_INTRUSIVE_TO_SHARED_HPP_ 
