#ifndef GINTENLIB_INCLUDED_LEXICAL_CAST_DEFAULT_HPP_
#define GINTENLIB_INCLUDED_LEXICAL_CAST_DEFAULT_HPP_

/*
      <gintenlib/lexical_cast_default.hpp>

  lexical_cast_default ： 例外を投げない boost::lexical_cast

  宣言：
    template< typename Target, typename Source >
    Target lexical_cast_default( const Source& src );

    template< typename Target, typename Source >
    Target lexical_cast_default( const Source& src, const Target& default_value );

  機能：
    boost::lexical_cast と同様に、文字列表現を仲立ちに型変換を行います。
    boost::lexical_cast との違いは、失敗した場合に例外を投げず default_value を返す点。
    第二引数を省略した場合には、デフォルト初期化値が返ります。

  使用例：
    // hoge は数値を表しているかもしれない「文字列に変換できる何か」
    int i = gintenlib::lexical_cast_default<int>( hoge, -1 ),
      j = gintenlib::lexical_cast_default<int>( hoge );
    // hoge が数値を表していなければ、i = -1, j = 0（intの初期値）になる

  補足：
    ・基本的に boost::lexical_cast の例外処理が面倒な場合に使います。
    ・default_valueと同じ値が返ったとしても、キャストに失敗したとは限らないので要注意。
      （変換元の表現する値が、たまたま default_value と同じだったのかもしれないので）

*/

#include <boost/lexical_cast.hpp>

namespace gintenlib
{
  // src を文字列に変換し、その文字列を Target 型に変換
  // 失敗した場合には default_value が返る
  template< typename Target, typename Source >
  Target lexical_cast_default( const Source& src, const Target& default_value )
  {
    using namespace boost;

    try
    {
      return lexical_cast<Target>( src );
    }
    catch( bad_lexical_cast )
    {
      return default_value;
    }
    
  }   // lexical_cast_default

  // src を文字列に変換し、その文字列を Target 型に変換
  // 失敗した場合には Target() が返る
  template< typename Target, typename Source >
  Target lexical_cast_default( const Source& src )
  {
    using namespace boost;

    try
    {
      return lexical_cast<Target>( src );
    }
    catch( bad_lexical_cast )
    {
      return Target();
    }
    
  }   // lexical_cast_default

}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_LEXICAL_CAST_DEFAULT_HPP_
