#ifndef GINTENLIB_INCLUDED_LIST_FORMATTER_HPP_
#define GINTENLIB_INCLUDED_LIST_FORMATTER_HPP_

/*

      <gintenlib/list_formatter.hpp>

  list_formatter ： ファンクタ版 list_format（const char* 版）

  宣言：
  template< typename Elem >
  struct basic_list_formatter
  {
    explicit basic_list_formatter
      ( 
        gintenlib_param_t(Elem) pre_ = "( ",
        gintenlib_param_t(Elem) delim_ = ", ",
        gintenlib_param_t(Elem) post_ = " )" 
      );

    typedef gintenlib_value_t(Elem) element_type;
    // メンバ「変数」、自由に外部からアクセスできる
    element_type pre, delim, post;

    // 出力範囲を指定し、ストリームに流せるようにする
    template< typename InIte >
      list_format_impl_< InIte, Elem, Elem, Elem >
    operator()( const InIte& begin, const InIte& end ) const;

    // 出力範囲を range で指定
    template< typename Range >
      list_format_impl_< gintenlib_const_ite_of(Range), Elem, Elem, Elem >
    operator()( const Range& src ) const;

    // ostream_iterator のように振舞うイテレータを製作。
    // イテレータが既に製作されていた場合は、いったんリストが閉じられる
    template< typename Stream >
    iterator<Stream> make_iterator( Stream& os ) const;

    template< typename Stream >
    struct iterator
      : std::iterator< std::output_iterator_tag, void, void, void, void >
    {
      iterator( const basic_list_formatter& formatter, Stream& os );
      iterator( const iterator& src );

      ~iterator();

      iterator& operator=( const iterator& src );
      iterator& swap( iterator& other );

      template< typename T >
      iterator& operator=( const T& val );

      iterator& close();

      iterator& operator*(){ return *this; }
      iterator& operator++(){ return *this; }
      iterator& operator++(int){ return *this; }

    };  // struct iterator

  };  // struct basic_list_formatter

  typedef basic_list_formatter<const char*> list_formatter;

  機能：
    list_format(v) や list_format( begin, end ) と使う部分で、代わりに使用できるファンクタ。

    list_formatter formatter( pre, delim, post );
    cout << formatter(v) << endl;

    このコードの実行結果は cout << list_format( v, pre, delim, post ) << endl; と等価。
    予め pre, delim, post の設定をしておいて使いまわしたい場合に使用できる。

    もう一つの機能として、make_iterator() メンバ関数を呼ぶことで出力イテレータを構築できる機能もあり、

    cout << formatter(v);
    cout << formatter( v.begin(), v.end() );
    copy( v.begin(), v.end(), formatter.make_iterator(cout) );

    この三つの文の実行結果は、どれも等しい。

  使用例：
    // 構築して
    gintenlib::list_formatter formatter( "[ ", " | ", " ]\n" );

    // 任意のコンテナ v に対し、適用する
    cout << formatter(v);
    // 逆順に表示をしてみる
    cout << formatter( v.rbegin(), v.rend() );
    // 逆順に表示するには、こう書いてもよい
    reverse_copy( v.begin(), v.end(), formatter.ite(cout) );

  補足：
    ・pre, delim, post は const char* でなければならない。
      任意要素を格納したい場合は、list_formatter_ex を用いる。
    ・operator() の適用結果の取り扱いは、 list_format 関数の戻り値と同じように扱う。
    ・出力イテレータを使用する場合の注意事項：
      ・pre 要素は「最初の要素が実際に出力される直前」に出力される。
      ・delim 要素は「二回目以降の要素が出力される直前」に出力される。
      ・post 要素は「コピーを含め、全てのイテレータが破棄されたとき」に出力される。
        ただし、未だ pre が出力されていなければ、post も出力されない。
      ・もし全てのイテレータが破棄される前に新たなイテレータを製作した場合、
        その時点で post 要素が出力され、「何も出力されていない」状態になる。

*/

// operator() で使う
#include "list_format.hpp"
// タイプ数削減のための using 宣言
#include "call_traits.hpp"

#include <iterator>
#include <cassert>
#include <algorithm>  // for swap()

#include <boost/range.hpp>


namespace gintenlib
{
  template< typename Elem >
  class basic_list_formatter
  {
    typedef typename call_traits<Elem>::param_type param_type;

   public:
    typedef typename call_traits<Elem>::value_type element_type;

    basic_list_formatter()
      : pre("( "), delim(", "), post(" )"), iterator_count_(0) {}
    
    explicit basic_list_formatter
      ( param_type pre_, param_type delim_, param_type post_ )
       : pre(pre_), delim(delim_), post(post_), iterator_count_(0) {}

    // メンバ変数なので、自由にアクセスできる。
    element_type pre, delim, post;

    // 出力範囲をイテレータで指定して、ストリームに流せるように
    template< typename InIte >
      list_format_impl_< InIte, Elem, Elem, Elem >
    operator()( const InIte& begin, const InIte& end ) const
    {
      return list_format_impl_< InIte, Elem, Elem, Elem >( begin, end, pre, delim, post );
    }
    // 出力範囲を range で指定して、ストリームに流せるように
    template< typename Range >
      list_format_impl_< typename boost::range_iterator<const Range>::type, Elem, Elem, Elem >
    operator()( const Range& src ) const
    {
      return list_format_impl_< typename boost::range_iterator<const Range>::type, Elem, Elem, Elem >
          ( boost::begin(src), boost::end(src), pre, delim, post );
    }

    // イテレータ
    template< typename Stream >
    struct iterator
      : std::iterator< std::output_iterator_tag, void, void, void, void >
    {
      // イテレータを制作する
      iterator( const basic_list_formatter& formatter, Stream& os )
        : formatter_(&formatter), os_(&os)
      {
        // 出力が「閉じて」ないなら、閉じる
        close();

        // この時点でカウントは負
        --formatter_->iterator_count_;
      }
      
      // コピーコンストラクタ
      iterator( const iterator& src )
        : formatter_(src.formatter_), os_(src.os_)
      {
        // イテレータカウントを変更する
        if( formatter_->iterator_count_ > 0 )
        {
          ++formatter_->iterator_count_;
        }
        else if( formatter_->iterator_count_ < 0 )
        {
          --formatter_->iterator_count_;
        }
        else
        {
          assert( !"sould not get here." );
        }
      }

      ~iterator()
      {
        if( formatter_->iterator_count_ > 0 )
        {
          // count が正 <==> 既に何か出力されている
          if( --formatter_->iterator_count_ == 0 )
          {
            // count が 0 になった <==> もう他のイテレータは無い
            *os_ << formatter_->post;
          }
        }
        else if( formatter_->iterator_count_ < 0 )
        {
          // count が負 <==> 閉じる必要はない
          ++formatter_->iterator_count_;
        }
        else
        {
          assert( !"sould not get here." );
        }
      }

      // 代入演算
      iterator& operator=( const iterator& src )
      {
        iterator(src).swap(*this);
        return *this;
      }
      void swap( iterator& other )
      {
        using std::swap;
        swap( formatter_, other.formatter_ );
        swap( os_, other.os_ );
      }
      friend void swap( iterator& one, iterator& other )
      {
        one.swap(other);
      }

      // 実際の出力
      template< typename T >
      void output( const T& val ) const
      {
        if( formatter_->iterator_count_ > 0 )
        {
          // 既に何かが出力された
          *os_ << formatter_->delim << val;
        }
        else if( formatter_->iterator_count_ < 0 )
        {
          // まだなにも出力されてない
          *os_ << formatter_->pre << val;
          formatter_->iterator_count_ = -formatter_->iterator_count_;
        }
        else
        {
          assert( !"sould not get here." );
        }
      }

      // 「出力を閉じる」
      iterator& close()
      {
        // カウントが正 <==> 出力が「閉じて」ない
        if( formatter_->iterator_count_ > 0 )
        {
          // 閉じる
          *os_ << formatter_->post;
          // 「未出力」へと設定
          formatter_->iterator_count_ = -formatter_->iterator_count_;
        }
        return *this;
      }

      // 参照を外した場合、出力用プロキシを作って返す
      struct output_proxy
      {
        const iterator* p;
        explicit output_proxy( const iterator* p_ )
          : p( p_ ){}
        
        // 出力操作
        template<typename T>
        output_proxy& operator=( const T& val )
        {
          p->output( val );
          return *this;
        }
      };
      output_proxy operator*() const { return output_proxy(this); }
      
      // 前置／後置インクリメントは何もしない
      iterator& operator++(){ return *this; }
      iterator& operator++(int){ return *this; }

    private:
      const basic_list_formatter* formatter_;
      Stream* os_;

    };  // struct iterator

    template< typename Stream >
    friend struct iterator;

    // 出力イテレータを作る
    template< typename Stream >
    iterator<Stream> make_iterator( Stream& os ) const
    {
      return iterator<Stream>( *this, os );
    }
    template< typename Stream >
    friend iterator<Stream> make_iterator( const basic_list_formatter& fmt, Stream& os )
    {
      return iterator<Stream>( fmt, os );
    }

  private:
    mutable int iterator_count_;

  };  // struct basic_list_formatter

  typedef basic_list_formatter<const char*> list_formatter;

}   // namespace gintenlib


#endif  // #ifndef GINTENLIB_INCLUDED_LIST_FORMATTER_HPP_
