#ifndef GINTENLIB_INCLUDED_NEW_HPP_
#define GINTENLIB_INCLUDED_NEW_HPP_

/*

      <gintenlib/new_.hpp>

  new_ ： スマートポインタ版 new 演算子

  宣言：
    template<typename T>
    struct ptr
    {
      typedef shared_ptr<T> type;
    };
    
    template<typename T>
    inline typename ptr<T>::type new_();
    
    template< typename T, typename Arg1, ... , typename ArgN >
    inline typename ptr<T>::type new_( const Arg1& arg1, ... , const ArgN& argN );
    
    class new_core_access;
    
    template<typename Derived>
    struct enable_static_new_;
    
    // ほかに、new_ と同じ機能を持ったファンクタ、 new_ptr<T> が存在する。

  機能：
     T 型のオブジェクトを new 演算子で製作し、boost::shared_ptr<T> に格納して返す関数。
    基本的な動作は boost::make_shared と全く同様であるが、幾つかの違いがある。
    この関数からの new アクセスは、全て new_core_access クラスを通して行われるため、
      friend class gintenlib::new_core_access;
    と宣言した上でコンストラクタを private 部分におけば、new_ 以外からの構築を禁止出来る。
    もし private コンストラクタの全てを公開するのが嫌な場合は、
     enable_static_new_ テンプレートクラスを使うことで、呼び出しをカスタマイズできる。
     その場合、コンストラクタの代わりに new_ という名前の静的関数が呼ばれるようになる。
    
  使用例：
    boost::shared_ptr<hoge> p = gintenlib::new_<hoge>( 1, 2, 3 );
    // boost::shared_ptr<hoge> p( new hoge( 1, 2, 3 ) ); と同じ
    
  利点：
    基本的な利点は boost::make_shared と同じです。
    すなわち例外安全性の強化と実行速度の最適化を期待できます。
    この new_ ライブラリは、それに加え、アクセス制限を行うことが出来ます。
    これにより、 enable_shared_from_this を使ったクラスなど「確実に shared_ptr に入れたい」
    クラスを扱う場合の安全性が上昇します。
    とはいえ、一般にはスタックにも置けた方がいいので、使いどころは考えましょう。
    
  備考：
    引数は const& で渡されます。非const参照を渡したい場合には boost::ref を使ってください。

*/

#include "shared_ptr.hpp"
#include "enable_if.hpp"
#include "optional_storage.hpp"

#include <boost/preprocessor/arithmetic/inc.hpp>
#include <boost/preprocessor/repetition/enum_params.hpp>
#include <boost/preprocessor/repetition/enum_binary_params.hpp>
#include <boost/preprocessor/repetition/repeat_from_to.hpp>

#include <boost/type_traits/is_base_of.hpp>

// 引数の最大値
#ifndef GINTENLIB_ARG_COUNT
#define GINTENLIB_ARG_COUNT 10
#endif

namespace gintenlib
{
  // new_ の戻り値のポインタ型（現在は shared_ptr 一択）
  template<typename T>
  struct ptr
  {
    typedef shared_ptr<T> type;
  };
  
  // このクラスから継承することで、コンストラクタではなく static の new_ 関数が呼ばれるようになる
  template<typename Derived >
  struct enable_static_new_
  {
  };
  // 上のクラスを継承しているかどうかを判断するクラス
  template<typename T>
  struct enable_if_enabled_static_new_
    : enable_if< boost::is_base_of< enable_static_new_<T>, T > > {};
  
  
  // クラスの private メンバアクセス用のクラス。
  // コンストラクタないし new_ 静的関数を private においた上で、
  // friend class gintenlib::new_core_access; 
  // とすれば、 new_ を介してしか製作できないクラスになる。
  class new_core_access
  {
    // このクラスは new_ptr ファンクタのみが使用できる
    template<typename T, typename>
    friend class new_ptr;
    
    // 通常のコンストラクタを呼び出す new
    // placement new が出来たのでほぼ陳腐化状態
    template<typename T>
    static typename ptr<T>::type normal_new()
    {
      return typename ptr<T>::type( new T() );
    }
    // 多引数版
    #define GINTENLIB_GEN( z, n, d )                                              \
      template< typename T, BOOST_PP_ENUM_PARAMS( n, typename A ) >               \
      static typename ptr<T>::type normal_new                                     \
        ( BOOST_PP_ENUM_BINARY_PARAMS( n, const A, &arg ) )                       \
      {                                                                           \
        return typename ptr<T>::type( new T( BOOST_PP_ENUM_PARAMS( n, arg ) ) );  \
      }
    
    BOOST_PP_REPEAT_FROM_TO( 1, BOOST_PP_INC(GINTENLIB_ARG_COUNT), GINTENLIB_GEN, _ )
    
    #undef GINTENLIB_GEN
    
    
    // placement new を呼び出す
    template<typename T>
    static T* placement_new( void* p )
    {
      return ::new (p) T();
    }
    // 多引数版
    #define GINTENLIB_GEN( z, n, d )                                  \
      template< typename T, BOOST_PP_ENUM_PARAMS( n, typename A ) >   \
      static T* placement_new                                         \
        ( void* p, BOOST_PP_ENUM_BINARY_PARAMS( n, const A, &arg ) )  \
      {                                                               \
        return ::new (p) T( BOOST_PP_ENUM_PARAMS( n, arg ) );         \
      }
    
    BOOST_PP_REPEAT_FROM_TO( 1, BOOST_PP_INC(GINTENLIB_ARG_COUNT), GINTENLIB_GEN, _ )
    
    #undef GINTENLIB_GEN
    
    
    // new_ 静的関数を呼び出す new
    template<typename T>
    static typename ptr<T>::type static_new()
    {
      return typename ptr<T>::type( T::new_() );
    }
    // 多引数版
    #define GINTENLIB_GEN( z, n, d )                                \
      template< typename T, BOOST_PP_ENUM_PARAMS( n, typename A ) > \
      static typename ptr<T>::type static_new                       \
        ( BOOST_PP_ENUM_BINARY_PARAMS( n, const A, &arg ) )         \
      {                                                             \
        return typename ptr<T>::type                                \
          ( T::new_( BOOST_PP_ENUM_PARAMS( n, arg ) ) );            \
      }
    
    BOOST_PP_REPEAT_FROM_TO( 1, BOOST_PP_INC(GINTENLIB_ARG_COUNT), GINTENLIB_GEN, _ )
    
    #undef GINTENLIB_GEN
    
   private:
    // 元から private だが、強調する意味でつけておく。
    // このクラスは構築できない。
    new_core_access();
    
  };
  
  // 関数オブジェクト版 new_（コンストラクタ版）
  template<typename T, typename = void>
  struct new_ptr
  {
    typedef typename ptr<T>::type result_type;
    
    // <boost/make_shared.hpp> に習い、効率的な実装を行う
    result_type operator()() const
    {
      // 仮の shared_ptr を作って参照カウントを作成し
      shared_ptr<T> pt( static_cast<T*>(0), deleter() );
      // 参照カウント内のデリータのアドレスを取得する
      optional_storage<T> &storage = boost::get_deleter<deleter>(pt)->storage;
      void* pv = storage.address();
      
      // デリータ内に用意された領域にオブジェクトを構築し、
      T* pt2 = new_core_access::template placement_new<T>( pv );
      storage.set_initialized();
      
      // 構築されたオブジェクトのアドレスと最初に作った参照カウントから、
      // 改めて shared_ptr を制作する
      return make_result( pt, pt2 );
    }
    // 多引数版
    #define GINTENLIB_GEN( z, n, d )                              \
      template< BOOST_PP_ENUM_PARAMS( n, typename A ) >           \
      result_type operator()                                      \
        ( BOOST_PP_ENUM_BINARY_PARAMS( n, const A, &arg ) ) const \
      {                                                           \
        shared_ptr<T> pt( static_cast<T*>(0), deleter() );        \
        optional_storage<T> &storage =                            \
          boost::get_deleter<deleter>(pt)->storage;               \
        void* pv = storage.address();                             \
                                                                  \
        T* pt2 = new_core_access::template placement_new<T>       \
          ( pv, BOOST_PP_ENUM_PARAMS( n, arg ) );                 \
        storage.set_initialized();                                \
                                                                  \
        return make_result( pt, pt2 );                            \
      }
    
    BOOST_PP_REPEAT_FROM_TO( 1, BOOST_PP_INC(GINTENLIB_ARG_COUNT), GINTENLIB_GEN, _ )
    
    #undef GINTENLIB_GEN
    
   private:
    // カスタムデリータ
    struct deleter
    {
      deleter(){}
      // コピーでは何もしないよう最適化
      deleter( const deleter& ) {}
      
      // オブジェクトが格納される領域
      optional_storage<T> storage;
      
      typedef void result_type;
      void operator()( T* )
      {
        storage.destory();
      }
      
    };  // struct deleter
    
    // 終了処理
    result_type make_result( const result_type& pt, T* pt2 ) const
    {
      // enable_shared_from_this を使っている場合、正しくアドレスを設定する必要がある
      boost::detail::sp_enable_shared_from_this( &pt, pt2, pt2 );
      // このコンストラクタは pt の参照カウントと pt2 のアドレスを使って shared_ptr を作る
      return result_type( pt, pt2 );
    }
    
  };  // struct new_ptr<T>
  
  // 関数オブジェクト版 new_（静的関数版）
  template<typename T>
  struct new_ptr< T, typename enable_if_enabled_static_new_<T>::type >
  {
    typedef typename ptr<T>::type result_type;
    
    result_type operator()() const
    {
      return new_core_access::template static_new<T>();
    }
    // 多引数版
    #define GINTENLIB_GEN( z, n, d )                              \
      template< BOOST_PP_ENUM_PARAMS( n, typename A ) >           \
      result_type operator()                                      \
        ( BOOST_PP_ENUM_BINARY_PARAMS( n, const A, &arg ) ) const \
      {                                                           \
        return new_core_access::template static_new<T>            \
          ( BOOST_PP_ENUM_PARAMS( n, arg ) );                     \
      }
    
    BOOST_PP_REPEAT_FROM_TO( 1, BOOST_PP_INC(GINTENLIB_ARG_COUNT), GINTENLIB_GEN, _ )
    
    #undef GINTENLIB_GEN
  };
  
  
  // 本体
  template<typename T>
  inline typename ptr<T>::type new_()
  {
    return new_ptr<T>()();
  }
  // 多引数版
  #define GINTENLIB_GEN( z, n, d )                                \
    template< typename T, BOOST_PP_ENUM_PARAMS( n, typename A ) > \
    inline typename ptr<T>::type new_                             \
      ( BOOST_PP_ENUM_BINARY_PARAMS( n, const A, &arg ) )         \
    {                                                             \
      return new_ptr<T>()( BOOST_PP_ENUM_PARAMS( n, arg ) );      \
    }
  
  BOOST_PP_REPEAT_FROM_TO( 1, BOOST_PP_INC(GINTENLIB_ARG_COUNT), GINTENLIB_GEN, _ )
  
  #undef GINTENLIB_GEN

}   // namespace gintenlib


#endif  // #ifndef GINTENLIB_INCLUDED_NEW_HPP_
