#ifndef GINTENLIB_INCLUDED_OPTIONAL_STORAGE_HPP_
#define GINTENLIB_INCLUDED_OPTIONAL_STORAGE_HPP_

/*

      <gintenlib/optional_storage.hpp>

  optional_storage ： 後から初期化できるメモリ領域

  宣言：
    template<typename T>
    struct optional_storage
    {
      typedef T element_type;
      typedef T value_type;
      
      // 構築／破棄
      optional_storage();
      
      template<typename Constructor>
      explicit optional_storage( Constructor ctor );
      
      ~optional_storage() throw ();
      
      // 中のデータを破棄する
      void destory();
      // カスタムデストラクタを使ってデータを破棄する
      template<typename Destructor>
      void destory( Destructor d );
      
      // storage の生アドレスを得る
      void* address();
      const void* address() const;
      
      // 初期化ずみか否か
      bool initialized() const;
      // 初期化ずみマークをつける
      void set_initialized();
      
      // 構築済みなら、構築されたオブジェクトのアドレスを取得する
      T* get();
      const T* get() const;
      // アロー演算
      T* operator->();
      const T* operator->() const;
      // 参照外し
      T& operator*();
      const T& operator*() const;
      
     private:
      optional_storage( optional_storage const& ) = delete;
      void operator=( optional_storage const& ) = delete;
      
    };  // optional_storage<T>

  機能：
    予めメモリ領域を確保して云々、という、少し汚い処理を行いたい場合ってのが時々あります。
    そういうときに、ある程度安心して処理を行えるようにする為のクラステンプレートです。
    予めメモリ領域を確保して何かしらの処理をしてから、 address() で未初期化メモリのアドレスを取得。
    そのアドレスに placement new でオブジェクトを作ったら set_initialized() を呼ぶ。
    この一連の動作さえ行えば、後の破棄や何やらは、全て optional_storage<T> がやってくれます。
    boost::optional<T> でも似たようなことは出来ますが、未初期化アドレスを直接得られません。
    
    また、optional_storage<T> を使って構築済みオブジェクトを破棄する場合には、
    普通のデストラクタだけでなく、カスタムデストラクタを指定することもできます。
    そういう場合は明示的に destory を呼び出し、その引数にカスタムデストラクタを渡しましょう。
    
    （追記）
    「指定したアドレスに対して関数を呼び出し、 set_initialized() する」
    という construct 関数を新しく作りました。
    こちらの方がより安全なので、今後はこちらを使いましょう。
    set_initialized は近々削除予定です。

  使用例：
    // とりあえず領域だけ確保して
    gintenlib::optional_storage<hoge> storage;
    
    // storage が必要な何かしらの処理
    
    // placement new で構築
    ::new ( storage.address() ) hoge( args );
    // 構築したよマークをつける
    storage.set_initialized();
    
    // めんどいので参照変数で以降のアクセスを簡略化
    hoge& h = *storage;
    
    // 以降、 h は普通の hoge オブジェクトとして扱える
    // storage の有効期限が切れると、自動でデストラクトされるので安心

  補足：
    ・set_initialized ではなく construct を使いましょう。

*/

#include <cassert>

#include "destructor.hpp"
#include "storage.hpp"

namespace gintenlib
{
  // ちょっと安全性の低い boost::optional
  template<typename T>
  struct optional_storage
  {
    typedef       T     element_type;
    typedef       T       value_type;
    typedef       T*         pointer;
    typedef const T*   const_pointer;
    typedef       T&       reference;
    typedef const T& const_reference;
    
    // 構築／破棄
    optional_storage()
      : initialized_( false ) {}
    
    template<typename Constructor>
    explicit optional_storage( Constructor ctor )
      : initialized_( false )
    {
      construct_( ctor );
    }
    
    ~optional_storage() throw ()
    {
      destory();
    }
    
    // 中のデータを破棄する
    void destory()
    {
      destory( destructor() );
    }
    // カスタムデストラクタを使ってデータを破棄する
    template<typename Destructor>
    void destory( Destructor d )
    {
      if( initialized_ )
      {
        d( get() );
        initialized_ = false;
      }
    }
    
    // storage の生アドレスを得る
    void* address(){ return storage_.address(); }
    const void* address() const { return storage_.address(); }
    
    // 初期化ずみか否か
    bool initialized() const { return initialized_; }
    
    // 初期化ずみマークをつける（非推奨）
    void set_initialized() throw () { initialized_ = true; }
    
    // 関数を使って初期化する
    // 初期化済みの場合は破棄してから初期化する
    template<typename Constructor>
    T* construct( Constructor ctor )
    {
      if( initialized() ){ destory(); }
      
      construct_();
      
      return static_cast<T*>( address() );
    }
    
    
    // 構築済みなら、構築されたオブジェクトのアドレスを取得する
    T* get(){ return initialized() ? static_cast<T*>( address() ) : 0; }
    const T* get() const { return initialized() ? static_cast<const T*>( address() ) : 0; }
    
    // アロー演算
    T* operator->()
    {
      using namespace std;
      assert( initialized() );
      return get();
    }
    const T* operator->() const
    {
      using namespace std;
      assert( initialized() );
      return get();
    }
    // 参照外し
    T& operator*()
    {
      using namespace std;
      assert( initialized() );
      return *get();
    }
    const T& operator*() const
    {
      using namespace std;
      assert( initialized() );
      return *get();
    }
    
   private:
    bool initialized_;
    typedef storage<T> storage_type;
    storage_type storage_;
    
    // 実装用関数
    
    // initialized_ によらずオブジェクトを構築する（ unsafe ） 
    template<typename Ctor>
    void construct_( Ctor ctor )
    {
      ctor( address() );
      initialized_ = true;
    }
    
  };  // optional_storage<T>

}   // namespace gintenlib


#endif  // #ifndef GINTENLIB_INCLUDED_OPTIONAL_STORAGE_HPP_
