#ifndef GINTENLIB_PLANE_INCLUDED_ANGLE_HPP_
#define GINTENLIB_PLANE_INCLUDED_ANGLE_HPP_

/*

      <gintenlib/plane/angle.hpp>

  angle ： 平面角

  機能：
    平面角を扱うクラス。
    Ｃ＋＋において角度は radian で扱いますが、たまに degree も使いたくなるでしょう。
    そういうときに、二つの表し方を混同しないように扱えるクラスです。
    あ、それから角度の正規化（-πからπの間になるよう調節する）もできます。ついでに。
    
    主に弾幕を作るときに使う程度のクラスです。

  使用例：
    angle x, y = degree(60);  // x は 0, y は 60°
    // 三角関数
    cout << sin(x) << ", " << sin(y) << endl;
    // 逆三角関数
    angle z = angle::asin(0.5);
    // 数値に変換
    cout << "z は " << to_rad(z) << " ラジアン、 " << to_deg(z) << " 度です。";

*/

#include "fwd.hpp"
#include "vector.hpp"

#include "../math.hpp"   // for M_PI
#include <cassert>
#include <boost/operators.hpp>

namespace gintenlib
{
 namespace plane
 {
  // angle : 平面角
  template<typename Real>
  struct basic_angle
    : private boost::totally_ordered< basic_angle<Real> >,
     private boost::additive< basic_angle<Real> >,
     private boost::multiplicative< basic_angle<Real>, Real >,
     private boost::modable< basic_angle<Real> >
  {
    typedef Real real_type;
    real_type theta;
    
    basic_angle()
      : theta() {}
    
    template<typename T>
    basic_angle( const basic_angle<T>& src )
      : theta( src.theta ) {}
    
    // 実数値から生成、単位は rad 
    explicit basic_angle( real_type theta_ )
      : theta(theta_) {}
    
    // theta の値を得る
    real_type to_deg() const
    {
      return theta * 180 / M_PI;
    }
    friend real_type to_deg( const basic_angle& target )
    {
      return target.to_deg();
    }
    friend real_type deg( const basic_angle& target )
    {
      return target.to_deg();
    }
    real_type to_rad() const
    {
      return theta;
    }
    friend real_type to_rad( const basic_angle& target )
    {
      return target.to_rad();
    }
    friend real_type rad( const basic_angle& target )
    {
      return target.to_rad();
    }
    
    // operator overloads
    // 比較
    friend bool operator==( const basic_angle& lhs, const basic_angle& rhs )
    {
      return lhs.theta == rhs.theta;
    }
    friend bool operator<( const basic_angle& lhs, const basic_angle& rhs )
    {
      return lhs.theta < rhs.theta;
    }
    // 加減算
    basic_angle& operator+=( const basic_angle& rhs )
    {
      theta += rhs.theta;
      return *this;
    }
    basic_angle& operator-=( const basic_angle& rhs )
    {
      theta -= rhs.theta;
      return *this;
    }
    // 符号反転
    basic_angle operator-() const
    {
      return basic_angle( -theta );
    }
    // 掛け算、割り算
    basic_angle& operator*=( const real_type& rhs )
    {
      theta *= rhs;
      return *this;
    }
    basic_angle& operator/=( const real_type& rhs )
    {
      theta /= rhs;
      return *this;
    }
    // 比の算出
    real_type operator/( const basic_angle& rhs ) const
    {
      return theta / rhs.theta;
    }
    // 剰余演算
    // std::fmod を呼び出すだけ
    basic_angle& operator%=( const basic_angle& rhs )
    {
      using std::fmod;
      
      theta = fmod( theta, rhs.theta );
      
      return *this;
    }
    // ベクトルに対して掛け算すると回転操作
    // ただし、左から掛けた時のみ
    basic_vector<Real> operator*( const basic_vector<Real>& target ) const
    {
      return rotate( target, *this );
    }
    
    // [ -π, π ] の範囲に補正
    basic_angle& normalize()
    {
      using std::fmod;
      
      if( theta > M_PI )
      {
        // πだけずらして元に戻す
        theta = fmod( theta + M_PI, M_PI * 2 ) - M_PI;
      }
      else if( theta < -M_PI )
      {
        // ちょっとややこしいけど分かるはず（たぶん）
        theta = -fmod( -theta + M_PI, M_PI * 2 ) + M_PI;
      }
      
      // 一応確かめておくよ？
      assert( -M_PI <= theta && theta <= M_PI );
      
      return *this;
    }
    friend basic_angle normalized( const basic_angle& target )
    {
      basic_angle temp = target;
      return temp.normalize();
    }
    // [ 0, 2π ) の範囲に補正
    basic_angle& unique()
    {
      using std::fmod;
      
      if( theta >= M_PI * 2 )
      {
        theta = fmod( theta, M_PI * 2 );
      }
      else if( theta < 0 )
      {
        theta = M_PI * 2 - fmod( -theta, M_PI * 2 );
      }
      
      // 一応確かめておくよ？
      assert( 0 <= theta && theta < M_PI * 2 );
      
      return *this;
    }
    friend basic_angle uniqued( const basic_angle& target )
    {
      basic_angle temp = target;
      return temp.unique();
    }
    
    // 三角関数
    #define GINTENLIB_PLANE_ANGLE_GEN_FUNCTION( func )  \
      real_type func() const                            \
      {                                                 \
        using std::func;                                \
        return func( theta );                           \
      }                                                 \
      friend real_type func( const basic_angle& x )     \
      {                                                 \
        return x.func();                                \
      }
    
      // 実際に生成
      GINTENLIB_PLANE_ANGLE_GEN_FUNCTION( sin )
      GINTENLIB_PLANE_ANGLE_GEN_FUNCTION( cos )
      GINTENLIB_PLANE_ANGLE_GEN_FUNCTION( tan )
    
    #undef GINTENLIB_PLANE_ANGLE_GEN_FUNCTION
    
    // 逆三角関数
    static basic_angle asin( real_type x )
    {
      using std::asin;
      return basic_angle( asin(x) );
    }
    static basic_angle acos( real_type x )
    {
      using std::acos;
      return basic_angle( acos(x) );
    }
    static basic_angle atan( real_type x )
    {
      using std::atan;
      return basic_angle( atan(x) );
    }
    static basic_angle atan2( real_type y, real_type x )
    {
      using std::atan2;
      return basic_angle( atan2( y, x ) );
    }
    
  };
  typedef basic_angle<double> angle;
  
  // ラジアン <--> 度 換算
  template<typename Real>
  Real degree( const basic_angle<Real>& x )
  {
    return x.to_deg();
  }
  template<typename Real>
  basic_angle<Real> degree( Real deg )
  {
    return basic_angle<Real>( deg * M_PI / 180 );
  }
  template<typename Real>
  Real radian( const basic_angle<Real>& x )
  {
    return x.to_rad();
  }
  template<typename Real>
  basic_angle<Real> radian( Real rad )
  {
    return basic_angle<Real>( rad );
  }
  // 短縮名
  template<typename Real>
  basic_angle<Real> deg( Real deg_ )
  {
    return degree( deg_ );
  }
  template<typename Real>
  basic_angle<Real> rad( Real rad_ )
  {
    return radian( rad_ );
  }
  
 }  // namespace plane
}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_PLANE_INCLUDED_ANGLE_HPP_
