#ifndef GINTENLIB_PLANE_INCLUDED_VECTOR_HPP_
#define GINTENLIB_PLANE_INCLUDED_VECTOR_HPP_

/*

      <gintenlib/plane/vector.hpp>

  vector ： 二次元ベクトル

  宣言：
  template< typename Real >
  struct basic_vector
    : private boost::equality_comparable< basic_vector<Real> >,
     private boost::additive< basic_vector<Real> >,
     private boost::multiplicative< basic_vector<Real>, Real >
  {
    typedef Real real_type;
    real_type x, y;
    
    basic_vector();
    basic_vector( real_type x_, real_type y_ );
      
    template<typename T>
    basic_vector( const basic_vector<T>& src );
    
    template<typename T>
    basic_vector( const std::pair<T, T>& src );
    
    // 演算子多重定義
    // それぞれ見たまんまです。operator+ とかは boost:: operators によって各種自動定義されます。
    basic_vector& operator+=( const basic_vector& rhs );
    basic_vector& operator-=( const basic_vector& rhs );
    basic_vector operator-() const;
    basic_vector& operator*=( real_type rhs );
    basic_vector& operator/=( real_type rhs );
    bool operator==( const basic_vector& rhs ) const;
    
    // 2-Norm
    real_type norm() const;
    friend real_type norm( const basic_vector& target );
    
    // 絶対値
    real_type absolute() const;
    friend real_type absolute( const basic_vector& target );
    
    // 正規化
    // 正規化した結果を返すのではなく、このベクトル自体を正規化します。
    basic_vector& normalize();
    // friend 版はコピーしてから正規化します。
    friend basic_vector normalized( const basic_vector& x );
    
    // 偏角
    real_type argument() const;
    friend real_type argument( const basic_vector& x );
    
    // 内積
    real_type dot( const basic_vector& rhs ) const;
    friend real_type dot( const basic_vector& lhs, const basic_vector& rhs );
    friend real_type operator*( const basic_vector& lhs, const basic_vector& rhs );
    
    // ローテート。回転させた結果を返すのではなく、このベクトル自体を回します。
    // 角度指定バージョン。テンプレートなのは、gintenlib::plane::angle も渡せるように、です。
    template< typename Angle >
    basic_vector& rotate( const Angle& angle );
    // サインとコサインを指定したバージョン。高速です。
    basic_vector& rotate( real_type s, real_type c );
    
    // 上二つの friend 版。コピーしてから回します。
    template< typename Angle >
    friend basic_vector rotate( const basic_vector& target, const Angle& angle );
    friend basic_vector rotate( const basic_vector& target, real_type s, real_type c );
  };
  typedef basic_vector<double> vector;

  機能：
    二次元の相対ベクトルを表すクラス。
    絶対値計算、偏角計算、正規化、回転など、とりあえず一通り必要そうなものは揃えています。

*/

#include "fwd.hpp"

#include <cmath>
#include <utility>
#include <boost/operators.hpp>

namespace gintenlib
{
 namespace plane
 {
  template< typename Real >
  struct basic_vector
    : private boost::equality_comparable< basic_vector<Real> >,
     private boost::additive< basic_vector<Real> >,
     private boost::multiplicative< basic_vector<Real>, Real >
  {
    typedef Real real_type;
    real_type x, y;
    
    basic_vector()
      : x(), y() {}
    
    basic_vector( real_type x_, real_type y_ )
      : x(x_), y(y_) {}
      
    template<typename T>
    basic_vector( const basic_vector<T>& src )
      : x(src.x), y(src.y) {}
    
    template<typename T>
    basic_vector( const std::pair<T, T>& src )
      : x(src.first), y(src.second) {}
    
    // 演算子多重定義
    basic_vector& operator+=( const basic_vector& rhs )
    {
      x += rhs.x;
      y += rhs.y;
      
      return *this;
    }
    basic_vector& operator-=( const basic_vector& rhs )
    {
      x -= rhs.x;
      y -= rhs.y;
      
      return *this;
    }
    basic_vector operator-() const
    {
      return basic_vector( -x, -y );
    }
    basic_vector& operator*=( real_type rhs )
    {
      x *= rhs;
      y *= rhs;
      
      return *this;
    }
    basic_vector& operator/=( real_type rhs )
    {
      x /= rhs;
      y /= rhs;
      
      return *this;
    }
    bool operator==( const basic_vector& rhs ) const
    {
      return ( x == rhs.x ) && ( y == rhs.y );
    }
    
    // 2-Norm
    real_type norm() const
    {
      return x * x + y * y;
    }
    friend real_type norm( const basic_vector& target )
    {
      return target.norm();
    }
    
    // 絶対値
    real_type absolute() const
    {
      return std::sqrt( x * x + y * y );
    }
    friend real_type absolute( const basic_vector& target )
    {
      return target.absolute();
    }
    
    // 正規化
    basic_vector& normalize()
    {
      real_type r = absolute();
      
      if( r == 0 )
      {
        x = 1;
        y = 0;
      }
      else
      {
        x /= r;
        y /= r;
      }
      
      return *this;
    }
    friend basic_vector normalized( const basic_vector& x )
    {
      basic_vector temp = x;
      temp.normalize();
      return temp;
    }
    
    // 偏角
    real_type argument() const
    {
      if( x == 0 && y == 0 )
      {
        return 0;
      }
      return std::atan2( y, x );
    }
    friend real_type argument( const basic_vector& x )
    {
      return x.argument();
    }
    
    // 内積
    real_type dot( const basic_vector& rhs ) const
    {
      return x * rhs.x + y * rhs.y;
    }
    friend real_type dot( const basic_vector& lhs, const basic_vector& rhs )
    {
      return lhs.dot(rhs);
    }
    friend real_type operator*( const basic_vector& lhs, const basic_vector& rhs )
    {
      return lhs.dot(rhs);
    }
    
    // ローテート
    template< typename Angle >
    basic_vector& rotate( const Angle& angle )
    {
      using std::sin; using std::cos;
      return rotate( sin(angle), cos(angle) );
    }
    basic_vector& rotate( real_type s, real_type c )
    {
      real_type sx = s*x, sy = s*y, cx = c*x, cy = c*y;
      
      x = cx - sy;
      y = sx + cy;
      
      return *this;
    }
    template< typename Angle >
    friend basic_vector rotate( const basic_vector& target, const Angle& angle )
    {
      basic_vector temp = target;
      temp.rotate( angle );
      return temp;
    }
    friend basic_vector rotate( const basic_vector& target, real_type s, real_type c )
    {
      basic_vector temp = target;
      temp.rotate( s, c );
      return temp;
    }
  };
  typedef basic_vector<double> vector;
  
 }  // namespace plane
}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_PLANE_INCLUDED_VECTOR_HPP_
