#ifndef GINTENLIB_PREPROCESSOR_INCLUDED_ENABLE_IF_HPP
#define GINTENLIB_PREPROCESSOR_INCLUDED_ENABLE_IF_HPP

/*

      <gintenlib/preprocessor/enable_if.hpp>

  preprocessor.enable_if ： らくちん enable_if

  宣言：
    #define GINTENLIB_ENABLE_IF (( condition ))   〜
    #define GINTENLIB_DISABLE_IF(( condition ))   〜
    
  使用例：
    // d_enable_if.hpp の例をやっつける
    // using namespace gintenlib; // こんなの必要ない
    
    // 引数を Target 型へ暗黙変換する
    // 通常版、Target が参照型でなければこっちが呼ばれる
    // 第二引数に注目。ちなみにこの引数はダミーなので無視してＯＫ
    template< typename Target, typename Source >
    inline Target cast( const Source& src,
      GINTENLIB_DISABLE_IF(( boost::is_reference<Target> )) )
    {
      return src;
    }
    
    // 参照版
    template< typename Target, typename Source >
    inline Target cast( Source& src,
      GINTENLIB_ENABLE_IF((  boost::is_reference<Target> )) )
    {
      return src;
    }
    
    // c.f.) d_enable_if.hpp の例
    //
    //   // 参照版
    //   template< typename Target, typename Source >
    //   inline Target cast( Source& src,
    //     typename d_enable_if<  boost::is_reference<Target> >::type = dummy_arg_t() )
    //   {
    //     return src;
    //   }
    // 
    // 比較してみると、どれだけ楽か分かるはず

  解説：
    関数オーバーロード用の enable_if をらくちんにするためのマクロです。
    実際にどのように展開されているかは下の例をご覧ください。
    まー真面目に書いても良いのですが、
      typename gintenlib::d_enable_if< なんたら, gintenlib::dummy_arg_t >::type 
        = gintenlib::dummy_arg_t()
    正直こんな長いソースコードなんて書いてられませんからね。
    d_enable_if と dummy_arg_t の存在意義は <gintenlib/d_enable_if.hpp> を参照してください。
    
    使い方は、上の例の通り、enable_if の <> の中の記述を「二重括弧でくくって」マクロに渡して、
    関数の引数リストの最後に置いておくだけ。簡単です。
    普通、こういう「メタ関数を引数に取る」マクロってのは、テンプレート引数リストのカンマが邪魔をして
    正常にマクロ展開されなかったりするのですが、これに関しては気にする必要はありません。
    ユーザ側で注意すべきは一点のみ、マクロの引数は必ず二重括弧でくるむこと。
    そこにさえ気をつければ簡単に関数呼び出しの切り替えが出来ます。便利です。

*/

#include "../d_enable_if.hpp"
#include "dequote.hpp"

namespace gintenlib
{
  // GINTENLIB_DEQUOTE(( TYPE )) は、 なんとか<〜>::type という形で展開される
  // この type が TYPE 型に定義されるような仕掛けとなっている
  // よって GINTENLIB_DEQUOTE 前の typename は必須
  #define GINTENLIB_ENABLE_IF( quoted_type )    \
    typename ::gintenlib::d_enable_if<          \
      typename GINTENLIB_DEQUOTE( quoted_type ) \
    >::type = ::gintenlib::dummy_arg_t()
  
  #define GINTENLIB_DISABLE_IF( quoted_type )   \
    typename ::gintenlib::d_disable_if<         \
      typename GINTENLIB_DEQUOTE( quoted_type ) \
    >::type = ::gintenlib::dummy_arg_t()

}   // namespace gintenlib


#endif  // #ifndef GINTENLIB_PREPROCESSOR_INCLUDED_ENABLE_IF_HPP
