#include "../gintenlib/clonable_ptr.hpp"

// boost の単体テストフレームワーク
#include <boost/test/minimal.hpp>

// ポインタ自身のテスト
template<typename T>
void test_pointer( const gintenlib::clonable_ptr<T>& p )
{
  // まず get() と get_pointer() のチェック
  BOOST_CHECK( p.get() == get_pointer(p) );
  
  // NULL でない場合
  if( p )
  {
    // operator! のチェック
    BOOST_CHECK( !p == false );
    
    // operator*, operator-> のチェック
    BOOST_CHECK( p.get() == &*p );
    BOOST_CHECK( p.get() == p.operator->() );
    
    // use_count と unique のチェック
    BOOST_CHECK( ( p.use_count() == 1 ) == p.unique() );
  }
  else
  {
    // get() は NULLか
    BOOST_CHECK( p.get() == 0 );
    // operator! のチェック
    BOOST_CHECK( !p );
    
    // use_count は 0 か？
    BOOST_CHECK( p.use_count() == 0 );
    // !unique() か？
    BOOST_CHECK( !p.unique() );
    
  }
}

// 基本的なチェック
template<typename T>
void test1( const gintenlib::clonable_ptr<T>& p0 )
{
  using gintenlib::clonable_ptr;
  
  // まず p0 をテスト
  test_pointer( p0 );
  
  // デフォルト構築
  {
    clonable_ptr<T> p1;
    // NULL にセットされているか？
    BOOST_CHECK( !p1 );
    
    // 他、もろもろはおｋ？
    test_pointer( p1 );
  }
  
  // コピー
  {
    // コピーに先立って参照カウントを調べる
    int count = p0.use_count();
    
    // コピーする
    clonable_ptr<T> p1 = p0;
    // とりあえずテスト
    test_pointer( p1 );
    // アドレスと参照カウントの両面で等しいことを確認
    BOOST_CHECK( p0 == p1 );
    BOOST_CHECK( p0.use_count() == p1.use_count() );
    
    // また、コピーが出来た分、参照カウントは増えているはず
    if( p0 )
    {
      BOOST_CHECK( p0.use_count() == count+1 );
    }
    else
    {
      // NULL の場合は参照カウントはゼロになる
      BOOST_CHECK( count == 0 );
      BOOST_CHECK( p0.use_count() == 0 );
    }
    
    // リセットのテスト
    p1.reset();
    BOOST_CHECK( p1 == 0 );
    // ふたたびテスト
    test_pointer( p1 );
    
    // この時点で参照カウントは元に戻るはず
    BOOST_CHECK( p0.use_count() == count );
  }
  
  // 本題。clone のテスト
  {
    // まず参照カウントを調べる
    int count = p0.use_count();
    
    // clone を作る
    clonable_ptr<T> p1;
    p1 = p0.clone();
    // とりあえずテスト
    test_pointer( p1 );
    
    // clone() しても参照カウントは変わらない
    BOOST_CHECK( p0.use_count() == count );
    
    if( p0 )
    {
      // clone で作られたオブジェクトは unique
      BOOST_CHECK( p1.unique() );
      // clone したため、別のオブジェクトが格納されてるはず
      BOOST_CHECK( p0 != p1 );
      // 不等号比較関連をついでに
      BOOST_CHECK( p0 < p1 || p1 < p0 );
      BOOST_CHECK( !( p0 < p1 && p1 < p0 ) );
    }
    else
    {
      // NULL の場合は p1 もゼロのはず
      BOOST_CHECK( !p1 );
    }
    
    // 生ポインタで to_unique の確認
    T* p_ = p1.get();
    // これまでの過程で p1 は null か unique のはず
    BOOST_CHECK( !p1 || p1.unique() );
    // unique のときは to_unique() を呼んでも何もしないはず
    p1.to_unique();
    BOOST_CHECK( p_ == p1.get() );
    // 念のためテスト
    test_pointer( p1 );
    
    // 次に p1 にふたたび p0 の値を入れる
    p1 = p0;
    // 念のためテスト
    test_pointer( p1 );
    // 生ポインタの値を更新
    p_ = p1.get();
    // これにより、NULL であるかユニークじゃないかの状態になった
    BOOST_CHECK( !p1 || !p1.unique() );
    // unique を呼ぶと p0 は変化せず p1 が変化するはず
    p1.to_unique();
    BOOST_CHECK( p_ == p0.get() );
    BOOST_CHECK( !p1 || p_ != p1.get() );
    // またまたテスト
    test_pointer( p1 );
  }
  
  // おまけで to_shared
  {
    // 先に参照カウントを調べる
    int count = p0.use_count();
    
    boost::shared_ptr<T> p1 = to_shared( p0 );
    // アドレスは一緒
    BOOST_CHECK( p0.get() == p1.get() );
    // カウントは一つ増える
    BOOST_CHECK( !p0 || p0.use_count() == count+1 );
    
    // shared_ptr がリセットされた場合、ちゃんと参照カウントは減る？
    p1.reset();
    BOOST_CHECK( p0.use_count() == count );
  }
  
  // const 関連
  {
    // T から const T への変換
    clonable_ptr<const T> p1 = p0;
    // 念のためテスト
    test_pointer( p1 );
    // アドレスとカウントが等しいことを確認
    BOOST_CHECK( p0 == p1 );
    BOOST_CHECK( p0.use_count() == p1.use_count() );
    
    // const T 型のポインタで reset してみる
    p1.reset( static_cast<const T*>( new T() ) );
    // reset 後は当然 unique()
    BOOST_CHECK( p1 && p1.unique() );
    // 念のため更にテスト
    test_pointer( p1 );
    
    // const T => T の変換（ clone すれば可能 ）
    clonable_ptr<T> p2 = p1.clone();
    // clone しないと無理
    // clonable_ptr<T> p2 = p1;
    
    // 念には念を入れてテスト
    test_pointer( p2 );
    // clone() 後なので当然 unique
    BOOST_CHECK( p2 && p2.unique() );
    BOOST_CHECK( p1 != p2 );
    BOOST_CHECK( p1 < p2 || p2 < p1 );
    BOOST_CHECK( !( p1 < p2 && p2 < p1 ) );
  }
  
  // おわり
}

int test_main( int, char** )
{
  test1( gintenlib::clonable_ptr<int>( new int() ) );
  test1( gintenlib::clonable_ptr<int>( static_cast<int*>(0) ) );
    
  return 0;
}
