#include "../gintenlib/new_.hpp"

// 仮テストコードなので
// boost.test はまだ使わない。
// というより、簡単な使い道を説明する程度のコード。

#include <iostream>
using namespace std;

// shared_ptr を受け取る関数
template<typename T>
void hoge( const boost::shared_ptr<T>& ptr )
{
  // 内容は別にどうでもよろしい
  cout << *ptr << endl;
}

void test1()
{
  // 基本的な使いかた
  // shared_ptr を受け取る関数があるとする。例えば上の hoge だ。
  // この手の関数に、新しく作ったオブジェクトを入れたい。
  
  // hoge( new int(23) );
  // こう書ければいいが、生ポインタから shared_ptr への暗黙変換は不可能である。
  
  // そこでこう書き、明示的に変換して渡すこととなる。
  hoge( boost::shared_ptr<int>( new int(23) ) );
  
  // これは型名 int が二回使われていて冗長である。ここで new_ を使えば 
  hoge( gintenlib::new_<int>( 42 ) );
  // と書ける。多くのプログラマは、これを自然と感じるのではないだろうか。
  
  // また、これは例外安全性の観点からも重要である。
  // 詳しくは http://boost.cppll.jp/HEAD/libs/smart_ptr/shared_ptr.htm#BestPractices を参照。
  // gintenlib::new_ は、この問題を非常にうまく扱うことが出来る。
}

// また gintenlib::new_ は、それ以外からの構築を禁止するようなクラスを簡単に作成できる。
// その一番の使い道は、boost::enable_shared_from_this を使う場合だろう。
// gintenlib::new_ 以外で構築できないクラスは、確実に shared_ptr に格納されるからである。
// これらには二つの実装法がある。順に見てみよう。

#include <boost/enable_shared_from_this.hpp>

// 実装１
struct force_shared1
  : boost::enable_shared_from_this<force_shared1>
{
  ~force_shared1() throw () {}
  
  void foo()
  {
    cout << "force_shared1::foo();\n";
  }
  
 private:
  // private に全てのコンストラクタを置き、一般からの構築を禁止
  force_shared1() {}
  // その上で gintenlib::new_core_access 構造体を friend に指定する
  friend class gintenlib::new_core_access;
  
};

// 実装２
struct force_shared2
  : boost::enable_shared_from_this<force_shared2>,
    gintenlib::enable_static_new_<force_shared2>  // これを指定する
{
  ~force_shared2() throw () {}
  
  // 適当にメンバを置く
  void foo();
  void bar();
  
  // enable_static_new_<クラス名> から継承させることで、
  // new_ による構築は、そのクラスの静的関数 new_ を通して行われるようになる
  static boost::shared_ptr<force_shared2> new_()
  {
    cout << "force_shared2::new_();\n";
    return boost::shared_ptr<force_shared2>( new force_shared2() );
  }
  // 当然複数の new_ を置ける
  static boost::shared_ptr<force_shared2> new_( int )
  {
    cout << "force_shared2::new_( int );\n";
    return boost::shared_ptr<force_shared2>( new force_shared2() );
  }
  
 private:
  // 同様に private に全てのコンストラクタを置き、一般からの構築を禁止
  force_shared2() {}
    
};

// 使い方
void test2()
{
  // 使う分にはその差を意識することはない。
  boost::shared_ptr<force_shared1> p1 = gintenlib::new_<force_shared1>();
  p1->foo();
  
  boost::shared_ptr<force_shared2> p2 = gintenlib::new_<force_shared2>(),
                                   p3 = gintenlib::new_<force_shared2>( 0 );
}

int main()
{
  test1();
  test2();
  
}
