#include "tcpnetwork.h"
#include <cstring>
#include "../structures/header.h"

using namespace network;
using namespace enc_hash;
using namespace structures;
using namespace std;

//Tcp socket(for client) constructor
tcpSocket::tcpSocket(const QString &senderName,const quint64 buffersize,QObject *parent):QTcpSocket(parent){
    this->buffer_size=buffersize;
    this->senderName=senderName;
    this->event=tcpSocket::Size;
    connect(this,SIGNAL(readyRead()),SLOT(client_receive_flag()));
}
//Moving event
void tcpSocket::client_move_section(){
    switch(this->event){
    case tcpSocket::Size:
        this->event=tcpSocket::Header;
        break;
    case tcpSocket::Header:
        this->event=(this->head_data.fileName().isNull()?tcpSocket::Msg:tcpSocket::File);
        break;
    default:
        this->event=tcpSocket::End;
        break;
    }
    this->client_process_event();
}
//Dispatch events
bool tcpSocket::client_process_event(){
    bool result;
#ifdef DEBUG
        qDebug()<<"Client Event:"<<this->event;
#endif
    switch(this->event){
    case tcpSocket::Size:
        result=this->client_send_size();
        break;
    case tcpSocket::Header:
        result=this->client_send_header();
        break;
    case tcpSocket::Msg:
        result=this->client_send_msg();
        break;
    case tcpSocket::File:
        result=this->client_send_file();
    case tcpSocket::End:
        emit this->sentData();
        result=true;
        break;
    }
    return result;
}
bool tcpSocket::client_flush(){
    bool result=this->flush();
    if(!result){
        this->setErrorString(tr("The data couldn't be sent"));
        emit this->error(QAbstractSocket::UnknownSocketError);
    }
    return result;
}

bool tcpSocket::client_send_size(){
    this->head_data=header(this->senderName,this->msg);
    QDataStream datastream(&this->tmp_buffer,QIODevice::WriteOnly);
    datastream<<this->head_data;
    quint16 size=(quint16)this->tmp_buffer.size();
    this->write((char*)&size,sizeof(size));
    if(!this->client_flush()) return false;
#ifdef DEBUG
    qDebug()<<"Client:size of header has been sent successfully. The size is:"<<size;
#endif
    this->client_move_section();
    return true;
}

bool tcpSocket::client_send_header(){
    this->write(this->tmp_buffer);
    if(!this->client_flush()) return false;
#ifdef DEBUG
    qDebug()<<"Client:header has been sent successfully.";
#endif
    this->client_move_section();
    return true;
}

bool tcpSocket::client_send_msg(){
    QBuffer memoryStream(&this->msg);
    if(!memoryStream.open(QIODevice::ReadOnly)){
        this->setErrorString(tr("Memory Stream couldn't open."));
        emit this->error(QAbstractSocket::UnknownSocketError);
        return false;
    }
    while(this->write(memoryStream.read(this->buffer_size))>0)
        if(!this->client_flush()){
        memoryStream.close();
        return false;
    }
    memoryStream.close();
#ifdef DEBUG
    qDebug()<<"Client:data has been sent successfully.";
#endif
    this->client_move_section();
    return true;
}

bool tcpSocket::client_send_file(){
return false;
}

void tcpSocket::client_receive_flag(){
    if(this->bytesAvailable()<4) return;
    QByteArray data=this->readAll();
#ifdef DEBUG
    qDebug()<<"Client:received flag:"<<(Flag)data.toInt();
#endif
    switch((Flag)data.toInt()){
    case tcpSocket::accepted:
        break;
    case tcpSocket::refused:
        this->setErrorString(tr("Access Refused"));
        emit this->error(QAbstractSocket::UnknownSocketError);
        return;
    }
}

tcpSocket &tcpSocket::operator<<(const QString &msg){
    if(this->state()!=QAbstractSocket::ConnectedState){
        this->setErrorString(tr("Haven't been connecting yet."));
        emit this->error(QAbstractSocket::NetworkError);
        return (*this);
    }
    this->msg=msg.toUtf8();
    this->event=tcpSocket::Size;
    this->client_process_event();
    return (*this);
}

tcpSocket &tcpSocket::operator<<(const QFileInfo &src_file){
    this->src_file=src_file;
    QtLockedFile file(src_file.fileName());
    if(!this->state()!=QAbstractSocket::ConnectedState)return (*this);
    this->head_data=header(this->senderName,QFileInfo(file));
    QByteArray tmp_buffer;
    QDataStream datastream(tmp_buffer);
    quint16 size=(quint16)tmp_buffer.size();
    datastream<<this->head_data;
    this->write((char*)&size,sizeof(size)/sizeof(char));
    if(!this->flush())return (*this);
    this->write(tmp_buffer);
    if(!this->flush())return (*this);
    tmp_buffer.clear();
    emit this->file_header_sent();
    if(!file.open(QIODevice::ReadOnly)){
        this->setErrorString(tr("The file stream couldn't open."));
        emit this->error(QAbstractSocket::UnknownSocketError);
        return (*this);
    }
    file.lock(QtLockedFile::ReadLock);
    while(this->write(file.read(this->buffer_size))>0&&!this->check_canceled_then_abort()){
        if(!this->flush())return (*this);
        emit this->sending_file_progress(file.pos());
    }
    file.unlock();
    file.close();
    emit this->sentData();
    return (*this);
}

//Threaded TCP socket (for client) implementation
threadedTcpSocket::threadedTcpSocket(const AddressAndPort &to,const QString &senderName, const quint64 buffersize, QObject *parent):QThread(parent){
    connect(this,SIGNAL(finished()),SLOT(deleteLater()));
    connect(this,SIGNAL(terminated()),SLOT(deleteLater()));
    this->locks[threadedTcpSocket::Mode].lockForWrite();
    this->locks[threadedTcpSocket::SenderName].lockForWrite();
    this->locks[threadedTcpSocket::BufferSize].lockForWrite();
    this->locks[threadedTcpSocket::RWMode].lockForWrite();
    this->locks[threadedTcpSocket::AddrAndPort].lockForWrite();
        this->_addrPort=to;
        this->mode=threadedTcpSocket::Client;
        this->_senderName=senderName;
        this->_buffersize=buffersize;
        this->open_mode=QAbstractSocket::ReadWrite;
    this->locks[threadedTcpSocket::Mode].unlock();
    this->locks[threadedTcpSocket::SenderName].unlock();
    this->locks[threadedTcpSocket::BufferSize].unlock();
    this->locks[threadedTcpSocket::RWMode].unlock();
    this->locks[threadedTcpSocket::AddrAndPort].unlock();
}
//Implementation of threadedTcpSocket
threadedTcpSocket &threadedTcpSocket::operator<<(const QString &msg){
    this->locks[threadedTcpSocket::Msg].lockForWrite();
        this->_msg=msg;
    this->locks[threadedTcpSocket::Msg].unlock();
    this->start();
    return (*this);
}
threadedTcpSocket &threadedTcpSocket::operator<<(const QFileInfo &file){
    this->locks[threadedTcpSocket::File].lockForWrite();
        this->_file=file;
    this->locks[threadedTcpSocket::File].unlock();
    this->start();
    return (*this);
}
