// TRSCodecBase.cs created with MonoDevelop
// User: tarai at 22:30 2008/06/29
//
// To change standard headers go to Edit->Preferences->Coding->Standard Headers
//

using System;
using Holo.Image;

namespace Holo.Codec.TRS {
	
	
	public class TRSCodecBase {
		
		/// <summary>
		/// Intermediate representation of IRaster object.
		/// </summary>
		public class RasterData {
			public int ClassIndex;
			public IRaster Raster;
			
			public RasterData(int index, IRaster raster) {
				ClassIndex = index;
				Raster = raster;
			}
			
			override public bool Equals(object obj) {
				return Equals((RasterData)obj);
			}
			
			public bool Equals(RasterData rhs) {
				return ClassIndex == rhs.ClassIndex &&
					Raster == rhs.Raster;
			}
			
			override public int GetHashCode() {
				return ClassIndex.GetHashCode() ^ Raster.GetHashCode();
			}
		};

		/// <summary>
		/// XML representation of IRaster class.
		/// </summary>
		public class RasterClass {
			public string Class;
			public int Index;
			public int BitsPerPixel;
			public int PixelStride;
			public int RowStride;
			public int NumChannels;
			
			public RasterClass() {
			}
			
			public RasterClass(IRaster raster) {
				Class = raster.GetType().Name;
				BitsPerPixel = raster.BitsPerPixel << 3;
				PixelStride = raster.PixelStride;
				RowStride = raster.RowStride;
				NumChannels = (raster.HasColorChannels || raster.HasAlpha)? raster.NumChannels: 0;
			}
			
			override public bool Equals(object obj) {
				if (obj is IRaster)
					return Equals((IRaster)obj);
				else if (obj is RasterClass) {
					return Equals((RasterClass)obj);
				}
				return false;
			}
			
			public bool Equals(IRaster raster) {
				return Equals(new RasterClass(raster));
			}
			
			public bool Equals(RasterClass rhs) {
				return Class == rhs.Class &&
						BitsPerPixel == rhs.BitsPerPixel &&
						PixelStride == rhs.PixelStride &&
						RowStride == rhs.RowStride &&
						NumChannels == rhs.NumChannels;
			}
			
			override public int GetHashCode() {
				return Class.GetHashCode() ^ 
						BitsPerPixel.GetHashCode() ^ 
						PixelStride.GetHashCode() ^ 
						RowStride.GetHashCode() ^ 
						NumChannels.GetHashCode();
			}
		};

		/// <summary>
		/// XML representation of IRaster instance.
		/// </summary>
		public class Raster {
			public int ClassIndex;
			public byte[] PrimaryAlpha;
			public byte[] PrimaryColor;
			public byte[] Data;
			
			public Raster() {
			}
			
			public Raster(RasterClass rasterClass, RasterData data) {
				ClassIndex = rasterClass.Index;
				PrimaryAlpha = data.Raster.PrimaryAlphaValue;
				PrimaryColor = data.Raster.PrimaryColor;
				if (data.Raster.HasColorChannels || data.Raster.HasAlpha) {
					Data = data.Raster.Buffer;
				}
			}
		}
		
		/// <summary>
		/// XML representation of container of IRaster classes and instances.
		/// </summary>
		public class RasterClassDictionary {
			public RasterClass[] Classes;
			public Raster[] Instances;
			
			public RasterClassDictionary() {
			}
		};

		/// <summary>
		/// XML representation of ISurface instance.
		/// </summary>
		public class Surface {
			public string Class;
			public string FactoryClass;
			public int OffsetX;
			public int OffsetY;
			public int Width;
			public int Height;
			public int TileWidth;
			public int TileHeight;
			public RasterClassDictionary RasterClasses;
			public int[] Mapping;
			
			public Surface() {
			}

			public Surface(ISurface surface) {
				Class = surface.GetType().Name;
				if (surface.RasterFactory is Holo.IWrapper) {
					FactoryClass = ((IWrapper)surface.RasterFactory).Source.GetType().Name;
				} else {
					FactoryClass = surface.RasterFactory.GetType().Name;
				}
				OffsetX = surface.OffsetX;
				OffsetY = surface.OffsetY;
				Width = surface.Width;
				Height = surface.Height;
				TileWidth = surface.TileWidth;
				TileHeight = surface.TileHeight;
				RasterClasses = new RasterClassDictionary();
			}
		}		
	}
}
