// /home/tarai/Projects/gsaw/Viewport.cs created with MonoDevelop
// User: tarai at 22:41 2008/05/05
//
// To change standard headers go to Edit->Preferences->Coding->Standard Headers
//

using System;

namespace Lawrence.View {
	
	using Lawrence.Common;
	using Holo.Image;
	using Holo.Image.Generic;
	using Holo.Operation;
	
	public class Viewport {
		public delegate void UpdateViewHandler(int minX, int minY, int maxX, int maxY);
		private Image image;
		private GenericSurface scaledProjection;
		private GenericSurface transformedProjection;
		private RotateOperation roperation;
		private TransformOperation operation;
		
		private int viewWidth;
		private int viewHeight;
		private int offsetX;
		private int offsetY;
		private bool disableUpdate;
		private UpdateViewHandler onUpdateView;
		private double cos, sin, rcos, rsin;
		
		public int WidthOfView {
			get { return viewWidth; }
		}
		
		public int HeightOfView {
			get { return viewHeight; }
		}
		
		public int SourceScale {
			get { return operation.SourceScale; }
		}
		
		public int DestScale {
			get { return operation.DestScale; }
		}
		
		public double Degree {
			get { return roperation.Degree; }
			set {
				double degree = value;
				roperation.Degree = degree;
				sin = Math.Sin(degree * Math.PI / 180);
				cos = Math.Cos(degree * Math.PI / 180);
				rsin = Math.Sin(-degree * Math.PI / 180);
				rcos = Math.Cos(-degree * Math.PI / 180);
				if (transformedProjection != null) {
					Array.Clear(transformedProjection.Raster.Buffer, 0, transformedProjection.Raster.Buffer.Length);
					OnUpdateScaledView(0, 0, transformedProjection.Width, transformedProjection.Height);
				} else {
					Configure();
				}
			}
		}
		
		public int OffsetX {
			get { return offsetX; }
		}
		
		public int OffsetY {
			get { return offsetY; }
		}
		
		public Viewport(Image image) {
			Initialize(image);
		}
		
		public UpdateViewHandler OnUpdateView {
			get { return onUpdateView; }
			set { onUpdateView = value; }
		}
		
		public Viewport(Image image, int viewWidth, int viewHeight) {
			Initialize(image);

			this.viewWidth = viewWidth;
			this.viewHeight = viewHeight;
			operation.SourceScale = image.Width;
			operation.DestScale = viewWidth;
			
			Configure();
		}
		
		public Viewport(Image image, int viewWidth, int viewHeight, int maxWidth, int maxHeight) {
			Initialize(image);

			transformedProjection = new GenericSurface(0, 0, 
			                                           maxWidth,
			                                           maxHeight, 
			                                           false);
			this.viewWidth = viewWidth;
			this.viewHeight = viewHeight;
			operation.SourceScale = image.Width;
			operation.DestScale = viewWidth;
			
			Configure();
		}

		public void Initialize(Image image) {
			this.image = image;
			this.operation = new SubsampleOperation();
			operation.Operator = ((SubsampleOperation)operation).ApplySubSample;
			image.OnUpdateArea += this.OnUpdateImageArea;
			operation.SourceScale = 1;
			operation.DestScale = 1;
			roperation = new RotateOperation();
			Degree = 0;
			offsetX = 0;
			offsetY = 0;
			viewWidth = 0;
			viewHeight = 0;
			transformedProjection = null;
			disableUpdate = false;
		}
		
		public void Configure() {
			if (viewWidth == 0 || viewHeight == 0) {
				return;
			}

			int vWidth = viewWidth + (operation.GridSize * operation.DestScale / operation.SourceScale);
			int vHeight = viewHeight + (operation.GridSize * operation.DestScale / operation.SourceScale);
			
			Console.WriteLine("vWidth={0}, vHeight={1}, GridSize={2}",vWidth, vHeight, operation.GridSize);
			
			if (transformedProjection == null || 
			    vWidth > transformedProjection.Width ||
			    vHeight > transformedProjection.Height) {

				scaledProjection = new GenericSurface(0, 0, 
				                                           vWidth,
				                                           vHeight, 
				                                           false);
				transformedProjection = new GenericSurface(0, 0, 
				                                           viewWidth,
				                                           viewHeight, 
				                                           false);
			}
			scaledProjection.MakeRastersWriteable(0, 0, vWidth, vHeight);
			Array.Clear(scaledProjection.Raster.Buffer, 0, scaledProjection.Raster.Buffer.Length);
			transformedProjection.MakeRastersWriteable(0, 0, viewWidth, viewHeight);
			Array.Clear(transformedProjection.Raster.Buffer, 0, transformedProjection.Raster.Buffer.Length);
			OnUpdateImageArea(0, 0, image.Width, image.Height);
		}
		
		public void Resize(int width, int height) {
			viewWidth = width;
			viewHeight = height;
			Configure();
		}
		
		public void MoveOffsetTo(int x, int y) {
			offsetX = x;
			offsetY = y;
			Configure();
		}
		
		public void Rescale(int sourceScale, int destScale) {
			if (sourceScale > 0 || destScale > 0) {
				operation.SourceScale = sourceScale;
				operation.DestScale = destScale;
				Configure();
			}
		}
		
		public void SetViewport(int width, int height, int offsetX, int offsetY, int sourceScale, int destScale) {
			disableUpdate = true;
			Resize(width, height);
			MoveOffsetTo(offsetX, offsetY);
			Rescale(sourceScale, destScale);
			disableUpdate = false;
			OnUpdateImageArea(0, 0, image.Width, image.Height);
		}
		
		protected void OnUpdateImageArea(int minX, int minY, int maxX, int maxY) {
			try {
				D.StopWatches.Viewport_OnUpdateImage.Start();
					
			if (transformedProjection == null || disableUpdate)
				return;
			ImageXYToScaledXY(ref minX, ref minY);
			ImageXYToScaledXY(ref maxX, ref maxY);
			minX = Math.Max(0, minX - 1);
			minY = Math.Max(0, minY - 1);
			maxX = Math.Min(maxX + 1, viewWidth);
			maxY = Math.Min(maxY + 1, viewHeight);
			
			if (minX < maxX && minY < maxY) {
				int imageMinX = minX;
				int imageMinY = minY;
				int imageMaxX = maxX;
				int imageMaxY = maxY;
				ScaledXYToImageXY(ref imageMinX, ref imageMinY);
				ScaledXYToImageXY(ref imageMaxX, ref imageMaxY);
//				Console.WriteLine("Update-Viewport:"+imageMinX+","+imageMinY+"-"+imageMaxX+","+imageMaxY+"->"+minX+","+minY+"-"+maxX+","+maxY);
				operation.GetNormalizedBoundary(image.Projection, ref imageMinX, ref imageMinY, ref imageMaxX, ref imageMaxY);
				minX = imageMinX * operation.DestScale / operation.SourceScale;
				minY = imageMinY * operation.DestScale / operation.SourceScale;
				minX -= (offsetX / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale;
				minY -= (offsetY / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale;
				maxX = imageMaxX * operation.DestScale / operation.SourceScale;
				maxY = imageMaxY * operation.DestScale / operation.SourceScale;
				maxX -= (offsetX / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale;
				maxY -= (offsetY / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale;
				operation.Apply(image.Projection, imageMinX, imageMinY, imageMaxX, imageMaxY, minX, minY, scaledProjection);

#if USE_ROTATION
				OnUpdateScaledView(minX, minY, maxX, maxY);
#else
				minX -= offsetX * operation.DestScale / operation.SourceScale - (offsetX / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale; 
				minY -= offsetY * operation.DestScale / operation.SourceScale - (offsetY / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale;
				maxX -= offsetX * operation.DestScale / operation.SourceScale - (offsetX / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale; 
				maxY -= offsetY * operation.DestScale / operation.SourceScale - (offsetY / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale;
				transformedProjection = scaledProjection;
				if (onUpdateView != null) {
					onUpdateView(minX, minY, maxX, maxY);
				}
#endif
			}
				
			} finally {
				D.StopWatches.Viewport_OnUpdateImage.Stop();
			}
		}
		
		protected void OnUpdateScaledView(int minX, int minY, int maxX, int maxY) {
			try {
				try {
				D.StopWatches.Viewport_OnUpdateScaledView.Start();
				} catch (Exception e) {
					Console.WriteLine(e);
				}
			int scaledViewOffsetX = offsetX * operation.DestScale / operation.SourceScale - (offsetX / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale; 
			int scaledViewOffsetY = offsetY * operation.DestScale / operation.SourceScale - (offsetY / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale;
			int destGridSize = operation.GridSize * operation.DestScale / operation.SourceScale;
			
			minX = Math.Max(minX, scaledViewOffsetX);
			minY = Math.Max(minY, scaledViewOffsetY);
			maxX = Math.Min(maxX, scaledProjection.Width);
			maxY = Math.Min(maxY, scaledProjection.Height);

			int srcCX = (scaledProjection.Width - destGridSize) / 2 + scaledViewOffsetX;
			int srcCY = (scaledProjection.Height - destGridSize) / 2 + scaledViewOffsetY;
			int destCX = transformedProjection.Width / 2;
			int destCY = transformedProjection.Height / 2;
			roperation.Apply(scaledProjection, srcCX, srcCY, minX, minY, maxX, maxY, destCX, destCY, transformedProjection);
				
			if (onUpdateView != null) {
//					onUpdateView(minX, minY, maxX, maxY);
//					Console.WriteLine("Last Updated: {0},{1}-{2},{3}",roperation.LastDestStartX, roperation.LastDestStartY, roperation.LastDestEndX, roperation.LastDestEndY);
				onUpdateView(roperation.LastDestStartX, roperation.LastDestStartY, roperation.LastDestEndX, roperation.LastDestEndY);
			}
			} finally {
				D.StopWatches.Viewport_OnUpdateScaledView.Stop();
			}
		}
		
		public void ImageCoordToViewportCoord(ImageCoordinate coords) {
			ImageXYToScaledXY(ref coords.X, ref coords.Y);
#if USE_ROTATION
			ScaledXYToViewportXY(ref coords.X, ref coords.Y);
#endif
		}
		
		public void ViewportCoordToImageCoord(ImageCoordinate coords) {
#if USE_ROTATION
			ViewportXYToScaledXY(ref coords.X, ref coords.Y);
#endif
			ScaledXYToImageXY(ref coords.X, ref coords.Y);
		}
		
		protected void ScaledXYToViewportXY(ref double x, ref double y) {
			double oldX = x;
			double oldY = y;
			double originX = transformedProjection.Width / 2;
			double originY = transformedProjection.Height / 2;
			x = cos * (oldX-originX) - sin * (oldY-originY) + originX;
			y = sin * (oldX-originX) + cos * (oldY-originY) + originY;
		}

		protected void ViewportXYToScaledXY(ref double x, ref double y) {
			double oldX = x;
			double oldY = y;
			double originX = transformedProjection.Width / 2;
			double originY = transformedProjection.Height / 2;
			x = rcos * (oldX-originX) - rsin * (oldY-originY) + originX;
			y = rsin * (oldX-originX) + rcos * (oldY-originY) + originY;
		}
		
		protected void ImageXYToScaledXY(ref int x, ref int y) {
			x = (x - offsetX) * operation.DestScale / operation.SourceScale;
			y = (y - offsetY) * operation.DestScale / operation.SourceScale;
		}
		
		protected void ScaledXYToImageXY(ref int x, ref int y) {
			x = x * operation.SourceScale / operation.DestScale + offsetX;
			y = y * operation.SourceScale / operation.DestScale + offsetY;
		}
		
		protected void ImageXYToScaledXY(ref double x, ref double y) {
			x = (x - offsetX) * operation.DestScale / operation.SourceScale;
			y = (y - offsetY) * operation.DestScale / operation.SourceScale;
		}
		
		protected void ScaledXYToImageXY(ref double x, ref double y) {
			x = x * operation.SourceScale / operation.DestScale + offsetX;
			y = y * operation.SourceScale / operation.DestScale + offsetY;
		}
		
		public void RenderTo(Gdk.Drawable drawable, Gdk.Rectangle rectangle)	{
			if (rectangle.X > viewWidth || rectangle.Y > viewHeight)
				return;
			
			if (rectangle.X + rectangle.Width > viewWidth)
				rectangle.Width = viewWidth - rectangle.X; 
			if (rectangle.Y + rectangle.Height > viewHeight)
				rectangle.Height = viewHeight - rectangle.Y; 

//			Console.WriteLine("Viewport:RenderTo:"+rectangle.ToString());
			
			ISurfaceIterator iter;
#if USE_ROTATION
			int projectionOffsetX = rectangle.X; 
			int projectionOffsetY = rectangle.Y;
#else
			int projectionOffsetX = rectangle.X + offsetX * operation.DestScale / operation.SourceScale - (offsetX / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale; 
			int projectionOffsetY = rectangle.Y + offsetY * operation.DestScale / operation.SourceScale - (offsetY / operation.GridSize) * operation.GridSize * operation.DestScale / operation.SourceScale;
#endif
			iter = transformedProjection.GetIteratorAt(projectionOffsetX, projectionOffsetY);
			iter.SetRange(projectionOffsetX, projectionOffsetY, projectionOffsetX + rectangle.Width, projectionOffsetY + rectangle.Height);

			while (!iter.IsYEnded()) {
				while (!iter.IsXEnded()) {
					byte[] buf = new byte[rectangle.Width * rectangle.Height * 3];
					int bufOffset = 0;
					int bufRowStride = rectangle.Width * 3;
					int offset = iter.OffsetInBuffer;
					for (int y = 0; y < rectangle.Height; y ++) {
						Array.Copy(iter.Raster.Buffer, offset,  buf, bufOffset, bufRowStride);
						offset += iter.Raster.RowStride;
						bufOffset += bufRowStride;
					}
					drawable.DrawRgbImage(new Gdk.GC(drawable),
					                      rectangle.Left, rectangle.Top,
					                      rectangle.Width, rectangle.Height,
					                      Gdk.RgbDither.Normal, buf, bufRowStride);
					iter.AddX(iter.WidthOfRaster);
				}
				iter.ResetXAndAddY(iter.HeightOfRaster);
			}
		}		
		
	}
}
