// /home/tarai/Projects/gsaw/gsaw/BaseBrushProvider.cs created with MonoDevelop
// User: tarai at 23:01 2008/05/21
//
// To change standard headers go to Edit->Preferences->Coding->Standard Headers
//

using System;

namespace Lawrence.Tool {
	using Lawrence.Common;
	using Holo.Image.Generic;
	using Holo.Image.Tiled;
	using Holo.Image;
	using Holo.Operation;
	
	public class BaseBrushProvider : IColorAllocatable {
		
		// for brush mask
		private GenericSurface[] dabCache;
		private ISurface dabSurface;
		private double radius;
		private byte[] color;
		private double opacity;
		private double edgeRate;
		private SubsampleOperation subsampler;
		private SubsampleOperation halfSampler;
		private double stepRate;
		
		public double Opacity {
			get { return opacity; }
			set { opacity = value; }
		}
		
		public double EdgeRate {
			get { return edgeRate; }
			set { 
				edgeRate = value; 
				dabSurface = null;
				Array.Clear(dabCache, 0, dabCache.Length);
			}
		}
		
		public double StepRate {
			get { return stepRate; }
			set { stepRate = value; }
		}
		
		public byte[] Color {
			get { return color; }
			set { 
				color = value; 
				dabSurface = null;
				Array.Clear(dabCache, 0, dabCache.Length);
			}
		}
		
		public double Radius {
			get { return radius; }
			set {
				radius = value;
				dabSurface = null;
				Array.Clear(dabCache, 0, dabCache.Length);
			}
		}


		public BaseBrushProvider() {
			Initialize();
		}

		public void Initialize() {
			radius = 16;
			edgeRate = 1.0;
			opacity = 1.0;
			stepRate = 0.1;
			dabCache = new GenericSurface[101];
			color = new byte[3];
			color[0] = color[1] = color[2] = 0;
		}
		
		public void Setup(BrushTool tool) {
			tool.SteppingDistance = radius * 2 * stepRate;
			byte[] newAlpha = {0};

			tool.CreateDab = this.CreateDab;
			if (color == null) {
				tool.RasterFactory = new CompactMonochromeRasterFactory(color, newAlpha);
			}
			dabSurface = CreateDabSurface(radius * 2);
		}

		
		public IRasterFactory CreateRasterFactory(PaintContext context) {
			return new MonochromeRasterFactory(color);
		}

		
		const int BRUSH_SOURCE_SCALE = 2;
		
		protected virtual ISurface CreateDabSurface(double dsize) {
			GenericSurface dabSurface;
			int size = (int)(dsize + 0.9);
			size *= BRUSH_SOURCE_SCALE;
			if (size < 32 * BRUSH_SOURCE_SCALE)
				size = 32 * BRUSH_SOURCE_SCALE;
			Console.WriteLine("size={0}", size);
			dabSurface = new GenericSurface(0, 0, size, size, true, 
			                                CreateRasterFactory(null));
			
			if (size <= 32 * BRUSH_SOURCE_SCALE) {
				halfSampler = new SubsampleOperation();
				halfSampler.Operator = halfSampler.ApplySubSample;
				halfSampler.SourceScale = 2;
				halfSampler.DestScale = 1;
			} else {
				halfSampler = null;
			}

			dabSurface.MakeRastersWriteable(0, 0, size, size);
			byte[] buf = dabSurface.Raster.Buffer;
			int offset = 0;
			double radiusSqr = (radius * BRUSH_SOURCE_SCALE)  * (radius * BRUSH_SOURCE_SCALE);
			double innerRadius = edgeRate * radius * BRUSH_SOURCE_SCALE;

			if (subsampler == null) {
				subsampler = new SubsampleOperation();
				subsampler.Operator = subsampler.ApplySubSample;
			}
			
			subsampler.SourceScale = 100 * BRUSH_SOURCE_SCALE;
			if (size <= 32)
				subsampler.SourceScale = (int)(size / dsize * 100);
			
			for (int y = 0; y < size; y ++) {
				for (int x = 0; x < size; x ++) {
					double ydiff = (radius * BRUSH_SOURCE_SCALE - y ) * (radius * BRUSH_SOURCE_SCALE - y);
					double xdiff = (radius * BRUSH_SOURCE_SCALE - x ) * (radius * BRUSH_SOURCE_SCALE - x);
					double dist = Math.Sqrt(xdiff + ydiff);
					if (xdiff + ydiff <= radiusSqr) {
						byte alpha = 255;
						if (dist > innerRadius)
							alpha = (byte)((1 - (dist - innerRadius)  / (radius * BRUSH_SOURCE_SCALE)) * 255);
						buf[offset] = alpha;
					}
					offset ++;
				}
			}
			return dabSurface;
		}

		public ISurface CreateDab(PaintContext context, ImageCoordinate coords, BrushTool tool) {
			double r = (int)(radius * coords.Pressure);
			if (dabSurface == null) {
				dabSurface = CreateDabSurface(radius * 2);
			}
			
			int pressureRate = (int)(coords.Pressure * 100);
			int cachedSize = (int)(radius * 2 + 0.9);

			if (cachedSize < 1)
				cachedSize = 1;

			GenericSurface dabSurfaceForSize = dabCache[pressureRate];
			if (dabSurfaceForSize == null) {
				if (halfSampler != null) {
					subsampler.DestScale = 2 * pressureRate;
//					Console.WriteLine("scale={0}/{1}", subsampler.DestScale, subsampler.SourceScale);
					dabSurfaceForSize = new GenericSurface(0, 0, cachedSize * 2 + 3, cachedSize * 2 + 3, true, 
					                                CreateRasterFactory(null));
					subsampler.Apply(dabSurface, 0, 0, dabSurface.Width, dabSurface.Height, 1, 1, dabSurfaceForSize);
				} else {
					subsampler.DestScale = pressureRate;
					dabSurfaceForSize = new GenericSurface(0, 0, cachedSize, cachedSize, true, 
					                                CreateRasterFactory(null));
					subsampler.Apply(dabSurface, 0, 0, dabSurface.Width, dabSurface.Height, 0, 0, dabSurfaceForSize);
				}
				dabCache[pressureRate] = dabSurfaceForSize;
			}

			if (halfSampler != null) {
				int dabOffsetX = 1 - (int)(((coords.X - r ) * 2) % 2);
				int dabOffsetY = 1 - (int)(((coords.Y - r ) * 2) % 2);
//				Console.WriteLine("r="+radius+",cached={0},(x,y)={1},{2}", cachedSize, dabOffsetX, dabOffsetY);

				GenericSurface dab = new GenericSurface(0, 0, cachedSize + 1, cachedSize + 1, true, 
				                                                   CreateRasterFactory(null));
				halfSampler.Apply(dabSurfaceForSize, dabOffsetX, dabOffsetY, 
				                  dabOffsetX + cachedSize * 2 + 2, dabOffsetY + cachedSize * 2 + 2, 0, 0, dab);
//				int minX = (int)(coords.X + dabOffsetX * 0.5  - (int)(0.45 + r));
//				int minY = (int)(coords.Y + dabOffsetY * 0.5 - (int)(0.45 + r));
				int minX = (int)(coords.X - cachedSize / 4);
				int minY = (int)(coords.Y - cachedSize / 4);
//				int minX = (int)(coords.X - r);
//				int minY = (int)(coords.Y - r);
				dab.OffsetX = minX;
				dab.OffsetY = minY;
				return dab;
			} else {
				int minX = (int)(coords.X - r);
				int minY = (int)(coords.Y - r);
				dabSurfaceForSize.OffsetX = minX;
				dabSurfaceForSize.OffsetY = minY;
				((MonochromeRaster)dabSurfaceForSize.Raster).PrimaryColor = color;
				return dabSurfaceForSize;
			}
		}
	
	}
}