//-----------------------------------------------------------------------------
// Gura win32 module
//-----------------------------------------------------------------------------
#include "Module_win32.h"

Gura_BeginModule(win32)

Gura_DeclarePrivSymbol(import_const);
Gura_DeclarePrivSymbol(openkey);

Gura_DeclarePrivSymbol(non_volatile);
Gura_DeclarePrivSymbol(volatile_);
Gura_DeclarePrivSymbol(backup_restore);

Gura_DeclarePrivSymbol(create_link);
Gura_DeclarePrivSymbol(create_sub_key);
Gura_DeclarePrivSymbol(enumerate_sub_keys);
Gura_DeclarePrivSymbol(execute);
Gura_DeclarePrivSymbol(notify);
Gura_DeclarePrivSymbol(query_value);
Gura_DeclarePrivSymbol(set_value);
Gura_DeclarePrivSymbol(all_access);
Gura_DeclarePrivSymbol(read);
Gura_DeclarePrivSymbol(write);

//-----------------------------------------------------------------------------
// utilities
//-----------------------------------------------------------------------------
void SetError(Signal &sig, DWORD dwErrCode)
{
	LPVOID lpMsgBuf;
	::FormatMessageW(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM |
		FORMAT_MESSAGE_IGNORE_INSERTS, NULL, dwErrCode,
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPWSTR)&lpMsgBuf, 0, NULL);
	sig.SetError(ERR_SystemError, "Error: %s",
			BSTRToString(reinterpret_cast<OLECHAR *>(lpMsgBuf)).c_str());
	::LocalFree(lpMsgBuf);
}

Value RegDataToValue(Environment &env, Signal sig,
								DWORD dwType, LPCBYTE lpData, DWORD cbData)
{
	Value result;
	if (dwType == REG_BINARY) {
		result.InitAsBinary(env, reinterpret_cast<const char *>(lpData), cbData);
	} else if (dwType == REG_DWORD || dwType == REG_DWORD_LITTLE_ENDIAN) {
		result = Value(*reinterpret_cast<const DWORD *>(lpData));
	} else if (dwType == REG_DWORD_BIG_ENDIAN) {
		
	} else if (dwType == REG_EXPAND_SZ) {
		
	} else if (dwType == REG_LINK) {
		
	} else if (dwType == REG_MULTI_SZ) {
		ValueList &valList = result.InitAsList(env);
		size_t bytesSum = 0;
		while (bytesSum + 1 < static_cast<size_t>(cbData)) {
			Value value(env, reinterpret_cast<const char *>(lpData));
			valList.push_back(value);
			size_t bytes = ::strlen(reinterpret_cast<const char *>(lpData)) + 1;
			lpData += bytes;
			bytesSum += bytes;
		}
	} else if (dwType == REG_NONE) {
		// nothing to do
	} else if (dwType == REG_QWORD || dwType == REG_QWORD_LITTLE_ENDIAN) {
		
	} else if (dwType == REG_RESOURCE_LIST) {
		
	} else if (dwType == REG_SZ) {
		//**** encoding converter is necessary ****
		result = Value(env, reinterpret_cast<const char *>(lpData));
	}
	return result;
}

bool ValueToRegData(Environment &env, Signal sig, const Value &value,
								DWORD *pdwType, LPBYTE *lppData, DWORD *pcbData)
{
	if (value.IsNumber()) {
		*pdwType = REG_DWORD;
		*pcbData = sizeof(DWORD);
		*lppData = reinterpret_cast<LPBYTE>(::LocalAlloc(LMEM_FIXED, *pcbData));
		*reinterpret_cast<DWORD *>(*lppData) = value.GetULong();
		return true;
	} else if (value.IsBinary()) {
		const Binary &buff = value.GetBinary();
		*pdwType = REG_BINARY;
		*pcbData = static_cast<DWORD>(buff.size());
		*lppData = reinterpret_cast<LPBYTE>(::LocalAlloc(LMEM_FIXED, *pcbData));
		::memcpy(*lppData, buff.data(), *pcbData);
		return true;
	} else if (value.IsList()) {
		size_t bytesSum = 0;
		foreach_const (ValueList, pValue, value.GetList()) {
			if (!pValue->IsString()) goto error_done;
			const String &str = pValue->GetStringSTL();
			size_t bytes = str.size() + 1;
			bytesSum += bytes;
		}
		bytesSum++;
		*pdwType = REG_MULTI_SZ;
		*pcbData = static_cast<DWORD>(bytesSum);
		*lppData = reinterpret_cast<LPBYTE>(::LocalAlloc(LMEM_FIXED, *pcbData));
		BYTE *p = *lppData;
		foreach_const (ValueList, pValue, value.GetList()) {
			if (!pValue->IsString()) goto error_done;
			const String &str = pValue->GetStringSTL();
			size_t bytes = str.size() + 1;
			//**** encoding converter is necessary ****
			::memcpy(p, str.c_str(), bytes);
			p += bytes;
		}
		*p = '\0';
		return true;
	} else if (value.IsString()) {
		const String &str = value.GetStringSTL();
		*pdwType = REG_SZ;
		*pcbData = static_cast<DWORD>(str.size() + 1);
		*lppData = reinterpret_cast<LPBYTE>(::LocalAlloc(LMEM_FIXED, *pcbData));
		//**** encoding converter is necessary ****
		::memcpy(*lppData, str.c_str(), *pcbData);
		return true;
	}
error_done:
	sig.SetError(ERR_ValueError, "invalid data type for registry");
	return false;
}

//-----------------------------------------------------------------------------
// Symbol converter
//-----------------------------------------------------------------------------
DWORD SymbolToOption(Signal sig, const Symbol *pSymbol)
{
	DWORD dwOption = 0;
	if (pSymbol->IsIdentical(Gura_PrivSymbol(non_volatile))) {
		dwOption = REG_OPTION_NON_VOLATILE;
	} else if (pSymbol->IsIdentical(Gura_PrivSymbol(volatile_))) {
		dwOption = REG_OPTION_VOLATILE;
	} else if (pSymbol->IsIdentical(Gura_PrivSymbol(backup_restore))) {
		dwOption = REG_OPTION_BACKUP_RESTORE;
	} else {
		sig.SetError(ERR_ValueError, "option must be `non_volatile, `volatile or `backup_restore");
	}
	return dwOption;
}

REGSAM SymbolToSamDesired(Signal sig, const Symbol *pSymbol)
{
	REGSAM samDesired = 0;
	if (pSymbol->IsIdentical(Gura_PrivSymbol(create_link))) {
		samDesired = KEY_CREATE_LINK;
	} else if (pSymbol->IsIdentical(Gura_PrivSymbol(create_sub_key))) {
		samDesired = KEY_CREATE_SUB_KEY;
	} else if (pSymbol->IsIdentical(Gura_PrivSymbol(enumerate_sub_keys))) {
		samDesired = KEY_ENUMERATE_SUB_KEYS;
	} else if (pSymbol->IsIdentical(Gura_PrivSymbol(execute))) {
		samDesired = KEY_EXECUTE;
	} else if (pSymbol->IsIdentical(Gura_PrivSymbol(notify))) {
		samDesired = KEY_NOTIFY;
	} else if (pSymbol->IsIdentical(Gura_PrivSymbol(query_value))) {
		samDesired = KEY_QUERY_VALUE;
	} else if (pSymbol->IsIdentical(Gura_PrivSymbol(set_value))) {
		samDesired = KEY_SET_VALUE;
	} else if (pSymbol->IsIdentical(Gura_PrivSymbol(all_access))) {
		samDesired = KEY_ALL_ACCESS;
	} else if (pSymbol->IsIdentical(Gura_PrivSymbol(read))) {
		samDesired = KEY_READ;
	} else if (pSymbol->IsIdentical(Gura_PrivSymbol(write))) {
		samDesired = KEY_WRITE;
	} else {
		sig.SetError(ERR_ValueError, "invalid symbol for samDesired");
	}
	return samDesired;
}

//-----------------------------------------------------------------------------
// Object_RegKey implementation
//-----------------------------------------------------------------------------
Object_RegKey::~Object_RegKey()
{
	::RegCloseKey(_hKey);
}

Object *Object_RegKey::Clone() const
{
	return NULL;
}

String Object_RegKey::ToString(Signal sig, bool exprFlag)
{
	return String("<regkey>");
}

//-----------------------------------------------------------------------------
// Gura interfaces for Object_RegKey
//-----------------------------------------------------------------------------
// win32.RegKey#createkey(subkey:string, option?:symbol, samDesired[]?:symbol:nomap):map {block?}
Gura_DeclareMethod(RegKey, createkey)
{
	SetMode(RSLTMODE_Normal, FLAG_Map);
	DeclareArg(env, "subkey", VTYPE_String);
	DeclareArg(env, "option", VTYPE_Symbol, OCCUR_ZeroOrOnce);
	DeclareArg(env, "samDesired", VTYPE_Symbol, OCCUR_ZeroOrOnce, true, true);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

Gura_ImplementMethod(RegKey, createkey)
{
	Object_RegKey *pSelf = Object_RegKey::GetSelfObj(args);
	HKEY hKey = pSelf->GetKey();
	const char *lpSubKey = args.GetString(0);
	DWORD dwOptions = args.IsSymbol(1)?
			SymbolToOption(sig, args.GetSymbol(1)) : REG_OPTION_NON_VOLATILE;
	if (sig.IsSignalled()) return Value::Null;
	REGSAM samDesired = 0;
	if (args.IsList(2)) {
		foreach_const (ValueList, pValue, args.GetList(2)) {
			samDesired |= SymbolToSamDesired(sig, pValue->GetSymbol());
			if (sig.IsSignalled()) return Value::Null;
		}
	} else {
		samDesired = KEY_ALL_ACCESS;
	}
	HKEY hKeyResult;
	DWORD dwDisposition;
	DWORD dwErrCode = ::RegCreateKeyEx(hKey, lpSubKey, 0, NULL,
					dwOptions, samDesired, NULL, &hKeyResult, &dwDisposition);
	if (dwErrCode != ERROR_SUCCESS) {
		SetError(sig, dwErrCode);
		return Value::Null;
	}
	Value result(new Object_RegKey(hKeyResult));
	return ReturnValue(env, sig, args, result);
}

// win32.RegKey#openkey(subkey:string, samDesired[]?:symbol:nomap):map {block?}
Gura_DeclareMethod(RegKey, openkey)
{
	SetMode(RSLTMODE_Normal, FLAG_Map);
	DeclareArg(env, "subkey", VTYPE_String);
	DeclareArg(env, "samDesired", VTYPE_Symbol, OCCUR_ZeroOrOnce, true, true);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

Gura_ImplementMethod(RegKey, openkey)
{
	Object_RegKey *pSelf = Object_RegKey::GetSelfObj(args);
	HKEY hKey = pSelf->GetKey();
	const char *lpSubKey = args.GetString(0);
	if (sig.IsSignalled()) return Value::Null;
	REGSAM samDesired = 0;
	if (args.IsList(1)) {
		foreach_const (ValueList, pValue, args.GetList(2)) {
			samDesired |= SymbolToSamDesired(sig, pValue->GetSymbol());
			if (sig.IsSignalled()) return Value::Null;
		}
	} else {
		samDesired = KEY_ALL_ACCESS;
	}
	HKEY hKeyResult;
	DWORD dwErrCode = ::RegOpenKeyEx(hKey, lpSubKey, 0, samDesired, &hKeyResult);
	if (dwErrCode != ERROR_SUCCESS) {
		SetError(sig, dwErrCode);
		return Value::Null;
	}
	Value result(new Object_RegKey(hKeyResult));
	return ReturnValue(env, sig, args, result);
}

// win32.RegKey#deletekey(subkey:string):map:void
Gura_DeclareMethod(RegKey, deletekey)
{
	SetMode(RSLTMODE_Void, FLAG_Map);
	DeclareArg(env, "subkey", VTYPE_String);
}

Gura_ImplementMethod(RegKey, deletekey)
{
	Object_RegKey *pSelf = Object_RegKey::GetSelfObj(args);
	HKEY hKey = pSelf->GetKey();
	const char *lpSubKey = args.GetString(0);
	DWORD dwErrCode = ::RegDeleteKey(hKey, lpSubKey);
	if (dwErrCode != ERROR_SUCCESS) {
		SetError(sig, dwErrCode);
		return Value::Null;
	}
	return Value::Null;
}

// win32.RegKey#enumkey(samDesired[]?:symbol):[openkey] {block?}
Gura_DeclareMethod(RegKey, enumkey)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
	DeclareArg(env, "samDesired", VTYPE_Symbol, OCCUR_ZeroOrOnce, true);
	DeclareBlock(OCCUR_ZeroOrOnce);
	DeclareAttr(Gura_PrivSymbol(openkey));
}

Gura_ImplementMethod(RegKey, enumkey)
{
	Object_RegKey *pSelf = Object_RegKey::GetSelfObj(args);
	REGSAM samDesired = 0;
	if (!args.IsSet(Gura_PrivSymbol(openkey))) {
		// nothing to do
	} else if (args.IsList(0)) {
		foreach_const (ValueList, pValue, args.GetList(2)) {
			samDesired |= SymbolToSamDesired(sig, pValue->GetSymbol());
			if (sig.IsSignalled()) return Value::Null;
		}
	} else {
		samDesired = KEY_ALL_ACCESS;
	}
	Iterator *pIterator =
			new Iterator_RegEnumKey(Object_RegKey::Reference(pSelf), samDesired);
	return ReturnIterator(env, sig, args, pIterator);
}

// win32.RegKey#setvalue(valueName:string, data:nomap):map
Gura_DeclareMethod(RegKey, setvalue)
{
	SetMode(RSLTMODE_Normal, FLAG_Map);
	DeclareArg(env, "valueName", VTYPE_String);
	DeclareArg(env, "data", VTYPE_Any, OCCUR_Once, false, true);
}

Gura_ImplementMethod(RegKey, setvalue)
{
	Object_RegKey *pSelf = Object_RegKey::GetSelfObj(args);
	HKEY hKey = pSelf->GetKey();
	const char *lpValueName = args.GetString(0);
	DWORD dwType = 0;
	BYTE *lpData = NULL;
	DWORD cbData = 0;
	if (!ValueToRegData(env, sig, args.GetValue(1), &dwType, &lpData, &cbData)) {
		return Value::Null;
	}
	DWORD dwErrCode = ::RegSetValueEx(hKey, lpValueName, 0, dwType, lpData, cbData);
	::LocalFree(lpData);
	if (dwErrCode != ERROR_SUCCESS) {
		SetError(sig, dwErrCode);
		return Value::Null;
	}
	return Value::Null;
}

// win32.RegKey#deletevalue(valueName:string):map:void
Gura_DeclareMethod(RegKey, deletevalue)
{
	SetMode(RSLTMODE_Void, FLAG_Map);
	DeclareArg(env, "valueName", VTYPE_String);
}

Gura_ImplementMethod(RegKey, deletevalue)
{
	Object_RegKey *pSelf = Object_RegKey::GetSelfObj(args);
	HKEY hKey = pSelf->GetKey();
	const char *lpValueName = args.GetString(0);
	DWORD dwErrCode = ::RegDeleteValue(hKey, lpValueName);
	if (dwErrCode != ERROR_SUCCESS) {
		SetError(sig, dwErrCode);
		return Value::Null;
	}
	return Value::Null;
}

// win32.RegKey#queryvalue(valueName?:string):map
Gura_DeclareMethod(RegKey, queryvalue)
{
	SetMode(RSLTMODE_Normal, FLAG_Map);
	DeclareArg(env, "valueName", VTYPE_String, OCCUR_ZeroOrOnce);
}

Gura_ImplementMethod(RegKey, queryvalue)
{
	Object_RegKey *pSelf = Object_RegKey::GetSelfObj(args);
	HKEY hKey = pSelf->GetKey();
	const char *lpValueName = args.IsString(0)? args.GetString(0) : NULL;
	DWORD dwType;
	DWORD cbData;
	DWORD dwErrCode = ::RegQueryValueEx(hKey, lpValueName, NULL, &dwType, NULL, &cbData);
	if (dwErrCode != ERROR_SUCCESS) {
		SetError(sig, dwErrCode);
		return Value::Null;
	}
	LPBYTE lpData = reinterpret_cast<LPBYTE>(::LocalAlloc(LMEM_FIXED, cbData));
	dwErrCode = ::RegQueryValueEx(hKey, lpValueName, NULL, &dwType, lpData, &cbData);
	if (dwErrCode != ERROR_SUCCESS) {
		::LocalFree(lpData);
		SetError(sig, dwErrCode);
		return Value::Null;
	}
	Value result = RegDataToValue(env, sig, dwType, lpData, cbData);
	::LocalFree(lpData);
	return result;
}

// win32.RegKey#enumvalue()
Gura_DeclareMethod(RegKey, enumvalue)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(RegKey, enumvalue)
{
	Object_RegKey *pSelf = Object_RegKey::GetSelfObj(args);
	HKEY hKey = pSelf->GetKey();
	Iterator *pIterator =
			new Iterator_RegEnumValue(Object_RegKey::Reference(pSelf));
	return ReturnIterator(env, sig, args, pIterator);
}

// implementation of class RegKey
Gura_ImplementPrivClass(RegKey)
{
	Gura_AssignMethod(RegKey, createkey);
	Gura_AssignMethod(RegKey, openkey);
	Gura_AssignMethod(RegKey, deletekey);
	Gura_AssignMethod(RegKey, enumkey);
	Gura_AssignMethod(RegKey, setvalue);
	Gura_AssignMethod(RegKey, deletevalue);
	Gura_AssignMethod(RegKey, queryvalue);
	Gura_AssignMethod(RegKey, enumvalue);
}

//-----------------------------------------------------------------------------
// Iterator_RegEnumKey implementation
//-----------------------------------------------------------------------------
Iterator_RegEnumKey::Iterator_RegEnumKey(Object_RegKey *pObjRegKey, REGSAM samDesired) :
	Iterator(false), _pObjRegKey(pObjRegKey), _samDesired(samDesired), _dwIndex(0)
{
}

Iterator_RegEnumKey::~Iterator_RegEnumKey()
{
	Object::Delete(_pObjRegKey);
}

bool Iterator_RegEnumKey::DoNext(Environment &env, Signal sig, Value &value)
{
	char name[256];
	FILETIME ftLastWriteTime;
	HKEY hKey = _pObjRegKey->GetKey();
	DWORD pcName = NUMBEROF(name);
	DWORD dwErrCode = ::RegEnumKeyEx(hKey, _dwIndex, name, &pcName,
										NULL, NULL, NULL, &ftLastWriteTime);
	if (dwErrCode != ERROR_SUCCESS) {
		if (dwErrCode != ERROR_NO_MORE_ITEMS) SetError(sig, dwErrCode);
		return false;
	}
	if (_samDesired == 0) {
		value = Value(env, name);
	} else {
		HKEY hKeyResult;
		DWORD dwErrCode = ::RegOpenKeyEx(hKey, name, 0, _samDesired, &hKeyResult);
		if (dwErrCode != ERROR_SUCCESS) {
			SetError(sig, dwErrCode);
			return false;
		}
		value = Value(new Object_RegKey(hKeyResult));
	}
	_dwIndex++;
	return true;
}

String Iterator_RegEnumKey::ToString(Signal sig) const
{
	return String("<iterator:win32.regenumkey>");
}

//-----------------------------------------------------------------------------
// Iterator_RegEnumValue implementation
//-----------------------------------------------------------------------------
Iterator_RegEnumValue::Iterator_RegEnumValue(Object_RegKey *pObjRegKey) :
	Iterator(false), _pObjRegKey(pObjRegKey), _dwIndex(0)
{
}

Iterator_RegEnumValue::~Iterator_RegEnumValue()
{
	Object::Delete(_pObjRegKey);
}

bool Iterator_RegEnumValue::DoNext(Environment &env, Signal sig, Value &value)
{
	char valueName[256];
	DWORD cValueName = NUMBEROF(valueName);
	FILETIME ftLastWriteTime;
	HKEY hKey = _pObjRegKey->GetKey();

	DWORD dwType;
	DWORD cbData;
	DWORD dwErrCode = ::RegEnumValue(hKey, _dwIndex, 
						valueName, &cValueName, NULL, &dwType, NULL, &cbData);
	if (dwErrCode != ERROR_SUCCESS) {
		if (dwErrCode != ERROR_NO_MORE_ITEMS) SetError(sig, dwErrCode);
		return false;
	}
	cValueName = NUMBEROF(valueName);
	LPBYTE lpData = reinterpret_cast<LPBYTE>(::LocalAlloc(LMEM_FIXED, cbData));
	dwErrCode = ::RegEnumValue(hKey, _dwIndex, 
						valueName, &cValueName, NULL, &dwType, lpData, &cbData);
	if (dwErrCode != ERROR_SUCCESS) {
		::LocalFree(lpData);
		if (dwErrCode != ERROR_NO_MORE_ITEMS) SetError(sig, dwErrCode);
		return false;
	}
	ValueList &valList = value.InitAsList(env);
	Value valueWk(env, valueName);
	valList.push_back(valueWk);
	valList.push_back(RegDataToValue(env, sig, dwType, lpData, cbData));
	::LocalFree(lpData);
	if (sig.IsSignalled()) return false;
	_dwIndex++;
	return true;
}

String Iterator_RegEnumValue::ToString(Signal sig) const
{
	return String("<iterator:win32.regenumvalue>");
}

//-----------------------------------------------------------------------------
// Gura module functions: win32
//-----------------------------------------------------------------------------
// obj = win32.olecreate(progid:string):map[:import_const] {block?}
Gura_DeclareFunction(olecreate)
{
	SetMode(RSLTMODE_Normal, FLAG_Map);
	DeclareArg(env, "progid", VTYPE_String);
	DeclareAttr(Gura_PrivSymbol(import_const));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

Gura_ImplementFunction(olecreate)
{
	Object_Win32Ole *pObj = new Object_Win32Ole(env);
	if (!pObj->Create(sig, args.GetString(0))) {
		delete pObj;
		return Value::Null;
	}
	if (args.IsSet(Gura_PrivSymbol(import_const))) {
		pObj->ImportConstant(*pObj, sig);
		if (sig.IsSignalled()) return Value::Null;
	}
	Value result(pObj);
	return ReturnValue(env, sig, args, result);
}

// obj = win32.oleconnect(progid:string):map[:import_const] {block?}
Gura_DeclareFunction(oleconnect)
{
	SetMode(RSLTMODE_Normal, FLAG_Map);
	DeclareArg(env, "progid", VTYPE_String);
	DeclareAttr(Gura_PrivSymbol(import_const));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

Gura_ImplementFunction(oleconnect)
{
	Object_Win32Ole *pObj = new Object_Win32Ole(env);
	if (!pObj->Connect(sig, args.GetString(0))) {
		delete pObj;
		return Value::Null;
	}
	if (args.IsSet(Gura_PrivSymbol(import_const))) {
		pObj->ImportConstant(*pObj, sig);
		if (sig.IsSignalled()) return Value::Null;
	}
	Value result(pObj);
	return ReturnValue(env, sig, args, result);
}



Gura_ModuleEntry()
{
	::CoInitialize(0);
	// symbol realization
	Gura_RealizePrivSymbol(import_const);
	Gura_RealizePrivSymbol(openkey);
	Gura_RealizePrivSymbol(non_volatile);
	Gura_RealizePrivSymbolEx(volatile_, "volatile");
	Gura_RealizePrivSymbol(backup_restore);
	Gura_RealizePrivSymbol(create_link);
	Gura_RealizePrivSymbol(create_sub_key);
	Gura_RealizePrivSymbol(enumerate_sub_keys);
	Gura_RealizePrivSymbol(execute);
	Gura_RealizePrivSymbol(notify);
	Gura_RealizePrivSymbol(query_value);
	Gura_RealizePrivSymbol(set_value);
	Gura_RealizePrivSymbol(all_access);
	Gura_RealizePrivSymbol(read);
	Gura_RealizePrivSymbol(write);
	// class realization
	Gura_RealizePrivClass(RegKey, "regkey", env.LookupClass(VTYPE_Object));
	// value assignment
	Gura_AssignValue(HKEY_CLASSES_ROOT, Value(new Object_RegKey(HKEY_CLASSES_ROOT)));
	Gura_AssignValue(HKEY_CURRENT_CONFIG, Value(new Object_RegKey(HKEY_CURRENT_CONFIG)));
	Gura_AssignValue(HKEY_CURRENT_USER, Value(new Object_RegKey(HKEY_CURRENT_USER)));
	Gura_AssignValue(HKEY_LOCAL_MACHINE, Value(new Object_RegKey(HKEY_LOCAL_MACHINE)));
	Gura_AssignValue(HKEY_USERS, Value(new Object_RegKey(HKEY_USERS)));
	Gura_AssignValue(HKEY_PERFORMANCE_DATA, Value(new Object_RegKey(HKEY_PERFORMANCE_DATA)));
	Gura_AssignValue(HKEY_DYN_DATA, Value(new Object_RegKey(HKEY_DYN_DATA)));
	// function assignment
	Gura_AssignFunction(olecreate);
	Gura_AssignFunction(oleconnect);
}

Gura_ModuleTerminate()
{
	::CoUninitialize();
}

Gura_EndModule(win32, win32)

Gura_RegisterModule(win32)
