//
// Object_Expr
//

#include "gura/Object_Expr.h"
#include "gura/Expr.h"

#define ImplementTypeChecker(funcName, func) \
Gura_DeclareMethod(Expr, funcName) {} \
Gura_ImplementMethod(Expr, funcName) { \
	return Value(Object_Expr::GetSelfObj(args)->GetExpr()->func()); \
}

namespace Gura {

//-----------------------------------------------------------------------------
// Object_Expr
//-----------------------------------------------------------------------------
Object_Expr::Object_Expr(const Object_Expr &obj) :
								Object(obj), _pExpr(obj._pExpr->IncRef())
{
}

Object_Expr::~Object_Expr()
{
	Expr::Delete(_pExpr);
}

Object *Object_Expr::Clone() const
{
	return new Object_Expr(*this);
}

String Object_Expr::ToString(Signal sig, bool exprFlag)
{
	String str;
	if (_pExpr->IsValue() || _pExpr->IsSymbol() || _pExpr->IsCaller()) {
		if (exprFlag) str += '`';
		str += _pExpr->ToString();
	} else if (exprFlag) {
		if (_pExpr->IsUnary() || _pExpr->IsBinary()) {
			str += "`(";
			str += _pExpr->ToString();
			str += ")";
		} else {
			str += "`";
			str += _pExpr->ToString();
		}
	} else {
		str += _pExpr->ToString();
	}
	return str;
}

//-----------------------------------------------------------------------------
// Object_Expr::Iterator_Each
//-----------------------------------------------------------------------------
Object_Expr::Iterator_Each::~Iterator_Each()
{
	Expr::Delete(_pExprContainer);
}

bool Object_Expr::Iterator_Each::DoNext(Environment &env, Signal sig, Value &value)
{
	if (_ppExpr == _pExprContainer->GetExprList().end()) return false;
	Object_Expr *pObj = new Object_Expr(_env, (*_ppExpr)->IncRef());
	value = Value(pObj);
	_ppExpr++;
	return true;
}

String Object_Expr::Iterator_Each::ToString(Signal sig) const
{
	String rtn = "<iterator:expr:each>";
	return rtn;
}

//-----------------------------------------------------------------------------
// Gura interfaces for Object_Expr
//-----------------------------------------------------------------------------
// expr#child()
Gura_DeclareMethod(Expr, child)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Expr, child)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	if (!pExpr->IsUnary()) {
		sig.SetError(ERR_ValueError, "not a unary expression");
		return Value::Null;
	}
	Value result;
	result.InitAsExpr(env,
			dynamic_cast<const Expr_Unary *>(pExpr)->GetChild()->IncRef());
	return result;
}

// expr#left()
Gura_DeclareMethod(Expr, left)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Expr, left)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	if (!pExpr->IsBinary()) {
		sig.SetError(ERR_ValueError, "not a binary expression");
		return Value::Null;
	}
	Value result;
	result.InitAsExpr(env,
			dynamic_cast<const Expr_Binary *>(pExpr)->GetLeft()->IncRef());
	return result;
}

// expr#right()
Gura_DeclareMethod(Expr, right)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Expr, right)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	if (!pExpr->IsBinary()) {
		sig.SetError(ERR_ValueError, "not a binary expression");
		return Value::Null;
	}
	Value result;
	result.InitAsExpr(env,
			dynamic_cast<const Expr_Binary *>(pExpr)->GetRight()->IncRef());
	return result;
}

// expr#each() {block?}
Gura_DeclareMethod(Expr, each)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

Gura_ImplementMethod(Expr, each)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	if (!pExpr->IsContainer()) {
		sig.SetError(ERR_ValueError, "not a container expression");
		return Value::Null;
	}
	Iterator *pIterator = new Object_Expr::Iterator_Each(env, 
							dynamic_cast<Expr_Container *>(pExpr->IncRef()));
	return ReturnIterator(env, sig, args, pIterator);
}

// expr#car()
Gura_DeclareMethod(Expr, car)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Expr, car)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	if (!pExpr->IsCompound()) {
		sig.SetError(ERR_ValueError, "not a compound expression");
		return Value::Null;
	}
	Value result;
	result.InitAsExpr(env,
			dynamic_cast<const Expr_Compound *>(pExpr)->GetCar()->IncRef());
	return result;
}

// expr#cdr()
Gura_DeclareMethod(Expr, cdr)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Expr, cdr)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	if (!pExpr->IsCompound()) {
		sig.SetError(ERR_ValueError, "not a compound expression");
		return Value::Null;
	}
	Value result;
	ValueList &valList = result.InitAsList(env);
	foreach_const (ExprList, ppExpr,
					dynamic_cast<const Expr_Compound *>(pExpr)->GetExprList()) {
		const Expr *pExpr = *ppExpr;
		Value value;
		value.InitAsExpr(env, pExpr->IncRef());
		valList.push_back(value);
	}
	return result;
}

// expr#unquote()
Gura_DeclareMethod(Expr, unquote)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Expr, unquote)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	Value result;
	result.InitAsExpr(env, pExpr->Unquote()->IncRef());
	return result;
}

// expr#block()
Gura_DeclareMethod(Expr, block)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Expr, block)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	if (!pExpr->IsCaller()) {
		sig.SetError(ERR_ValueError, "not a caller expression");
		return Value::Null;
	}
	const Expr_Block *pExprBlock =
						dynamic_cast<const Expr_Caller *>(pExpr)->GetBlock();
	if (pExprBlock == NULL) return Value::Null;
	Value result;
	result.InitAsExpr(env, pExprBlock->IncRef());
	return result;
}

// expr#exprname()
Gura_DeclareMethod(Expr, exprname)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Expr, exprname)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	return Value(env, pExpr->GetTypeName());
}

// expr#getvalue()
Gura_DeclareMethod(Expr, getvalue)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Expr, getvalue)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	if (!pExpr->IsValue()) {
		sig.SetError(ERR_ValueError, "expression is not a value");
		return Value::Null;
	}
	return dynamic_cast<const Expr_Value *>(pExpr)->GetValue();
}

// expr#getstring()
Gura_DeclareMethod(Expr, getstring)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Expr, getstring)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	if (!pExpr->IsString()) {
		sig.SetError(ERR_ValueError, "expression is not a string");
		return Value::Null;
	}
	return Value(env, dynamic_cast<const Expr_String *>(pExpr)->GetString());
}

// expr#getsymbol()
Gura_DeclareMethod(Expr, getsymbol)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Expr, getsymbol)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	if (!pExpr->IsSymbol()) {
		sig.SetError(ERR_ValueError, "expression is not a symbol");
		return Value::Null;
	}
	return Value(dynamic_cast<const Expr_Symbol *>(pExpr)->GetSymbol());
}

// expr#tofunction(`args*)
Gura_DeclareMethod(Expr, tofunction)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
	DeclareArg(env, "args", VTYPE_Quote, OCCUR_ZeroOrMore);
}

Gura_ImplementMethod(Expr, tofunction)
{
	const Expr *pExpr = Object_Expr::GetSelfObj(args)->GetExpr();
	Function *pFunc = pExpr->ToFunction(env, sig,
								args.GetList(0), args.GetAttrs());
	return Value(env, pFunc, Value::Null);
}

// type chekers
ImplementTypeChecker(isunary,		IsUnary)
ImplementTypeChecker(isunaryop,		IsUnaryOp)
ImplementTypeChecker(isquote,		IsQuote)
ImplementTypeChecker(isforce,		IsForce)
ImplementTypeChecker(isprefix,		IsPrefix)
ImplementTypeChecker(issuffix,		IsSuffix)

ImplementTypeChecker(isbinary,		IsBinary)
ImplementTypeChecker(isbinaryop,	IsBinaryOp)
ImplementTypeChecker(isassign,		IsAssign)
ImplementTypeChecker(isdictassign,	IsDictAssign)
ImplementTypeChecker(isfield,		IsField)

ImplementTypeChecker(iscontainer,	IsContainer)
ImplementTypeChecker(isblockparam,	IsBlockParam)
ImplementTypeChecker(isblock,		IsBlock)
ImplementTypeChecker(islister,		IsLister)

ImplementTypeChecker(isvalue,		IsValue)
ImplementTypeChecker(isstring,		IsString)
ImplementTypeChecker(issymbol,		IsSymbol)
ImplementTypeChecker(isindexer,		IsIndexer)
ImplementTypeChecker(iscaller,		IsCaller)

// Assignment
Class_Expr::Class_Expr(Environment *pEnvOuter) : Class(pEnvOuter)
{
	Gura_AssignMethod(Expr, child);
	Gura_AssignMethod(Expr, left);
	Gura_AssignMethod(Expr, right);
	Gura_AssignMethod(Expr, each);
	Gura_AssignMethod(Expr, car);
	Gura_AssignMethod(Expr, cdr);
	Gura_AssignMethod(Expr, unquote);
	Gura_AssignMethod(Expr, block);
	Gura_AssignMethod(Expr, exprname);
	Gura_AssignMethod(Expr, getvalue);
	Gura_AssignMethod(Expr, getstring);
	Gura_AssignMethod(Expr, getsymbol);
	Gura_AssignMethod(Expr, tofunction);
	Gura_AssignMethod(Expr, isunary);
	Gura_AssignMethod(Expr, isunaryop);
	Gura_AssignMethod(Expr, isquote);
	Gura_AssignMethod(Expr, isforce);
	Gura_AssignMethod(Expr, isprefix);
	Gura_AssignMethod(Expr, issuffix);
	Gura_AssignMethod(Expr, isbinary);
	Gura_AssignMethod(Expr, isbinaryop);
	Gura_AssignMethod(Expr, isassign);
	Gura_AssignMethod(Expr, isdictassign);
	Gura_AssignMethod(Expr, isfield);
	Gura_AssignMethod(Expr, iscontainer);
	Gura_AssignMethod(Expr, isblockparam);
	Gura_AssignMethod(Expr, isblock);
	Gura_AssignMethod(Expr, islister);
	Gura_AssignMethod(Expr, isvalue);
	Gura_AssignMethod(Expr, isstring);
	Gura_AssignMethod(Expr, issymbol);
	Gura_AssignMethod(Expr, isindexer);
	Gura_AssignMethod(Expr, iscaller);
}

bool Class_Expr::CastFrom(Environment &env, Signal sig, Value &value)
{
	if (value.IsSymbol()) {		// cast Symbol to Expr
		const Symbol *pSymbol = value.GetSymbol();
		value.InitAsExpr(env, new Expr_Symbol(pSymbol));
		return true;
	}
	return false;
}

Object *Class_Expr::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	ERROREND(env, "this function must not be called");
	return NULL;
}

}
