//
// Object_Stream
//

#include "gura/Directory.h"
#include "gura/Object_Stream.h"
#include "gura/Object_Binary.h"

namespace Gura {

Gura_DeclarePrivSymbol(set);
Gura_DeclarePrivSymbol(cur);

//-----------------------------------------------------------------------------
// Object_Stream
//-----------------------------------------------------------------------------
Object_Stream::~Object_Stream()
{
	Stream::Delete(_pStream);
}

Value Object_Stream::DoPropGet(Signal sig, const Symbol *pSymbol, bool &evaluatedFlag)
{
	Environment &env = *this;
	evaluatedFlag = true;
	if (pSymbol->IsIdentical(Gura_Symbol(stat))) {
		Object *pObj = GetStream().GetStatObj(sig);
		if (pObj != NULL) return Value(pObj);
	} else if (pSymbol->IsIdentical(Gura_Symbol(name))) {
		return Value(env, GetStream().GetName());
	} else if (pSymbol->IsIdentical(Gura_Symbol(readable))) {
		return Value(GetStream().IsReadable());
	} else if (pSymbol->IsIdentical(Gura_Symbol(writable))) {
		return Value(GetStream().IsWritable());
	}
	evaluatedFlag = false;
	return Value::Null;
}

Iterator *Object_Stream::CreateIterator(Signal sig)
{
	return new IteratorLine(Object_Stream::Reference(this), -1, true);
}

String Object_Stream::ToString(Signal sig, bool exprFlag)
{
	String str;
	Stream &stream = GetStream();
	str += "<stream:";
	if (stream.IsReadable()) str += "R";
	if (stream.IsWritable()) str += "W";
	if (stream.IsCodecInstalled()) {
		str += ":";
		str += stream.GetEncoding();
		Codec_Encoder *pEncoder = stream.GetEncoder();
		if (pEncoder != NULL && pEncoder->IsProcessEOL()) {
			str += ":dosmode";
		}
	}
	if (*stream.GetName() != '\0') {
		str += ":";
		str += stream.GetName();
	}
	str += ">";
	return str;
}

//-----------------------------------------------------------------------------
// Gura interfaces for Object_Stream
//-----------------------------------------------------------------------------
// stream#close()
Gura_DeclareMethod(Stream, close)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Stream, close)
{
	Stream &stream = Object_Stream::GetSelfObj(args)->GetStream();
	stream.Close();
	return Value::Null;
}

// stream#read(len?:number)
Gura_DeclareMethod(Stream, read)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
	DeclareArg(env, "len", VTYPE_Number, OCCUR_ZeroOrOnce);
}

Gura_ImplementMethod(Stream, read)
{
	Stream &stream = Object_Stream::GetSelfObj(args)->GetStream();
	if (!stream.CheckReadable(sig)) return Value::Null;
	Value result;
	if (args.IsNumber(0)) {
		size_t len = static_cast<size_t>(args.GetNumber(0));
		char *buff = new char [len];
		size_t lenRead = stream.Read(sig, buff, len);
		if (lenRead == 0) {
			delete [] buff;
			return Value::Null;
		}
		result.InitAsBinary(env, Binary(buff, lenRead));
		delete [] buff;
	} else if (stream.IsInfinite()) {
		sig.SetError(ERR_IOError, "specify a reading size for infinite stream");
		return Value::Null;
	} else {
		const int bytesBuff = 4096;
		Object_Binary *pObjBinary = result.InitAsBinary(env);
		Binary &buff = pObjBinary->GetBinary();
		OAL::Memory memory(bytesBuff);
		char *buffSeg = reinterpret_cast<char *>(memory.GetPointer());
		size_t lenRead = stream.Read(sig, buffSeg, bytesBuff);
		if (lenRead == 0) {
			return Value::Null;
		}
		do {
			buff.append(buffSeg, lenRead);
		} while ((lenRead = stream.Read(sig, buffSeg, bytesBuff)) > 0);
		if (sig.IsSignalled()) return Value::Null;
	}
	return result;
}

// stream#peek(len:number)
Gura_DeclareMethod(Stream, peek)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
	DeclareArg(env, "len", VTYPE_Number, OCCUR_ZeroOrOnce);
}

Gura_ImplementMethod(Stream, peek)
{
	Stream &stream = Object_Stream::GetSelfObj(args)->GetStream();
	if (!stream.CheckReadable(sig)) return Value::Null;
	Value result;
	size_t len = static_cast<size_t>(args.GetNumber(0));
	char *buff = new char [len];
	size_t lenRead = stream.Peek(sig, buff, len);
	if (lenRead == 0) {
		delete [] buff;
		return Value::Null;
	}
	result.InitAsBinary(env, Binary(buff, lenRead));
	delete [] buff;
	return result;
}

// stream#write(buff:binary):reduce
Gura_DeclareMethod(Stream, write)
{
	SetMode(RSLTMODE_Reduce, FLAG_None);
	DeclareArg(env, "buff", VTYPE_Binary);
}

Gura_ImplementMethod(Stream, write)
{
	Stream &stream = Object_Stream::GetSelfObj(args)->GetStream();
	if (!stream.CheckWritable(sig)) return Value::Null;
	const Binary &binary = args.GetBinary(0);
	stream.Write(sig, binary.c_str(), binary.size());
	return args.GetSelf();
}

// stream#seek(offset:number, origin?:symbol):reduce
Gura_DeclareMethod(Stream, seek)
{
	SetMode(RSLTMODE_Reduce, FLAG_None);
	DeclareArg(env, "offset", VTYPE_Number);
	DeclareArg(env, "origin", VTYPE_Symbol, OCCUR_ZeroOrOnce);
}

Gura_ImplementMethod(Stream, seek)
{
	Stream &stream = Object_Stream::GetSelfObj(args)->GetStream();
	Stream::SeekMode seekMode = Stream::SeekSet;
	if (args.GetValue(1).IsSymbol()) {
		const Symbol *pSymbol = args.GetSymbol(1);
		if (pSymbol->IsIdentical(Gura_PrivSymbol(set))) {
			seekMode = Stream::SeekSet;
		} else if (pSymbol->IsIdentical(Gura_PrivSymbol(cur))) {
			seekMode = Stream::SeekCur;
		} else {
			sig.SetError(ERR_ValueError, "invalid seek mode %s", pSymbol->GetName());
			return Value::Null;
		}
	}
	stream.Seek(sig, args.GetLong(0), seekMode);
	return args.GetSelf();
}

// stream#tell()
Gura_DeclareMethod(Stream, tell)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Stream, tell)
{
	Stream &stream = Object_Stream::GetSelfObj(args)->GetStream();
	return Value(static_cast<unsigned long>(stream.Tell()));
}

// stream#compare(stream:stream):map
Gura_DeclareMethod(Stream, compare)
{
	SetMode(RSLTMODE_Normal, FLAG_Map);
	DeclareArg(env, "stream", VTYPE_Stream);
}

Gura_ImplementMethod(Stream, compare)
{
	Stream &stream1 = Object_Stream::GetSelfObj(args)->GetStream();
	Stream &stream2 = args.GetStream(0);
	bool sameFlag = stream1.Compare(sig, stream2);
	if (sig.IsSignalled()) return Value::Null;
	return Value(sameFlag);
}

// stream#readto(stream:stream, bytesunit:number => 65536):map:reduce {block?}
Gura_DeclareMethod(Stream, readto)
{
	SetMode(RSLTMODE_Reduce, FLAG_Map);
	DeclareArg(env, "stream", VTYPE_Stream);
	DeclareArg(env, "bytesunit", VTYPE_Number,
					OCCUR_Once, false, false, new Expr_Value(65536));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

Gura_ImplementMethod(Stream, readto)
{
	Stream &streamSrc = Object_Stream::GetSelfObj(args)->GetStream();
	Stream &streamDst = args.GetStream(0);
	size_t bytesUnit = args.GetSizeT(1);
	const Function *pFuncWatcher =
					args.GetBlockFunc(env, sig, GetSymbolForBlock());
	if (bytesUnit == 0 || bytesUnit > 1024 * 1024) {
		sig.SetError(ERR_ValueError, "wrong value for bytesunit");
		return Value::Null;
	}
	if (!streamSrc.ReadToStream(env, sig, streamDst, bytesUnit, pFuncWatcher)) {
		return Value::Null;
	}
	return args.GetSelf();
}

// stream#writefrom(stream:stream, bytesunit:number => 65536):map:reduce {block?}
Gura_DeclareMethod(Stream, writefrom)
{
	SetMode(RSLTMODE_Reduce, FLAG_Map);
	DeclareArg(env, "stream", VTYPE_Stream);
	DeclareArg(env, "bytesunit", VTYPE_Number,
					OCCUR_Once, false, false, new Expr_Value(65536));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

Gura_ImplementMethod(Stream, writefrom)
{
	Stream &streamDst = Object_Stream::GetSelfObj(args)->GetStream();
	Stream &streamSrc = args.GetStream(0);
	size_t bytesUnit = args.GetSizeT(1);
	const Function *pFuncWatcher =
					args.GetBlockFunc(env, sig, GetSymbolForBlock());
	if (bytesUnit == 0 || bytesUnit > 1024 * 1024) {
		sig.SetError(ERR_ValueError, "wrong value for bytesunit");
		return Value::Null;
	}
	if (!streamSrc.ReadToStream(env, sig, streamDst, bytesUnit, pFuncWatcher)) {
		return Value::Null;
	}
	return args.GetSelf();
}

// stream#setencoding(encoding:string, dos_flag?:boolean)
Gura_DeclareMethod(Stream, setencoding)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
	DeclareArg(env, "encoding", VTYPE_String);
	DeclareArg(env, "dos_flag", VTYPE_Boolean, OCCUR_ZeroOrOnce);
}

Gura_ImplementMethod(Stream, setencoding)
{
	Object_Stream *pSelf = Object_Stream::GetSelfObj(args);
	const char *encoding = args.GetString(0);
	bool processEOLFlag = args.IsBoolean(1)? args.GetBoolean(1) : true;
	if (!pSelf->GetStream().InstallCodec(encoding, processEOLFlag)) {
		sig.SetError(ERR_CodecError, "unsupported encoding '%s'", encoding);
		return Value::Null;
	}
	return Value::Null;
}

// stream#dosmode(dos_flag:boolean):reduce
Gura_DeclareMethod(Stream, dosmode)
{
	SetMode(RSLTMODE_Reduce, FLAG_None);
	DeclareArg(env, "dos_flag", VTYPE_Boolean);
}

Gura_ImplementMethod(Stream, dosmode)
{
	Object_Stream *pSelf = Object_Stream::GetSelfObj(args);
	Codec_Encoder *pEncoder = pSelf->GetStream().GetEncoder();
	if (pEncoder != NULL) {
		pEncoder->SetProcessEOLFlag(args.GetBoolean(0));
	}
	return args.GetSelf();
}

// stream#readline():[chop]
Gura_DeclareMethod(Stream, readline)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
	DeclareAttr(Gura_Symbol(chop));
}

Gura_ImplementMethod(Stream, readline)
{
	Stream &stream = Object_Stream::GetSelfObj(args)->GetStream();
	if (!stream.CheckReadable(sig)) return Value::Null;
	int cntChars = 4096;	// tentative
	bool includeEOLFlag = !args.IsSet(Gura_Symbol(chop));
	String str;
	while (cntChars-- > 0) {
		int ch = stream.GetChar(sig);
		if (sig.IsSignalled()) return Value::Null;
		if (ch < 0) break;
		if (ch == '\n') {
			if (includeEOLFlag) str += '\n';
			return Value(env, str.c_str());
		}
		str += ch;
	}
	if (str.empty()) return Value::Null;
	return Value(env, str.c_str());
}

// stream#readlines(nlines?:number):[chop] {block?}
// conrresponding to string#splitlines()
Gura_DeclareMethod(Stream, readlines)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
	DeclareAttr(Gura_Symbol(chop));
	DeclareArg(env, "nlines", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

Gura_ImplementMethod(Stream, readlines)
{
	Object_Stream *pSelf = Object_Stream::GetSelfObj(args);
	Stream &stream = pSelf->GetStream();
	if (!stream.CheckReadable(sig)) return Value::Null;
	int nLinesMax = args.IsNumber(0)? static_cast<int>(args.GetNumber(0)) : -1;
	bool includeEOLFlag = !args.IsSet(Gura_Symbol(chop));
	Iterator *pIterator = new Object_Stream::IteratorLine(
				Object_Stream::Reference(pSelf), nLinesMax, includeEOLFlag);
	return ReturnIterator(env, sig, args, pIterator);
}

// stream#readtext()
Gura_DeclareMethod(Stream, readtext)
{
	SetMode(RSLTMODE_Normal, FLAG_None);
}

Gura_ImplementMethod(Stream, readtext)
{
	Stream &stream = Object_Stream::GetSelfObj(args)->GetStream();
	if (!stream.CheckReadable(sig)) return Value::Null;
	String str;
	for (;;) {
		int ch = stream.GetChar(sig);
		if (sig.IsSignalled()) return Value::Null;
		if (ch < 0) break;
		str += ch;
	}
	return Value(env, str.c_str());
}

// stream#print(values*):map:void
Gura_DeclareMethod(Stream, print)
{
	SetMode(RSLTMODE_Void, FLAG_Map);
	DeclareArg(env, "values", VTYPE_Any, OCCUR_ZeroOrMore);
}

Gura_ImplementMethod(Stream, print)
{
	Stream &stream = Object_Stream::GetSelfObj(args)->GetStream();
	if (!stream.CheckWritable(sig)) return Value::Null;
	foreach_const (ValueList, pValue, args.GetList(0)) {
		String str(pValue->ToString(sig, false));
		if (sig.IsSignalled()) break;
		stream.Print(sig, str.c_str());
		if (sig.IsSignalled()) break;
	}
	return Value::Null;
}

// stream#println(values*):map:void
Gura_DeclareMethod(Stream, println)
{
	SetMode(RSLTMODE_Void, FLAG_Map);
	DeclareArg(env, "values", VTYPE_Any, OCCUR_ZeroOrMore);
}

Gura_ImplementMethod(Stream, println)
{
	Stream &stream = Object_Stream::GetSelfObj(args)->GetStream();
	if (!stream.CheckWritable(sig)) return Value::Null;
	foreach_const (ValueList, pValue, args.GetList(0)) {
		String str(pValue->ToString(sig, false));
		if (sig.IsSignalled()) break;
		stream.Print(sig, str.c_str());
		if (sig.IsSignalled()) break;
	}
	stream.Print(sig, "\n");
	return Value::Null;
}

// stream#printf(format, values*):map:void
Gura_DeclareMethod(Stream, printf)
{
	SetMode(RSLTMODE_Void, FLAG_Map);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "values", VTYPE_Any, OCCUR_ZeroOrMore);
}

Gura_ImplementMethod(Stream, printf)
{
	Stream &stream = Object_Stream::GetSelfObj(args)->GetStream();
	if (!stream.CheckWritable(sig)) return Value::Null;
	stream.Printf(sig, args.GetString(0), args.GetList(1));
	return Value::Null;
}

// Assignment
Class_Stream::Class_Stream(Environment *pEnvOuter) : Class(pEnvOuter)
{
	Gura_RealizePrivSymbol(set);
	Gura_RealizePrivSymbol(cur);
	Gura_AssignMethod(Stream, close);
	Gura_AssignMethod(Stream, read);
	Gura_AssignMethod(Stream, peek);
	Gura_AssignMethod(Stream, write);
	Gura_AssignMethod(Stream, seek);
	Gura_AssignMethod(Stream, tell);
	Gura_AssignMethod(Stream, compare);
	Gura_AssignMethod(Stream, readto);
	Gura_AssignMethod(Stream, writefrom);
	Gura_AssignMethod(Stream, setencoding);
	Gura_AssignMethod(Stream, dosmode);
	Gura_AssignMethod(Stream, readline);
	Gura_AssignMethod(Stream, readlines);
	Gura_AssignMethod(Stream, readtext);
	Gura_AssignMethod(Stream, print);
	Gura_AssignMethod(Stream, println);
	Gura_AssignMethod(Stream, printf);
}

bool Class_Stream::CastFrom(Environment &env, Signal sig, Value &value)
{
	if (value.IsString()) {
		Stream *pStream = Directory::OpenStream(env, sig,
						value.GetString(), Stream::ATTR_Readable, "utf-8");
		if (sig.IsSignalled()) return false;
		value = Value(new Object_Stream(env, pStream));
		return true;
	} else if (value.IsBinary()) {
		Object_Binary *pObjBinary = Object_Binary::Reference(
							dynamic_cast<Object_Binary *>(value.GetObject()));
		Object *pObj = new Object_Stream(env, new BinaryStream(sig, pObjBinary));
		value = Value(pObj);
		return true;
	}
	return false;
}

Object *Class_Stream::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	return NULL;
}

//-----------------------------------------------------------------------------
// Object_Stream::IteratorLine
//-----------------------------------------------------------------------------
Object_Stream::IteratorLine::~IteratorLine()
{
	Object::Delete(_pObj);
}

bool Object_Stream::IteratorLine::DoNext(Environment &env, Signal sig, Value &value)
{
	Stream &stream = _pObj->GetStream();
	String str;
	if (_nLines == _nLinesMax) return false;
	int ch = stream.GetChar(sig);
	if (ch < 0) return false;
	for ( ; ch >= 0; ch = stream.GetChar(sig)) {
		if (ch == '\n') {
			if (_includeEOLFlag) str += '\n';
			break;
		}
		str += ch;
	}
	if (sig.IsSignalled()) return false;
	_nLines++;
	value = Value(*_pObj, str.c_str());
	return true;
}

String Object_Stream::IteratorLine::ToString(Signal sig) const
{
	return String("<iterator:stream#readlines>");
}

}
