#include "GuraShell.h"
#include "Config.h"

Gura_IncludeModule(scribble)

//-----------------------------------------------------------------------------
// GuraShell
//-----------------------------------------------------------------------------
GuraShell::GuraShell(wxWindow *pParent, wxWindowID id, long style) :
		wxTextCtrl(pParent, id, wxT(""), wxDefaultPosition, wxDefaultSize,
								style | wxWANTS_CHARS | wxTE_MULTILINE),
		_console(_sig, this), _env(0, NULL),
		_threadRunningFlag(false), _scrollViewFlag(false)
{
	do { // import(scribble)
		Gura::Gura_Module(scribble)::Import(_env, _sig);
		if (_sig.IsSignalled()) return;
	} while (0);
	_env.SetEchoFlag(true);
	_env.SetConsole(true, &_console);
	_env.SetConsole(false, &_console);
	_console.Println(_sig, Gura::GetOpening());
	_console.Print(_sig, _env.GetPrompt(_parser.IsContinued()));
	_pLineHistory = _lineHistory.rend();
	MarkLineTop();
	UpdateFont();
	UpdateColor();
}

GuraShell::~GuraShell()
{
}

void GuraShell::UpdateFont()
{
	wxFont font(wxSystemSettings::GetFont(wxSYS_OEM_FIXED_FONT));
	font.SetPointSize(Config::GuraShell::ReadFontSize());
	SetFont(font);
}

void GuraShell::UpdateColor()
{
	SetForegroundColour(wxColour(Config::GuraShell::ReadFontColor()));
	SetBackgroundColour(wxColour(Config::GuraShell::ReadBackgroundColor()));
}

void GuraShell::EvalFile(const wxString &fileName)
{
	RunThread(new EvalFileThread(this, fileName));
}

BEGIN_EVENT_TABLE(GuraShell, wxTextCtrl)
	EVT_KEY_DOWN(GuraShell::OnKeyDown)
END_EVENT_TABLE()

void GuraShell::OnKeyDown(wxKeyEvent &event)
{
	int keyCode = event.GetKeyCode();
	if (keyCode == 'D' && event.ControlDown()) {
		if (IsThreadRunning()) {
			_sig.SetSignal(Gura::SIGTYPE_Terminate, Gura::Value::Null);
		}
		return;
	}
	if (event.AltDown()) {
		if (keyCode == WXK_RETURN) {
			if (IsScrollView()) {
				LeaveScrollView();
			} else {
				*this << "\n";
			}
		} else {
			event.Skip();
		}
	} else if (IsThreadRunning()) {
		// nothing to do
	} else if (keyCode == WXK_RETURN) {
		if (IsScrollView()) {
			LeaveScrollView();
		} else {
			wxString str = GetRange(_posLineTop, GetLastPosition());
			*this << "\n";
			if (!str.empty() && (_lineHistory.empty() || _lineHistory.back() != str)) {
				_lineHistory.push_back(str);
			}
			str << "\n";
			RunThread(new EvalLineThread(this, str));
		}
	} else if (keyCode == WXK_ESCAPE) {
		if (IsScrollView()) {
			LeaveScrollView();
		} else {
			Replace(GetLineTopPoint(), GetLastPosition(), wxT(""));
		}
	} else if (keyCode == WXK_HOME) {
		if (IsScrollView()) {
			event.Skip();
		} else {
			SetInsertionPoint(GetLineTopPoint());
		}
	} else if (keyCode == WXK_END) {
		if (IsScrollView()) {
			event.Skip();
		} else {
			SetInsertionPoint(GetLastPosition());
		}
	} else if (keyCode == WXK_BACK) {
		if (IsScrollView()) {
			event.Skip();
		} else {
			if (GetInsertionPoint() > GetLineTopPoint()) event.Skip();
		}
	} else if (keyCode == WXK_LEFT) {
		if (IsScrollView()) {
			event.Skip();
		} else {
			if (GetInsertionPoint() > GetLineTopPoint()) event.Skip();
		}
	} else if (keyCode == WXK_RIGHT) {
		event.Skip();
	} else if (keyCode == WXK_UP) {
		if (IsScrollView()) {
			event.Skip();
		} else if (event.ControlDown()) {
			EnterScrollView();
		} else if (_lineHistory.empty()) {
			// nothing to do
		} else if (_pLineHistory == _lineHistory.rend()) {
			_pLineHistory = _lineHistory.rbegin();
			Replace(GetLineTopPoint(), GetLastPosition(), *_pLineHistory);
		} else if (_pLineHistory + 1 != _lineHistory.rend()) {
			_pLineHistory++;
			Replace(GetLineTopPoint(), GetLastPosition(), *_pLineHistory);
		}
	} else if (keyCode == WXK_DOWN) {
		if (IsScrollView()) {
			event.Skip();
		} else if (_pLineHistory == _lineHistory.rend()) {
			// nothing to do
		} else if (_pLineHistory == _lineHistory.rbegin()) {
			_pLineHistory = _lineHistory.rend();
			Replace(GetLineTopPoint(), GetLastPosition(), wxT(""));
		} else {
			_pLineHistory--;
			Replace(GetLineTopPoint(), GetLastPosition(), *_pLineHistory);
		}
	} else {
		event.Skip();
	}
}

void GuraShell::EnterScrollView()
{
	SetForegroundColour(wxColour(128, 128, 128));
	Refresh();
	Update();
	_scrollViewFlag = true;
}

void GuraShell::LeaveScrollView()
{
	SetForegroundColour(*wxBLACK);
	Refresh();
	Update();
	_scrollViewFlag = false;
	SetInsertionPoint(GetLastPosition());
}

void GuraShell::RunThread(wxThread *pThread)
{
	_threadRunningFlag = true;
	pThread->Create();
	pThread->Run();
}

void GuraShell::NotifyEndOfThread()
{
	_pLineHistory = _lineHistory.rend();
	_console.Print(_sig, _env.GetPrompt(_parser.IsContinued()));
	ShowPosition(GetLastPosition());
	MarkLineTop();
	_threadRunningFlag = false;
}

//-----------------------------------------------------------------------------
// GuraShell::StreamEx
//-----------------------------------------------------------------------------
GuraShell::StreamEx::StreamEx(Gura::Signal sig, GuraShell *pGuraShell) :
	Gura::Stream(sig, Gura::Stream::ATTR_Writable), _pGuraShell(pGuraShell)
{
	InstallCodec("cp932", true);
}

const char *GuraShell::StreamEx::GetName() const
{
	return "<ascript-shell>";
}

void GuraShell::StreamEx::DoPutChar(Gura::Signal sig, char ch)
{
	_str << ch;
}

void GuraShell::StreamEx::FlushConsole()
{
	*_pGuraShell << _str;
	_str.clear();
}

//-----------------------------------------------------------------------------
// GuraShell::EvalLineThread
//-----------------------------------------------------------------------------
GuraShell::EvalLineThread::EvalLineThread(GuraShell *pGuraShell, const wxString &str) :
	wxThread(wxTHREAD_DETACHED), _pGuraShell(pGuraShell), _str(str)
{
}

wxThread::ExitCode GuraShell::EvalLineThread::Entry()
{
	Gura::Environment &env = _pGuraShell->GetEnvironment();
	Gura::Signal &sig = _pGuraShell->GetSignal();
	Gura::Parser &parser = _pGuraShell->GetParser();
	foreach_const (wxString, p, _str) {
		parser.EvalConsoleChar(env, sig, *p);
		if (sig.IsSignalled()) break;
	}
	_pGuraShell->NotifyEndOfThread();
	return 0;
}

//-----------------------------------------------------------------------------
// GuraShell::EvalFileThread
//-----------------------------------------------------------------------------
GuraShell::EvalFileThread::EvalFileThread(GuraShell *pGuraShell, const wxString &fileName) :
	wxThread(wxTHREAD_DETACHED), _pGuraShell(pGuraShell), _fileName(fileName)
{
}

wxThread::ExitCode GuraShell::EvalFileThread::Entry()
{
	Gura::Environment &env = _pGuraShell->GetEnvironment();
	Gura::Signal &sig = _pGuraShell->GetSignal();
	Gura::Stream &console = *env.GetConsole(false);
	console.Println(sig, "");
	Gura::Expr *pExpr = Gura::Parser().ParseStream(env, sig, _fileName.c_str());
	if (sig.IsSignalled()) {
		console.PrintSignal(sig, sig);
		sig.ClearSignal();
	} else {
		Gura::String dirOrg = Gura::OAL::GetCurDir();
		Gura::OAL::ChangeCurDir(Gura::Directory::ExtractDirName(_fileName.c_str()).c_str());
		Gura::Environment envLocal(&env, Gura::ENVTYPE_Local);
		envLocal.SetEchoFlag(false);
		pExpr->Exec(envLocal, sig);
		Gura::OAL::ChangeCurDir(dirOrg.c_str());
		envLocal.SetEchoFlag(true);
		if (sig.IsSignalled()) {
			console.PrintSignal(sig, sig);
			sig.ClearSignal();
		}
	}
	_pGuraShell->NotifyEndOfThread();
	return 0;
}
