/////////////////////////////////////////////////////////////////////////////
// Name:        wx/fswatcher.h
// Purpose:     wxFileSystemWatcher
// Author:      Bartosz Bekier
// Created:     2009-05-23
// RCS-ID:      $Id$
// Copyright:   (c) 2009 Bartosz Bekier <bartosz.bekier@gmail.com>
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

/**
    @class wxFileSystemWatcher

    The wxFileSystemWatcher class allows to receive notifications of file
    system changes.

    @note Implementation limitations: this class is currently implemented for
          MSW, OS X and GTK ports but doesn't detect all changes correctly
          everywhere: under MSW accessing the file is not detected (only
          modifying it is) and under OS X neither accessing nor modifying is
          detected (only creating and deleting files is). Moreover, OS X
          version doesn't currently collapse pairs of create/delete events in a
          rename event, unlike the other ones.

    For the full list of change types that are reported see wxFSWFlags.

    This class notifies the application about the file system changes by
    sending events of wxFileSystemWatcherEvent class. By default these events
    are sent to the wxFileSystemWatcher object itself so you can derive from it
    and use the event table @c EVT_FSWATCHER macro to handle these events in a
    derived class method. Alternatively, you can use
    wxFileSystemWatcher::SetOwner() to send the events to another object. Or
    you could use wxEvtHandler::Connect() with @c wxEVT_FSWATCHER to handle
    these events in any other object. See the fswatcher sample for an example
    of the latter approach.

    @library{wxbase}
    @category{file}

    @since 2.9.1
 */
class wxFileSystemWatcher: public wxEvtHandler
{
public:
    /**
        Default constructor.
     */
    wxFileSystemWatcher();

    /**
        Destructor. Stops all paths from being watched and frees any system
        resources used by this file system watcher object.
     */
    virtual ~wxFileSystemWatcher();

    /**
        Adds @a path to currently watched files.

        The @a path argument can currently only be a directory and any changes
        to this directory itself or its immediate children will generate the
        events. Use AddTree() to monitor the directory recursively.

        @param path
            The name of the path to watch.
        @param events
            An optional filter to receive only events of particular types.
     */
    virtual bool Add(const wxFileName& path, int events = wxFSW_EVENT_ALL);

    /**
        This is the same as Add(), but recursively adds every file/directory in
        the tree rooted at @a path.

        Additionally a file mask can be specified to include only files
        matching that particular mask.

        This method is implemented efficiently under MSW but shouldn't be used
        for the directories with a lot of children (such as e.g. the root
        directory) under the other platforms as it calls Add() there for each
        subdirectory potentially creating a lot of watches and taking a long
        time to execute.
     */
    virtual bool AddTree(const wxFileName& path, int events = wxFSW_EVENT_ALL,
                         const wxString& filter = wxEmptyString);

    /**
        Removes @a path from the list of watched paths.
     */
    virtual bool Remove(const wxFileName& path);

    /**
        Same as Remove(), but also removes every file/directory belonging to
        the tree rooted at @a path.
     */
    virtual bool RemoveTree(const wxFileName& path);

    /**
        Clears the list of currently watched paths.
     */
    virtual bool RemoveAll();

    /**
        Returns the number of currently watched paths.

        @see GetWatchedPaths()
     */
    int GetWatchedPathsCount() const;

    /**
        Retrieves all watched paths and places them in @a paths. Returns
        the number of watched paths, which is also the number of entries added
        to @a paths.
     */
    int GetWatchedPaths(wxArrayString* paths) const;

    /**
        Associates the file system watcher with the given @a handler object.

        All the events generated by this object will be passed to the specified
        owner.
     */
    void SetOwner(wxEvtHandler* handler);
};



/**
    @class wxFileSystemWatcherEvent

    A class of events sent when a file system event occurs. Types of events
    reported may vary depending on a platform, however all platforms report
    at least creation of new file/directory and access, modification, move
    (rename) or deletion of an existing one.

    @library{wxcore}
    @category{events}

    @see wxFileSystemWatcher
    @see @ref overview_events

    @since 2.9.1
*/
class wxFileSystemWatcherEvent : public wxEvent
{
public:
    /**
        Returns the path at which the event occurred.
     */
    const wxFileName& GetPath() const;

    /**
        Returns the new path of the renamed file/directory if this is a rename
        event.

        Otherwise it returns the same path as GetPath().
     */
    const wxFileName& GetNewPath() const;

    /**
        Returns the type of file system change that occurred. See wxFSWFlags for
        the list of possible file system change types.
     */
    int GetChangeType() const;

    /**
        Returns @c true if this error is an error event

        Error event is an event generated when a warning or error condition
        arises.
     */
    bool IsError() const;

    /**
        Return a description of the warning or error if this is an error event.
     */
    wxString GetErrorDescription() const;

    /**
        Returns a wxString describing an event, useful for logging, debugging
        or testing.
     */
    wxString ToString() const;
};


/**
    These are the possible types of file system change events.
    All of these events are reported on all supported platforms.

    @since 2.9.1
 */
enum wxFSWFlags
{
    wxFSW_EVENT_CREATE = 0x01,  ///< File or directory was created
    wxFSW_EVENT_DELETE = 0x02,  ///< File or directory was deleted
    wxFSW_EVENT_RENAME = 0x04,  ///< File or directory was renamed
    wxFSW_EVENT_MODIFY = 0x08,  ///< File or directory was modified
    wxFSW_EVENT_ACCESS = 0x10,  ///< File or directory was accessed

    wxFSW_EVENT_WARNING = 0x20, ///< A warning condition arose.
    wxFSW_EVENT_ERROR = 0x40,   ///< An error condition arose.

    wxFSW_EVENT_ALL = wxFSW_EVENT_CREATE | wxFSW_EVENT_DELETE |
                         wxFSW_EVENT_RENAME | wxFSW_EVENT_MODIFY |
                         wxFSW_EVENT_ACCESS |
                         wxFSW_EVENT_WARNING | wxFSW_EVENT_ERROR
};

