#pragma once
#include <cstdint>
#include <memory>
#include <vector>
#include <Windows.h>

#include "GVONoncopyable.h"


class GVOImage :private GVONoncopyable {
private:
	std::wstring m_fileName;
	HBITMAP m_hbmp;
	SIZE m_size;
	uint8_t * m_bits;
	uint32_t m_stride;

public:
	GVOImage() :
		m_hbmp(),
		m_size(),
		m_bits(),
		m_stride()
	{
	}
	~GVOImage()
	{
		reset();
	}
	void reset()
	{
		if ( m_hbmp ) {
			::DeleteObject( m_hbmp );
			m_hbmp = NULL;
		}
		m_fileName = L"";
		m_size = SIZE();
		m_stride = 0;
	}
	void copy( const GVOImage & src )
	{
		createDIBImage( src.m_size );
		::memcpy( m_bits, src.m_bits, src.m_size.cy * src.m_stride );
	}
	bool stretchCopy( const GVOImage& src, const SIZE& size )
	{
		return stretchCopy( src, size.cx, size.cy );
	}
	bool stretchCopy( const GVOImage& src, uint32_t width, uint32_t height );
	bool isCompatible( const SIZE& size ) const
	{
		if ( !m_hbmp ) {
			return false;
		}
		if ( m_size.cx != size.cx || m_size.cy != size.cy ) {
			return false;
		}
		return true;
	}
	HBITMAP bitmapHandle() const
	{
		return m_hbmp;
	}
	const SIZE& size() const
	{
		return m_size;
	}
	LONG width() const
	{
		return m_size.cx;
	}
	LONG height() const
	{
		return m_size.cy;
	}
	uint32_t stride() const
	{
		return m_stride;
	}
	const uint8_t * imageBits() const
	{
		return m_bits;
	}
	uint8_t * mutableImageBits()
	{
		return m_bits;
	}
	const std::wstring& fileName() const
	{
		return m_fileName;
	}

	bool createDIBImage( int width, int height );
	bool createDIBImage( const SIZE& size )
	{
		return createDIBImage( size.cx, size.cy );
	}
	bool loadFromFile( const std::wstring& fileName );
};
