#pragma once
#include <cstdint>
#include <memory>
#include <Windows.h>

// PNGƂ̓ǂݍݗp
#include <objbase.h>
#include <gdiplus.h>
#include <gdiplusbitmap.h>


class GVOImage {
private:
	GVOImage( const GVOImage& );
	GVOImage& operator=(const GVOImage&);

private:
	std::wstring m_fileName;
	HBITMAP m_hbmp;
	SIZE m_size;
	uint8_t * m_bits;

public:
	GVOImage() : m_hbmp(), m_size(), m_bits()
	{
	}
	~GVOImage()
	{
		reset();
	}
	void reset()
	{
		if ( m_hbmp ) {
			::DeleteObject( m_hbmp );
			m_hbmp = NULL;
		}
		m_fileName = L"";
	}
	bool stretchCopy( const GVOImage& src, const SIZE& size )
	{
		return stretchCopy( src, size.cx, size.cy );
	}
	bool stretchCopy( const GVOImage& src, uint32_t width, uint32_t height )
	{
		const SIZE size = { width, height };
		if ( !createDIBImage( size ) ) {
			return false;
		}

		HDC hdc = ::GetDC( NULL );
		HDC hdcSrc = ::CreateCompatibleDC( hdc );
		HDC hdcDst = ::CreateCompatibleDC( hdc );

		::SaveDC( hdcSrc );
		::SaveDC( hdcDst );
		::SelectObject( hdcDst, m_hbmp );
		::SelectObject( hdcSrc, src.m_hbmp );

		if ( m_size.cx != src.m_size.cx || m_size.cy != src.m_size.cy ) {
			POINT org;
			::GetBrushOrgEx( hdcDst, &org );
			::SetStretchBltMode( hdcDst, HALFTONE );
			::SetBrushOrgEx( hdcDst, org.x, org.y, NULL );

			::StretchBlt( hdcDst, 0, 0, m_size.cx, m_size.cy,
				hdcSrc, 0, 0, src.m_size.cx, src.m_size.cy,
				SRCCOPY );
		}
		else {
			::BitBlt( hdcDst, 0, 0, m_size.cx, m_size.cy,
				hdcSrc, 0, 0, SRCCOPY );
		}

		::RestoreDC( hdcSrc, -1 );
		::DeleteDC( hdcSrc );
		::RestoreDC( hdcDst, -1 );
		::DeleteDC( hdcDst );
		::ReleaseDC( NULL, hdc );
		return true;
	}
	bool isCompatible( const SIZE& size ) const
	{
		if ( !m_hbmp ) {
			return false;
		}
		if ( m_size.cx != size.cx || m_size.cy != size.cy ) {
			return false;
		}
		return true;
	}
	HBITMAP bitmapHandle() const
	{
		return m_hbmp;
	}
	const SIZE& size() const
	{
		return m_size;
	}
	LONG width() const
	{
		return m_size.cx;
	}
	LONG height() const
	{
		return m_size.cy;
	}
	const uint8_t * imageBits() const
	{
		return m_bits;
	}
	uint8_t * mutableImageBits()
	{
		return m_bits;
	}
	const std::wstring& fileName() const
	{
		return m_fileName;
	}

	bool createDIBImage( int width, int height )
	{
		reset();

		BITMAPINFOHEADER bmih = { sizeof(bmih) };
		bmih.biWidth = width;
		bmih.biHeight = -height;
		bmih.biPlanes = 1;
		bmih.biBitCount = 24;
		bmih.biSizeImage = width * height * 3;
		m_hbmp = ::CreateDIBSection( NULL, (LPBITMAPINFO)&bmih, DIB_RGB_COLORS, (void **)&m_bits, NULL, 0 );
		if ( !m_hbmp ) {
			return false;
		}
		m_size.cx = width;
		m_size.cy = height;
		return true;
	}
	bool createDIBImage( const SIZE& size )
	{
		reset();

		BITMAPINFOHEADER bmih = { sizeof(bmih) };
		bmih.biWidth = size.cx;
		bmih.biHeight = -size.cy;
		bmih.biPlanes = 1;
		bmih.biBitCount = 24;
		bmih.biSizeImage = bmih.biWidth * bmih.biHeight * 3;
		m_hbmp = ::CreateDIBSection( NULL, (LPBITMAPINFO)&bmih, DIB_RGB_COLORS, (void **)&m_bits, NULL, 0 );
		if ( !m_hbmp ) {
			return false;
		}
		m_size = size;
		return true;
	}
	bool loadFromFile( const std::wstring& fileName )
	{
		std::auto_ptr<Gdiplus::Bitmap> image;
		image.reset( Gdiplus::Bitmap::FromFile( fileName.c_str() ) );
		image->GetHBITMAP( Gdiplus::Color( 0, 0, 0 ), &m_hbmp );
		image.reset();
		if ( !m_hbmp ) {
			return false;
		}
		BITMAP bmp = { 0 };
		::GetObject( m_hbmp, sizeof(bmp), &bmp );
		m_size.cx = bmp.bmWidth;
		m_size.cy = ::abs( bmp.bmHeight );
		m_fileName = fileName;
		return true;
	}
};
