#include "stdafx.h"
#include "GVORenderer.h"
#include "GVOWorldMap.h"
#include "GVOConfig.h"



namespace {
	const double k_scaleStep = 0.125;	// 12.5%
	const double k_minScale = 0.125;	// 12.5%
	const double k_maxScale = 4.00;		// 400%

	// Google搶Hun̊O40,075kmvu1mbg1.85200kmv
	// 1EW40,075km/16384points
	// 1bŃQ[0.4
	//
	// nOԓaZoB
	// ԓa6378.137Ȃ̂ŊO2*M_PI_*6378.137
	inline double s_velocityByKnot( const double velocity )
	{
		static const double k_knotFactor = (2 * M_PI * 6378.137) / 16384.0 / 0.4 / 1.852;
		return velocity * k_knotFactor;
	}
}


void GVORenderer::setup( HDC hdcPrimary )
{
	m_hdcPrimary = hdcPrimary;
}


void GVORenderer::teardown()
{
	m_hdcPrimary = NULL;
}


void GVORenderer::setWorldMap( const GVOWorldMap& worldMap )
{
	m_worldMap = &worldMap;
	const GVOImage & map = m_worldMap->image();
	m_mapImage.copy( map );
}


void GVORenderer::setConfig( const GVOConfig& config )
{
	m_positionUpdated = config.m_traceShipPositionEnabled;
	m_focusPointInWorldCoord = config.m_initialSurveyCoord;
	m_shipPointInWorld = config.m_initialSurveyCoord;
	m_previousDrawPointInWorld = m_shipPointInWorld;
	m_shipVectorLineEnabled = config.m_shipVectorLineEnabled;
	m_speedMeterEnabled = config.m_speedMeterEnabled;
	m_traceShipEnabled = config.m_traceShipPositionEnabled;
}


void GVORenderer::setViewSize( const SIZE& viewSize )
{
	m_viewSize = viewSize;
}


SIZE GVORenderer::scaledMapSize() const
{
	SIZE size = {
		LONG( m_mapImage.width() * m_viewScale ),
		LONG( m_mapImage.height() * m_viewScale )
	};
	return size;
}


POINT GVORenderer::mapOriginInView() const
{
	const POINT viewCenter = viewCenterPoint();
	const SIZE mapSize = scaledMapSize();
	const POINT worldPosInView = drawOffsetFromWorldCoord( m_focusPointInWorldCoord );

	POINT mapTopLeft = {
		viewCenter.x - worldPosInView.x,
		viewCenter.y - worldPosInView.y
	};
	if ( m_viewSize.cx < mapSize.cx ) {
		while ( 0 < mapTopLeft.x ) {
			mapTopLeft.x -= mapSize.cx;
		}
	}

	return mapTopLeft;
}


void GVORenderer::offsetFocusInViewCoord( const POINT& offset )
{
	const double dx = ((double)offset.x / m_viewScale) / m_mapImage.width();
	const double dy = ((double)offset.y / m_viewScale) / m_mapImage.height();

	LONG x = m_focusPointInWorldCoord.x + LONG( dx * k_worldWidth );
	LONG y = m_focusPointInWorldCoord.y + LONG( dy * k_worldHeight );
	y = max( 0, min( y, k_worldHeight ) );
	while ( x < 0 ) {
		x += k_worldWidth;
	}
	while ( k_worldWidth < x ) {
		x -= k_worldWidth;
	}

	m_focusPointInWorldCoord.x = x;
	m_focusPointInWorldCoord.y = y;
}


bool GVORenderer::zoomIn()
{
	double scale = m_viewScale;
	double step = k_scaleStep;

	scale = m_viewScale + step;
	if ( k_maxScale < scale ) {
		scale = k_maxScale;
	}
	if ( m_viewScale != scale ) {
		m_viewScale = scale;
		return true;
	}
	return false;
}


bool GVORenderer::zoomOut()
{
	double scale = m_viewScale;
	double step = k_scaleStep;

	scale = m_viewScale - step;
	if ( scale < k_minScale ) {
		scale = k_minScale;
	}
	if ( m_viewScale != scale ) {
		m_viewScale = scale;
		return true;
	}
	return false;
}


void GVORenderer::resetViewScale()
{
	m_viewScale = 1.0;
}


POINT GVORenderer::drawOffsetFromWorldCoord( const POINT&worldCoord ) const
{
	const POINT worldPosInImage = m_worldMap->imageCoordFromWorldCoord( worldCoord );
	const POINT drawOffset = {
		LONG( worldPosInImage.x * m_viewScale ),
		LONG( worldPosInImage.y * m_viewScale )
	};
	return drawOffset;
}


void GVORenderer::updateShipState( const POINT& worldCoord, const GVOVector& shipVector, const double shipVelocity )
{
	m_shipVector = shipVector;
	m_shipVelocity = shipVelocity;

	if ( m_traceShipEnabled ) {
		m_focusPointInWorldCoord = worldCoord;
	}
	if ( m_shipPointInWorld.x != worldCoord.x
		|| m_shipPointInWorld.y != worldCoord.y ) {

		m_positionUpdated = true;
		m_shipPointInWorld = worldCoord;
	}
	updateShipRouteMap();
}


void GVORenderer::updateShipRouteMap()
{
	// ړĂȂΉȂ
	if ( !m_positionUpdated ) {
		return;
	}
	// ړĂĂqHqȂȂ`悵Ȃ
	if ( !m_linkRoute ) {
		m_previousDrawPointInWorld = m_shipPointInWorld;
		m_linkRoute = true;
		return;
	}

	const POINT& latestPoint = m_worldMap->imageCoordFromWorldCoord( m_shipPointInWorld );
	const POINT& previousPoint = m_worldMap->imageCoordFromWorldCoord( m_previousDrawPointInWorld );

	// `WꏏȂ牽Ȃ
	if ( latestPoint.x == previousPoint.x
		&& latestPoint.y == previousPoint.y ) {
		return;
	}


	// ŐVWO̍W֐
	// EׂꍇɍlKvB
	// ړʂ𒴂`揈Ȃ悤ɂ̂X}[g


	// EׂƂ݂Ȃ臒l
	const int k_distanceThreshold = m_mapImage.width() / 2;

	const LONG xMin = min( latestPoint.x, previousPoint.x );
	const LONG xMax = max( latestPoint.x, previousPoint.x );
	const int xDistance = xMax - xMin;


	HDC hdcMem = ::CreateCompatibleDC( m_hdcPrimary );
	::SaveDC( hdcMem );

	HPEN hpen = ::CreatePen( PS_SOLID, 1, RGB( 255, 255, 255 ) );

	::SelectObject( hdcMem, m_mapImage.bitmapHandle() );
	::SelectObject( hdcMem, hpen );

	if ( k_distanceThreshold < xDistance ) {
		if ( previousPoint.x < latestPoint.x ) {	// E𐼂ɌČׂ
			// ̉ʊO`
			LONG x3 = latestPoint.x - m_mapImage.width();

			::MoveToEx( hdcMem, x3, latestPoint.y, NULL );
			::LineTo( hdcMem, previousPoint.x, previousPoint.y );

			// ẺʊOɌ`
			LONG x4 = previousPoint.x + m_mapImage.width();
			::MoveToEx( hdcMem, latestPoint.x, latestPoint.y, NULL );
			::LineTo( hdcMem, x4, previousPoint.y );
		}
		else {				// E𓌂ɌČׂ
			// ̉ʊO`
			LONG x3 = previousPoint.x - m_mapImage.width();

			::MoveToEx( hdcMem, latestPoint.x, latestPoint.y, NULL );
			::LineTo( hdcMem, x3, previousPoint.y );

			// ẺʊOɌ`
			LONG x4 = latestPoint.x + m_mapImage.width();
			::MoveToEx( hdcMem, x4, latestPoint.y, NULL );
			::LineTo( hdcMem, previousPoint.x, previousPoint.y );
		}
	}
	else {
		// ׂȂ`
		::MoveToEx( hdcMem, latestPoint.x, latestPoint.y, NULL );
		::LineTo( hdcMem, previousPoint.x, previousPoint.y );
	}


	::RestoreDC( hdcMem, -1 );
	::DeleteDC( hdcMem );
	::DeleteObject( hpen );

	m_previousDrawPointInWorld = m_shipPointInWorld;
}


void GVORenderer::clearShipRoute()
{
	m_mapImage.copy( m_worldMap->image() );
}


void GVORenderer::render()
{
	if ( !m_backBuffer.isCompatible(m_viewSize) ) {
		m_backBuffer.createDIBImage( m_viewSize );
	}
	HDC hdcBackbuffer = ::CreateCompatibleDC( m_hdcPrimary );
	::SaveDC( hdcBackbuffer );

	::SelectObject( hdcBackbuffer, m_backBuffer.bitmapHandle() );
	RECT rc = { 0, 0, m_backBuffer.width(), m_backBuffer.height()};
	::FillRect( hdcBackbuffer, &rc, (HBRUSH)::GetStockObject( BLACK_BRUSH ) );

	drawMap( hdcBackbuffer, m_shipVector );

	if ( m_speedMeterEnabled ) {
		drawSpeedMeter( hdcBackbuffer, m_shipVelocity );
	}

	::BitBlt( m_hdcPrimary, 0, 0, m_backBuffer.width(), m_backBuffer.height(),
		hdcBackbuffer, 0, 0, SRCCOPY );
	::RestoreDC( hdcBackbuffer, -1 );
	::DeleteDC( hdcBackbuffer );
}


void GVORenderer::drawMap( HDC hdc, const GVOVector& shipVector )
{
	const SIZE mapSize = scaledMapSize();
	const GVOImage *mapImage = &m_mapImage;

	::SaveDC( hdc );
	if ( m_viewScale < 1.0 ) {
		POINT org;
		::GetBrushOrgEx( hdc, &org );
		::SetStretchBltMode( hdc, HALFTONE );
		::SetBrushOrgEx( hdc, org.x, org.y, NULL );
	}
	else {
		::SetStretchBltMode( hdc, COLORONCOLOR );
	}

	HDC hdcMem = ::CreateCompatibleDC( hdc );
	::SaveDC( hdcMem );

	::SelectObject( hdcMem, mapImage->bitmapHandle() );

	const POINT mapTopLeft = mapOriginInView();

	int xDrawOrigin, yDrawOrigin;
	xDrawOrigin = mapTopLeft.x;
	yDrawOrigin = mapTopLeft.y;

	if ( 0 < xDrawOrigin ) {
		xDrawOrigin = (xDrawOrigin % mapSize.cx) - mapSize.cx;
	}
	const int xInitial = xDrawOrigin;	// [̕`JnxW
	int drawn = xInitial;				// `ςݍW

	// En}ɕׂĕ`
	// i`œK͏ȗj
	while ( drawn < m_viewSize.cx ) {
		::StretchBlt( hdc,
			xDrawOrigin, yDrawOrigin,
			mapSize.cx, mapSize.cy,
			hdcMem,
			0, 0,
			mapImage->width(), mapImage->height(),
			SRCCOPY );

		xDrawOrigin += mapSize.cx;
		drawn += mapSize.cx;
	}


	const POINT shipPointOffset = drawOffsetFromWorldCoord( m_shipPointInWorld );

	// jH\`
	if ( shipVector.length() != 0.0 && m_shipVectorLineEnabled ) {
		const int penWidth = max( 1, int( 1 * m_viewScale ) );
		HPEN courseLinePen = ::CreatePen( PS_SOLID, penWidth, RGB( 255, 0, 255 ) );
		HGDIOBJ oldPen = ::SelectObject( hdc, courseLinePen );

		const LONG k_lineLength = k_worldHeight;
		const POINT reachPointOffset = drawOffsetFromWorldCoord(
			shipVector.pointFromOriginWithLength( m_shipPointInWorld, k_lineLength )
			);

		// Ăn}摜̕`悷
		drawn = xInitial;
		xDrawOrigin = xInitial;
		while ( drawn < m_viewSize.cx ) {
			const POINT shipPointInView = {
				xDrawOrigin + shipPointOffset.x,
				yDrawOrigin + shipPointOffset.y
			};
			const POINT reachPointInView = {
				xDrawOrigin + reachPointOffset.x,
				yDrawOrigin + reachPointOffset.y
			};
			::MoveToEx( hdc, shipPointInView.x, shipPointInView.y, NULL );
			::LineTo( hdc, reachPointInView.x, reachPointInView.y );
			xDrawOrigin += mapSize.cx;
			drawn += mapSize.cx;
		}

		::SelectObject( hdc, oldPen );
		::DeleteObject( courseLinePen );
	}

	// D̈ʒu`
	const SIZE shipMarkSize = { 6, 6 };
	HBRUSH shipBrush = ::CreateSolidBrush( RGB( 51, 238, 153 ) );
	HGDIOBJ prevBrush = ::SelectObject( hdc, shipBrush );

	// Ăn}摜̕`悷
	drawn = xInitial;
	xDrawOrigin = xInitial;
	while ( drawn < m_viewSize.cx ) {
		::Ellipse( hdc,
			xDrawOrigin + shipPointOffset.x - shipMarkSize.cx / 2,
			yDrawOrigin + shipPointOffset.y - shipMarkSize.cy / 2,
			xDrawOrigin + shipPointOffset.x + shipMarkSize.cx,
			yDrawOrigin + shipPointOffset.y + shipMarkSize.cy );

		xDrawOrigin += mapSize.cx;
		drawn += mapSize.cx;
	}
	::SelectObject( hdc, prevBrush );
	::DeleteObject( shipBrush );


	::RestoreDC( hdcMem, -1 );
	::DeleteDC( hdcMem );
	::RestoreDC( hdc, -1 );
}


void GVORenderer::drawSpeedMeter( HDC hdc, double shipVelocity )
{
	const double velocity = s_velocityByKnot( shipVelocity );
	wchar_t buf[4096] = { 0 };
	swprintf( buf, _countof( buf ), L"%.2f kt", velocity );

	RECT rc = { 0 };
	::DrawText( hdc, buf, -1, &rc, DT_SINGLELINE | DT_RIGHT | DT_TOP | DT_CALCRECT );
	const int width = rc.right - rc.left;
	rc.left = m_viewSize.cx - width;
	rc.right = rc.left + width;
	::DrawText( hdc, buf, -1, &rc, DT_SINGLELINE | DT_RIGHT | DT_TOP );
}
