﻿//----------------------------------------------------------------------
//
//			File:			"MainWindow.cpp"
//			Created:		26-Aug-2012
//			Author:			Nobuhide Tsuda
//			Description:	MainWindow クラス実装
//
//----------------------------------------------------------------------

/*

	Copyright (C) 2012 by Nobuhide Tsuda

	HalfGammon のライセンスは CDDL 1.0 です。
	http://opensource.org/licenses/cddl-1.0
	無保証・無サポートですが、無償で利用でき、商用アプリでもソースコードを流用することが可能です。 
	ソースコードを流用した場合、流用したファイルのライセンスは CDDL 1.0 となります。
	流用部分の著作権は HalfGammon のそれのままです。
	作者は、プログラマにとって不自由極まりないのに自由だ自由だと言い張るGPL系が嫌いなので、
	CDDL 1.0 を選択しました。
	このライセンスはGPL系より再利用しやすく、GPL系とは矛盾するライセンスなので、
	HalfGammon のソースをGPL系プロジェクトで使用することはできません。

*/

#include <QtGui>
#include <QtNetwork>
#include <time.h>
#include "mainwindow.h"
#include "Board.h"
#include "TreeSearch.h"
#include "NewGameDlg.h"
#include "SettingsDlg.h"
#include "StatisticsDlg.h"
#include "GameAgainDlg.h"

#define		VERSION_STR		"0.001.Dev"
#define		URL_INFO		"http://vivi.dyndns.org/games/NikoNikoGammon/info.txt"
#define		FPS				50
#define		INIT_BOARD		"G B2  W2B1W1B2  W2 G"			//	初期状態
//#define		INIT_BOARD		"G B2B2. . . W2W2W2W2. B2B2 G"		//	ピップ的には黒有利だが、白は形がいい
//#define		INIT_BOARD		"G W2W2W2            B1B1W2 G"		//	クローズアウト
//#define		INIT_BOARD		"G W2W2W2            B1W2B1 G"
//#define		INIT_BOARD		"G W2W2W2                B1W2"		//	クローズアウト＆バーに2個
//#define		INIT_BOARD		"G       W4        B4       G"

#define		SAME_DICE_SEQ		0		//	テスト用にサイコロの目を固定
#define		MULTI_THREAD		1		//	先読みをマルチスレッドで行う

#define		DST_MARK_MARGIN	(CELL_WD/3)
#define		DST_MARK_WD		(CELL_WD - CELL_WD*2/3)
#define		SEL_MARK_MARGIN	(CELL_WD/16)
#define		SEL_MARK_WD		(CELL_WD/16)
#define		SEL_MARK_LENGTH	(CELL_WD*3/16)

enum {
	GM_HUMAN_VS_COMP = 0,
	GM_COMP_VS_COMP,
	GM_HUMAN_VS_HUMAN,
};

MainWindow::MainWindow(QWidget *parent, Qt::WFlags flags)
	: QMainWindow(parent, flags)
{
#if	!SAME_DICE_SEQ
	qsrand((int)time(0));
#endif
	ui.setupUi(this);
	ui.mainToolBar->setWindowTitle("MainToolBar");
	ui.mainToolBar->setObjectName("MainToolBar");
	createDockWindows();
	setWindowTitle(QString(qApp->applicationName() + " version %1").arg(VERSION_STR));
	resize(900, 700);
#if 0
	QRect geo = geometry();
	geo.setWidth(900);
	geo.setHeight(700);
	setGeometry(geo);
#endif
	readSettings();
	m_firstMoveBlack = false;
	//m_hit = false;
	m_turn = 0;
	m_nextItem = 0;
	m_actDialog = 0;
	m_movingProgress = -1;
	m_gameInMatch = 0;
	m_blackWinInMatch = 0;
	m_blackPointInMatch = 0;
	m_whitePointInMatch = 0;
	m_view = new QGraphicsView(m_scene = new QGraphicsScene());
	m_view->setRenderHint(QPainter::Antialiasing);
	m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	m_view->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	setCentralWidget(m_view);
	setupBoardItems();
	m_treeSearch = new TreeSearch;
	m_thread = new QThread;
#if		MULTI_THREAD
	m_treeSearch->moveToThread(m_thread);
#endif
	connect(m_treeSearch, SIGNAL(searchFinished(double)), this, SLOT(onSearchFinished(double)));
	m_thread->start();


	m_board = new Board(INIT_BOARD);	//	初期状態
	placePieces();
	m_gameState = GS_INIT;
	//newGame();

	m_timer = new QTimer();
	connect(m_timer, SIGNAL(timeout()), this, SLOT(onTimer()));
	m_timer->start(1000/FPS);		//	50fps

	m_networkAccessManager = new QNetworkAccessManager(this);
	connect(m_networkAccessManager, SIGNAL(finished(QNetworkReply*)),
			this, SLOT(replyFinished(QNetworkReply*)));
	QString infoURL = QString("%1?from=%2").arg(URL_INFO).arg(VERSION_STR);
	m_networkAccessManager->get(QNetworkRequest(QUrl(infoURL)));
}
void MainWindow::on_action_NewGame_triggered()
{
	//QMessageBox mb;
	//mb.exec();

	if( m_gameState != GS_INIT ) return;
	//	undone: 先手後手を決める
	NewGameDlg aDlg;
	aDlg.setGameMode(m_gameMode);
	aDlg.setCompType(m_compType);
	aDlg.setComp2Type(m_comp2Type);
	aDlg.onModeChanged();
	if( aDlg.exec() == QDialog::Accepted ) {
		m_MonteCarlo = false;
		m_gameMode = aDlg.gameMode();
		m_compType = aDlg.compType();
		m_comp2Type = aDlg.comp2Type();
		writeSettings();
		if( !aDlg.willContinuePrevMatch() ) {
			m_gameInMatch = 0;
			m_blackWinInMatch = 0;
			m_blackPointInMatch = 0;
			m_whitePointInMatch = 0;
		}
		newGame();
	}
}
void MainWindow::on_action_MonteCarlo_triggered()
{
	m_MonteCarlo = true;
	m_gameInMatch = 0;
	m_blackWinInMatch = 0;
	m_blackPointInMatch = 0;
	m_whitePointInMatch = 0;
	newGame();
}
void MainWindow::on_action_Resign_triggered()
{
	if( m_gameState == GS_INIT || m_gameMode == GM_COMP_VS_COMP )
		return;
	if( QMessageBox::question(this, qApp->applicationName(),
								tr("Are you sure to resign this game ?\n") +
								tr("This game will be regarded as a defeat."),
							QMessageBox::Yes | QMessageBox::No)
			!= QMessageBox::Yes )
	{
		return;
	}
	QString mess;
	int hr = m_humanStat.m_rating;
	if( isBlackTurn() )
		onWhiteWin(mess, true);
	else
		onBlackWin(mess, true);
	onGameOver(mess, hr);
}
void MainWindow::on_action_Undo_triggered()
{
	if( m_gameState != GS_WAIT_FOR_DECIDED
		&& m_gameState != GS_HUMAN_SRC
		&& m_gameState != GS_HUMAN_DST 
		|| m_boardHist.isEmpty() )
	{
		return;
	}
	*m_board = m_boardHist.last();
	m_boardHist.pop_back();
	addDice(m_diceHist.last());
	m_diceHist.pop_back();
	m_gameState = GS_INIT_HUMAN_SRC;
	if( m_curTurnMvIX > 0 ) {
		--m_curTurnMvIX;
		m_curTurn.m_mvs[m_curTurnMvIX].m_src = 0;
		m_curTurn.m_mvs[m_curTurnMvIX].m_dst = 0;
	}
	placePieces();
	updateDiceItems();
}
void MainWindow::sound(const QString &path)
{
	if( !ui.action_Voice->isChecked() ) return;
#ifdef		_DEBUG
	QString soundPath = "C:/user/HalfGammon/dist";
#else
	QString soundPath = qApp->applicationDirPath();
#endif
	soundPath += "/sound/" + path;
	QDir soundDir(soundPath);
	QStringList filter; filter << "*.wav";
	QStringList lst = soundDir.entryList(filter);
	if( lst.isEmpty() ) return;
	int r = qrand() % lst.size();    //  再生ファイルをランダムに選ぶ
	QString wavFile = soundDir.absolutePath() + "/" + lst[r];
	QSound s(wavFile);
	s.play();
}
void MainWindow::newGame()
{
	m_gameState = GS_NEW_GAME;
	++m_gameInMatch;
	clearOutput();
	sound("gameStart");
	//QDialog aDlg;
	//aDlg.exec();
	//QMessageBox::information(this, "test", "test");		//	メッセージボックスを出すと音声再生が止まるみたいだ
	m_board->setPosition(INIT_BOARD);	//	初期状態
	m_turnRecord.clear();
	m_curTurn = TurnRecord();
	m_curTurnMvIX = 0;
	m_recordView->clear();
	//m_board->setPosition("G                   >1<1>1 G");
	//m_board->setPosition(">2. . <3. . >3<3. . >3. <2 G");
	//m_board->setPosition(/*b=*/0x00200030003000, /*w=*/0x00030003000110);
	//m_board->setPosition(/*b=*/0x0000000033000, /*w=*/0x30000000000);
	//	                      ○
	//	・○○・・・・・●●●○
	//	・○○・●・●・●●●○ ○
	//  １２３４５６７８９101112
	//m_board->setPosition(/*b=*/0x00000101022200, /*w=*/0x00220000000030, 0, /*bw=*/1);
	placePieces();
	m_gameState = m_nextGameState = GS_INIT;
	if( !m_MonteCarlo )
		m_firstMoveBlack = !m_firstMoveBlack;
	m_blackTurn = m_firstMoveBlack ? 1 : -1;	//	for BLACK
	m_turn = 1;
	//m_blackTurn = -1;	//	for WWHITE
	updateNextItem();

	m_blackNameTextItem->setText( m_gameMode == GM_HUMAN_VS_COMP ? "human" : "computer-2");
	m_blackNameTextItem->setPos(PIP_X - CELL_WD/2 - 8 - m_blackNameTextItem->boundingRect().width(), PIP_Y);
	m_whiteNameTextItem->setText( m_gameMode == GM_HUMAN_VS_HUMAN ? "guest" : "computer-1");

	m_gameState = GS_INIT_ROLLING_DICE;
#if 0
	if( m_gameMode == GM_HUMAN_VS_COMP && isBlackTurn() || m_gameMode == GM_HUMAN_VS_HUMAN )
		m_gameState = GS_INIT_ROLLING_DICE;
	else
		m_gameState = GS_INIT_COMP_TURN;
#endif
}
void MainWindow::createDockWindows()
{
	QDockWidget *dock;
	dock = new QDockWidget(tr("Output"));
	dock->setWidget(m_output = new QPlainTextEdit());
	m_output->setFont(QFont("Courier New", 12));
	m_output->setReadOnly(true);
	dock->setAllowedAreas(Qt::AllDockWidgetAreas);
	addDockWidget(Qt::BottomDockWidgetArea, dock);
	setDockOptions( /*AnimatedDocks |*/ AllowTabbedDocks );

	dock = new QDockWidget(tr("Record"));
	dock->setWidget(m_recordView = new QListWidget());
	m_recordView->setFont(QFont("Courier New", 12));
	//m_recordView->setReadOnly(true);
	connect(m_recordView, SIGNAL(currentRowChanged(int)), this, SLOT(currentRowChanged(int)));
	dock->setAllowedAreas(Qt::AllDockWidgetAreas);
	addDockWidget(Qt::BottomDockWidgetArea, dock);
	setDockOptions( /*AnimatedDocks |*/ AllowTabbedDocks );

	//appendToRecord();
	//appendToRecord();
}
void MainWindow::currentRowChanged(int row)
{
	if( m_gameState != GS_INIT ) return;
	m_turn = row;
	m_board->setPosition(INIT_BOARD);	//	初期状態
	m_blackTurn = m_firstMoveBlack;
	for(int i = 0; i < row; ++i) {
		for(int k = 0; k < 4; ++k) {
			const int src = m_turnRecord[i].m_mvs[k].m_src;
			const int dst = m_turnRecord[i].m_mvs[k].m_dst;
			if( !src && !dst ) break;
			if( m_blackTurn )
				m_board->moveBlack(src, dst);
			else
				m_board->moveWhite(reverseIX(src), reverseIX(dst));
		}
		m_blackTurn = !m_blackTurn;
	}
	placePieces();
	placeMovesItem(m_turnRecord[row]);
	updateNextItem();
	m_dice[0] = m_turnRecord[row].m_dice1;
	m_dice[1] = m_turnRecord[row].m_dice2;
	updateDiceItems();
	eval_playD(1);
	//const sn = m_board->seqNumber();
	Board bd(*m_board);
	if( m_blackTurn ) bd.swapBW();
	double ev = bd.expectedValue();
	//if( m_blackTurn ) ev = -ev;
	doOutput(QString("expectedValue = %1\n").arg(ev));

}
void MainWindow::eval_playD(int rdepth)
{
	if( !m_dice[0] ) return;
	setEvalFunc(::eval6);
	clearTerminalNodeCount();
	QList<Moves> lst = m_blackTurn
							? m_board->blackMovesList3(m_dice[0], m_dice[1])
							: m_board->whiteMovesList3(m_dice[0], m_dice[1]);
	foreach(Moves mvs, lst) {
		Board bd(*m_board);
		QString mvsText = "  ";
		foreach(Move mv, mvs) {
			int src = mv.m_src;
			int dst = qMax((int)IX_GOAL, (int)(src - mv.m_d));
			if( m_blackTurn )
				bd.moveBlack(mv);
			else {
				bd.moveWhite(mv);
				src = reverseIX(src);
				dst = reverseIX(dst);
			}
			mvsText += QString("%1/%2 ").arg(src).arg(dst);
		}
		int eval = 0;
		if( !m_blackTurn ) {	//	１手後なので、白黒逆になる
			bd.swapBW();
			eval = -negaChanceNodeEvalX(bd, rdepth);
			bd.swapBW();
		} else
			eval = -negaChanceNodeEvalX(bd, rdepth);
		//mvsText = "  " + bd.position() + mvsText;
		//mvsText += "\n";
		doOutput(QString("  %1 '%2'%3\n").arg(eval, 6).arg(bd.position()).arg(mvsText));
	}
	doOutput(QString("  (depth = %1, TerminalNodeCount = %2)\n")
					.arg(rdepth)
					.arg(terminalNodeCount()));
}
void MainWindow::on_action_Eval_ply1_triggered()
{
	doOutput(m_board->position() + "\n");
	eval_playD(1);
}
void MainWindow::on_action_Eval_ply2_triggered()
{
	doOutput(m_board->position() + "\n");
	eval_playD(2);
}
void MainWindow::on_action_Eval_ply3_triggered()
{
	doOutput(m_board->position() + "\n");
	eval_playD(3);
}
void MainWindow::on_action_Eval_ply4_triggered()
{
	doOutput(m_board->position() + "\n");
	eval_playD(4);
}
void MainWindow::replyFinished(QNetworkReply* reply)
{
	if( reply->error() == QNetworkReply::NoError ) {
		QTextCodec *codec = QTextCodec::codecForName("UTF-8");
		QString buffer = codec->toUnicode(reply->readAll());
		doOutput(buffer);
	} else {
		///doOutput(reply->errorString());
	}
}
void MainWindow::doOutput(const QString &text)
{
#if 1
	QTextCursor cur = m_output->textCursor();
	cur.movePosition(QTextCursor::End);		//	末尾にカーソル移動
	m_output->setTextCursor(cur);
	cur.insertText(text);
#endif
}
void MainWindow::clearOutput()
{
	m_output->clear();
}
void MainWindow::setupBoardItems()
{
	//	ダイス
	m_diceRect1 = m_scene->addRect(DICE_X, DICE_Y, DICE_WD, DICE_WD);
	m_diceRect2 = m_scene->addRect(DICE_X + CELL_WD, DICE_Y, DICE_WD, DICE_WD);
	//m_scene->addRect(DICE_X + CELL_WD * 2, DICE_Y, DICE_WD, DICE_WD);
	//m_scene->addRect(DICE_X + CELL_WD * 3, DICE_Y, DICE_WD, DICE_WD);
	QFont font("Arial Black", CELL_WD/2, QFont::Bold);
	m_diceItems << m_scene->addSimpleText("1", font);
	m_diceItems << m_scene->addSimpleText("1", font);
	//m_diceItems << m_scene->addSimpleText("1", font);
	//m_diceItems << m_scene->addSimpleText("1", font);
	const qreal margin = (DICE_WD - m_diceItems[0]->boundingRect().width()) / 2;
	m_diceItems[0]->setPos(DICE_X + margin, DICE_NUM_Y);
	m_diceItems[1]->setPos(DICE_X + CELL_WD + margin, DICE_NUM_Y);
	//m_diceItems[2]->setPos(DICE_X + CELL_WD * 2 + margin, DICE_NUM_Y);
	//m_diceItems[3]->setPos(DICE_X + CELL_WD * 3 + margin, DICE_NUM_Y);

	QFont font2("Arial Black", 12, QFont::Normal);
	//	プレイヤー名
	m_blackNameTextItem = m_scene->addSimpleText("human", font2);
	m_blackNameTextItem->setPos(PIP_X - CELL_WD/2 - 8 - m_blackNameTextItem->boundingRect().width(), PIP_Y);
	m_whiteNameTextItem = m_scene->addSimpleText("computer-1", font2);
	m_whiteNameTextItem->setPos(PIP_X - CELL_WD/2 - 8 - m_whiteNameTextItem->boundingRect().width(), PIP_Y2 - 6);
	//	ピップカウント
	const int pipR = CELL_WD / 4;
	m_blackPipsItem = m_scene->addEllipse(PIP_X - CELL_WD/2, PIP_Y + 6, pipR, pipR, QPen(Qt::black), m_blackColor);
	m_whitePipsItem = m_scene->addEllipse(PIP_X - CELL_WD/2, PIP_Y2, pipR, pipR, QPen(Qt::black), m_whiteColor);
	m_blackPipsTextItem = m_scene->addSimpleText("pips: 0 (0)", font2);
	m_blackPipsTextItem->setPos(PIP_X, PIP_Y);
	m_whitePipsTextItem = m_scene->addSimpleText("pips: 0 (0)", font2);
	m_whitePipsTextItem->setPos(PIP_X, PIP_Y2 - 6);
	//	盤面背景
	QBrush brush(QColor("lightgray"));
	m_scene->addRect(ORG_X, ORG_Y, CELL_WD * (N_CELL + 2), CELL_HT, QPen(Qt::black), brush);
	//m_scene->addRect(ORG_X, ORG_Y, CELL_WD, CELL_HT, QPen(Qt::black), brush);
	//m_scene->addRect(ixToPx(IX_BLACK_GOAL), ORG_Y, CELL_WD, CELL_HT, QPen(Qt::black), brush);
	QFont font3("Arial Black", 12, QFont::Bold);
	QPolygonF plygn;
	plygn << QPointF(0, -1) << QPointF(CELL_WD/2, -CELL_HT) << QPointF(CELL_WD, -1);
	for(int i = POINT_START; i >= POINT_GOAL; --i) {
		int px = pointToPx(i);
		QString label = "S/G";
		if( i != POINT_START && i != POINT_GOAL ) {
			QBrush brush(QColor(!(i & 1) ? "lightyellow" : "burlywood"));
			m_scene->addPolygon(plygn, QPen(Qt::transparent), brush)->setPos(px, ORG_Y + CELL_HT);
			label = QString::number(i);
		}
		QGraphicsSimpleTextItem *t = m_scene->addSimpleText(label, font3);
		t->setPos(px + (CELL_WD - t->boundingRect().width()) / 2, ORG_Y + CELL_HT);
	}
	for(int i = POINT_START - 1; i > POINT_GOAL; i -= MAX_DICE)
		m_scene->addRect(pointToPx(i), ORG_Y, CELL_WD * MAX_DICE, CELL_HT,
							QPen(Qt::black), QBrush(Qt::transparent));
}
void MainWindow::clearPieces()
{
	foreach(QGraphicsItem *item, m_pieces) {
		m_scene->removeItem(item);
		delete item;
	}
	m_pieces.clear();
	foreach(QGraphicsItem *item, m_movesItems) {
		m_scene->removeItem(item);
		delete item;
	}
	m_movesItems.clear();
}
bool isContinuousJump(const TurnRecord &rec, int k)
{
	int dst = rec.m_mvs[k].m_dst;
	for(int i = 0; i < 4; ++i) {
		if( rec.m_mvs[i].m_src == dst && rec.m_mvs[i].m_src != rec.m_mvs[i].m_dst )
			return true;
	}
	return false;
}
//	i < k と重複している場合は、その数を返す
int dupCount(const TurnRecord &rec, int k)
{
	int n = 0;
	for(int i = 0; i < k; ++i) {
		if( rec.m_mvs[i].m_src == rec.m_mvs[k].m_src &&
			rec.m_mvs[i].m_dst == rec.m_mvs[k].m_dst )
				++n;
	}
	return n;
}
//	着手を曲線矢印で表現
void MainWindow::placeMovesItem(const TurnRecord &rec)
{
	for(int i = 0; i < 4; ++i) {
		int src = rec.m_mvs[i].m_src;
		int dst = rec.m_mvs[i].m_dst;
		if( !src && !dst ) continue;
		//if( src > dst ) qSwap(src, dst);	//	make sure src <= dst（画面上では dst が左）
		int dupc = dupCount(rec, i);
		int srcPx = pointToPx(src) + CELL_WD / 2;
		int dstPx = pointToPx(dst) + CELL_WD / 2;
		QPainterPath path;
		path.moveTo(srcPx, ORG_Y + CELL_WD*4 - CELL_WD/2);
		path.arcTo(QRectF(dstPx, ORG_Y + CELL_WD + CELL_WD/2, srcPx - dstPx, CELL_WD * 4), 0, 180);
		if( !isContinuousJump(rec, i) ) {
			int py = ORG_Y + CELL_WD*4 - CELL_WD/2;
			path.lineTo(dstPx - CELL_WD/6, py - CELL_WD/6);
			path.moveTo(dstPx, py);
			path.lineTo(dstPx + CELL_WD/6, py - CELL_WD/6);
		}
		QGraphicsItem *item = m_scene->addPath(path,
												QPen(QBrush(QColor("black")), 6, Qt::SolidLine, Qt::RoundCap),
												QBrush(Qt::transparent));
		item->setPos(0, 2 - dupc * 16);
		m_movesItems << item;
		item = m_scene->addPath(path,
												QPen(QBrush(QColor("pink")), 6, Qt::SolidLine, Qt::RoundCap),
												QBrush(Qt::transparent));
		item->setPos(0, -dupc * 16);
		m_movesItems << item;
	}
}
void MainWindow::placePieces()
{
	doOutput(QString("%1: %2\n").arg(m_turn).arg(m_board->position()));
	clearPieces();
	const bool canBlackBearOff = m_board->canBlackBearOff();
	const bool canWhiteBearOff = m_board->canWhiteBearOff();
	int nbb = m_board->nBlackBar();
	const int ba = m_board->doesWhiteWin() ? -20 : !nbb ? 20 : 0;
	int nbw = m_board->nWhiteBar();
	const int wa = m_board->doesBlackWin() ? -20 : !nbw ? 20 : 0;
	int bpip = m_board->nBlackBar() * (N_CELL + 1);
	int wpip = m_board->nWhiteBar() * (N_CELL + 1);
	for(int i = POINT_START; i >= POINT_GOAL; --i) {
		int px = pointToPx(i);
		int n = i == POINT_WHITE_GOAL ? -m_board->nWhiteGoal() :
				i == POINT_BLACK_GOAL ? m_board->nBlackGoal() :
										m_board->position(i);
		if( n > 0 )
			bpip += i * n;		//	黒はポイント数がそのままピップス数
		if( n < 0 )
			wpip += (POINT_START - i) * -n;
		int py = ORG_Y;
		QString mouth;
		if( i == POINT_BLACK_GOAL || i == POINT_WHITE_GOAL )
			mouth = QString(QChar(0x25bd));		//	▽
		else if( n > 0 && canBlackBearOff || n < 0 && canWhiteBearOff )
			mouth = QString(QChar(0xff5e));		//	"～"
		if( n > 4 ) {
			int step = (PIECE_Y - ORG_Y) / (n - 1);
			do {
				m_pieces << addPiece(px, py, true, ba, mouth);
				py += step;
			} while( --n != 0 );
		} else if( n < -4 ) {
			int step = (PIECE_Y - ORG_Y) / (-n - 1);
			do {
				m_pieces << addPiece(px, py, false, wa, mouth);
				py += step;
			} while( ++n != 0 );
		}
		switch( n ) {
		case 4:
			m_pieces << addPiece(px, PIECE_Y - PIECE_WD * 3, true, ba, mouth);
		case 3:
			m_pieces << addPiece(px, PIECE_Y - PIECE_WD * 2, true, ba, mouth);
		case 2:
			m_pieces << addPiece(px, PIECE_Y - PIECE_WD, true, ba, mouth);
		case 1:
			m_pieces << addPiece(px, PIECE_Y, true, ba, mouth);
			break;
		case -4:
			m_pieces << addPiece(px, PIECE_Y - PIECE_WD * 3, false, wa, mouth);
		case -3:
			m_pieces << addPiece(px, PIECE_Y - PIECE_WD * 2, false, wa, mouth);
		case -2:
			m_pieces << addPiece(px, PIECE_Y - PIECE_WD, false, wa, mouth);
		case -1:
			m_pieces << addPiece(px, PIECE_Y, false, wa, mouth);
			break;
		}
	}
#if 1
	if( nbb != 0 ) {	//	バー上の黒石
		int px = ORG_X;
		int py = BAR_Y - CELL_WD / 3 * (nbb - 1);
		while( --nbb >= 0 ) {
			m_pieces << addPiece(px, py, true, -20);
			py += CELL_WD / 3;
		}
	}
	if( nbw != 0 ) {	//	バー上の白石
		int px = ORG_X + CELL_WD * IX_START;
		int py = BAR_Y - CELL_WD / 3 * (nbw - 1);
		while( --nbw >= 0 ) {
			m_pieces << addPiece(px, py, false, -20);
			py += CELL_WD / 3;
		}
	}
#else
	if( nbb != 0 ) {	//	バー上の黒石
		int px = ORG_X + CELL_WD / 3 * (nbb - 1);
		while( --nbb >= 0 ) {
			m_pieces << addPiece(px, BAR_Y, true, -20);
			px -= CELL_WD / 3;
		}
	}
	if( nbw != 0 ) {	//	バー上の白石
		int px = ORG_X + CELL_WD * IX_START - CELL_WD / 3 * (nbw - 1);
		while( --nbw >= 0 ) {
			m_pieces << addPiece(px, BAR_Y, false, -20);
			px += CELL_WD / 3;
		}
	}
#endif
	//	ピップカウント表示
	m_blackPipsTextItem->setText(QString("pips: %1 (%2)").arg(bpip).arg(bpip - wpip));
	m_whitePipsTextItem->setText(QString("pips: %1 (%2)").arg(wpip).arg(wpip - bpip));
}
QGraphicsEllipseItem *MainWindow::addCircle(qreal cx, qreal cy, qreal r,
											const QPen &pen, const QBrush &brush,
											QGraphicsItem *parent)
{
	QGraphicsEllipseItem *i = 
		m_scene->addEllipse(-r, -r, r * 2, r * 2, pen, brush);
	if( parent != 0 )
		i->setParentItem(parent);
	i->setPos(cx, cy);
	return i;
}
void MainWindow::addEyebrow(qreal cx, qreal cy, qreal angle, QGraphicsItem *parent)
{
	QPen pen(Qt::black);
	QBrush brush(Qt::black);
	QGraphicsRectItem *i = 
		m_scene->addRect(-4, -1, 8, 2, pen, brush);
	i->setParentItem(parent);
	i->setPos(cx, cy);
	i->setRotation(angle);
}
QGraphicsItem *MainWindow::addPiece(int px, int py, bool human, int ebAngle, QString mouth)
{
	QBrush brush(human ? m_blackColor : m_whiteColor);
	QPen pen(Qt::black);
	QGraphicsItem *p = addCircle(px + CELL_WD/2, py + CELL_WD/2, PIECE_WD/2, pen, brush);
	//QGraphicsDropShadowEffect *efct = new QGraphicsDropShadowEffect();
	//efct->setXOffset(2);
	//efct->setYOffset(2);
	//efct->setBlurRadius(2);
	//p->setGraphicsEffect(efct);
	//m_pieces << p;
	const qreal EYE_SIZE = (qreal)PIECE_WD / 6;
	const qreal EYE_SPACE = (qreal)PIECE_WD / 8;
	const qreal EYE_PY = -(qreal)PIECE_WD / 8;
	qreal cx = (human ? CELL_WD / 16 : -CELL_WD /16);
	QGraphicsEllipseItem *i = 0;
	addCircle(cx - EYE_SPACE, EYE_PY, EYE_SIZE/2, QPen(QColor(Qt::black)), QBrush(Qt::black), p);
	addEyebrow(cx - EYE_SPACE, EYE_PY - 8, ebAngle, p);
	addCircle(cx + EYE_SPACE, EYE_PY, EYE_SIZE/2, QPen(QColor(Qt::black)), QBrush(Qt::black), p);
	addEyebrow(cx + EYE_SPACE, EYE_PY - 8, -ebAngle, p);

	if( !mouth.isEmpty() ) {
		QGraphicsSimpleTextItem *m = new QGraphicsSimpleTextItem(mouth, p);
		m->setPos(cx - m->boundingRect().width() / 2, 0);
		m->setFont(QFont("Arial Black", 10, QFont::Normal));
	}
	return p;
}
QGraphicsItem *MainWindow::findPiece(int point)
{
	int cx = ORG_X + (POINT_START - point) * CELL_WD + CELL_WD / 2;
	int cy = ORG_Y + CELL_HT - CELL_WD / 2;
	return findPiece(cx, cy);
}
QGraphicsItem *MainWindow::findPiece(int cx, int cy)
{
	foreach(QGraphicsItem *item, m_pieces) {
		QRectF br = item->boundingRect();
		QRectF r(br.topLeft() + item->pos(), br.bottomRight() + item->pos());
		if( r.contains(QPointF(cx, cy)) )
			return item;
		//if( item->mapToScene(item->boundingRect()).contains(QPointF(cx, cy)) )
		//	return item;
	}
	return 0;
}
void MainWindow::updateNextItem()
{
	if( m_nextItem != 0 ) {
		m_scene->removeItem(m_nextItem);
		delete m_nextItem;
	}
#if 0
	QFont font("Arial Black", 12, QFont::Normal);
	m_nextItem = m_scene->addSimpleText(tr("next:"), font);
	int py = isBlackTurn() ? PIP_Y : PIP_Y2 - 6;
	m_nextItem->setPos(DICE_X + CELL_WD * 2 + 8, py);
#else
	m_nextItem = addPiece(DICE_X + CELL_WD * 2, DICE_Y - PIECE_MARGIN, isBlackTurn());
#endif
}
MainWindow::~MainWindow()
{

}
//	return:
//		true for ターミネイトおｋ
//		false for ターミネイト不可
bool MainWindow::confirmToTerminate()
{
	if( m_gameState == GS_INIT || m_gameMode != GM_HUMAN_VS_COMP )
		return true;
	if( QMessageBox::question(this, qApp->applicationName(),
								tr("Are you sure to terminate this game ?\n") +
								tr("This game will be regarded as a defeat."),
							QMessageBox::Yes | QMessageBox::No)
			!= QMessageBox::Yes )
	{
		return false;
	}
	return true;
}
void MainWindow::closeEvent(QCloseEvent *event)
{
	writeSettings();
	if( !confirmToTerminate() )
		event->ignore();
	else {
		//writeSettings();
		//QMainWindow::closeEvent(event);
		doExit();
	}
}
void MainWindow::on_action_Exit_triggered()
{
	if( !confirmToTerminate() )
		return;
	//m_gameCountList[m_gameBoardTypeIx] += 1;
	//writeIntList("gameCountList", m_gameCountList);
	doExit();
}
void MainWindow::doExit()
{
	if( m_gameState != GS_INIT && m_gameMode == GM_HUMAN_VS_COMP ) {
		++m_humanStat.m_gameCount;
		++m_compStat[m_compType].m_gameCount;
		++m_compStat[m_compType].m_winCount;
		writeSettings();
	}
	qApp->quit();
}
void MainWindow::keyPressEvent ( QKeyEvent * e )
{
	if( e->key() == Qt::Key_Escape ) {
		m_gameState = GS_INIT;
		return;
	}
}
void MainWindow::readSettings()
{
	QSettings settings;
	restoreGeometry(settings.value("geometry").toByteArray());
	restoreState(settings.value("windowState").toByteArray());
	ui.action_Voice->setChecked(settings.value("voice", true).toBool());
	m_undoAvailable = settings.value("undo", false).toBool();
	m_jumpMove = settings.value("jumpMove", true).toBool();
	m_gameMode = settings.value("gameMode", 0).toInt();
	m_compType = settings.value("compType", 0).toInt();
	m_comp2Type = settings.value("comp2Type", 0).toInt();
	m_autoMoveDst = settings.value("autoMoveDst", false).toBool();
	m_humanStat.m_winCount = settings.value("winCount", 0).toInt();
	m_humanStat.m_gameCount = settings.value("gameCount", 0).toInt();
	m_humanStat.m_rating = settings.value("humanRating", 1500).toInt();
	m_humanStat.m_point = settings.value("humanPoint", 0).toInt();
	m_guestStat.m_winCount = settings.value("guestWinCount", 0).toInt();
	m_guestStat.m_gameCount = settings.value("guestGameCount", 0).toInt();
	m_guestStat.m_rating = settings.value("guestRating", 1500).toInt();
	m_guestStat.m_point = settings.value("guestPoint", 0).toInt();
	for(int i = 0; i < N_LEVEL; ++i) {
		m_compStat[i].m_winCount = settings.value(QString("compWinCount-%1").arg(i+1), 0).toInt();
		m_compStat[i].m_gameCount = settings.value(QString("compGameCount-%1").arg(i+1), 0).toInt();
		m_compStat[i].m_rating = settings.value(QString("compRating-%1").arg(i+1), 1500).toInt();
		m_compStat[i].m_point = settings.value(QString("compPoint-%1").arg(i+1), 0).toInt();
	}
	m_animationSpeed = settings.value("animationSpeed", 2).toInt();
	m_blackColorText = settings.value("blackColor", "greenyellow").toString();
	m_whiteColorText = settings.value("whiteColor", "white").toString();
	m_blackColor = QColor(m_blackColorText);
	m_whiteColor = QColor(m_whiteColorText);
}
void MainWindow::writeSettings()
{
	QSettings settings;
	settings.setValue("geometry", saveGeometry());
	settings.setValue("windowState", saveState());
	settings.setValue("voice", ui.action_Voice->isChecked());
	settings.setValue("gameMode", m_gameMode);
	settings.setValue("compType", m_compType);
	settings.setValue("comp2Type", m_comp2Type);
	settings.setValue("undo", m_undoAvailable);
	settings.setValue("jumpMove", m_jumpMove);
	settings.setValue("autoMoveDst", m_autoMoveDst);
	settings.setValue("winCount", m_humanStat.m_winCount);
	settings.setValue("gameCount", m_humanStat.m_gameCount);
	settings.setValue("humanRating", m_humanStat.m_rating);
	settings.setValue("humanPoint", m_humanStat.m_point);
	settings.setValue("guestWinCount", m_guestStat.m_winCount);
	settings.setValue("guestGameCount", m_guestStat.m_gameCount);
	settings.setValue("guestRating", m_guestStat.m_rating);
	settings.setValue("guestPoint", m_guestStat.m_point);
	for(int i = 0; i < N_LEVEL; ++i) {
		settings.setValue(QString("compWinCount-%1").arg(i+1), m_compStat[i].m_winCount);
		settings.setValue(QString("compGameCount-%1").arg(i+1), m_compStat[i].m_gameCount);
		settings.setValue(QString("compRating-%1").arg(i+1), m_compStat[i].m_rating);
		settings.setValue(QString("compPoint-%1").arg(i+1), m_compStat[i].m_point);
	}
	settings.setValue("animationSpeed", m_animationSpeed);
	settings.setValue("blackColor", m_blackColorText);
	settings.setValue("whiteColor", m_whiteColorText);
}
void MainWindow::onSearchFinished(double eval)
{
	doOutput(QString("eval = %1\n").arg(eval));
	m_compMove = m_treeSearch->m_mvs;
	foreach(Move mv, m_compMove)
		doOutput(QString("  move src = %1 d = %2\n").arg((int)mv.m_src).arg((int)mv.m_d));
	if( m_compMove.isEmpty() ) {	//	パスの場合
#if 0
		{
			double eval;
			int searchType = isBlackTurn() ? m_comp2Type : m_compType;
			if( m_blackTurn )
				Moves mvs = m_board->blackMoves(searchType, m_dice[0], m_dice[1], eval);
			else
				Moves mvs = m_board->whiteMoves(searchType, m_dice[0], m_dice[1], eval);
		}
		//m_treeSearch->doSearch();	//	for Debug
#endif
		if( m_gameMode == GM_HUMAN_VS_COMP ) {
			//QMessageBox::information(this, qApp->applicationName(), tr("I can't move."));
			QMessageBox mb(QMessageBox::NoIcon, qApp->applicationName(), tr("I can't move."));
			mb.exec();
			QList<Moves> mvsList = m_board->whiteMovesList3(m_dice[0], m_dice[1]);	//	for DEBUG
		}
		m_gameState = GS_PASSED;
#if 0
		updateMoveRecord();		//	着手履歴更新
		m_blackTurn = -m_blackTurn;
		++m_turn;
		updateNextItem();
		m_gameState = GS_INIT_ROLLING_DICE;
#if 0
		if( m_gameMode == GM_HUMAN_VS_COMP && isBlackTurn() || m_gameMode == GM_HUMAN_VS_HUMAN )
			m_gameState = GS_INIT_ROLLING_DICE;
		else
			m_gameState = GS_INIT_COMP_TURN;
#endif
#endif
	} else
		m_gameState = GS_COMP_TURN;
}
//	着手履歴更新
void MainWindow::updateMoveRecord()
{
	QString text = QString("Record: d1 = %1 d2 = %2 ").arg(m_curTurn.m_dice1).arg(m_curTurn.m_dice2);
	for(int i = 0; i < 4; ++i) {
		if( !m_curTurn.m_mvs[i].m_src && !m_curTurn.m_mvs[i].m_dst ) continue;
		text += QString("%1/%2 ")
					.arg(m_curTurn.m_mvs[i].m_src)
					.arg(m_curTurn.m_mvs[i].m_dst);
	}
	text += "\n";
	doOutput(text);
	appendToRecord(m_curTurn);
	m_turnRecord << m_curTurn;
	m_curTurn = TurnRecord();
	m_curTurnMvIX = 0;
}
void MainWindow::appendToRecord(const TurnRecord &t)
{
	QString text;
	for(int i = 0; i < 4; ++i) {
		if( !t.m_mvs[i].m_src && !t.m_mvs[i].m_dst ) continue;
		if( !text.isEmpty() ) text += ", ";
		text += QString("%1/%2").arg(t.m_mvs[i].m_src).arg(t.m_mvs[i].m_dst);
	}
	text = QString("%1: %2%3 ")
				.arg(m_recordView->count() + 1, 3)
				.arg(t.m_dice1)
				.arg(t.m_dice2)
			+ text;
	m_recordView->addItem(text);
	m_recordView->setCurrentRow(m_recordView->count() - 1);
}
//	m_dstList の内容が実質ひとつだけか？
bool MainWindow::isForcedMoveDest() const
{
	if( m_dstList.size() == 1 ) return true;
	if( m_dstList.size() != 2 ) return false;
	return m_dstList[0] <= POINT_BLACK_GOAL
			&& m_dstList[1] <= POINT_BLACK_GOAL;
}
void MainWindow::onTimer()
{
	static int count = 0;
	switch( m_gameState ) {
	case GS_PASSED:
		updateMoveRecord();		//	着手履歴更新
		m_blackTurn = -m_blackTurn;
		++m_turn;
		updateNextItem();
		m_gameState = GS_INIT_ROLLING_DICE;
		break;
	case GS_INIT_ROLLING_DICE:
		m_boardHist.clear();
		m_diceHist.clear();
		updateDice();
		count = 0;
		statusBar()->showMessage(tr("click to stop rolling dice."), 5000);
		//doOutput("rolling dice...\n");
		m_rollDiceTime = QTime::currentTime();
		m_gameState = GS_ROLLING_DICE;
		break;
	case GS_ROLLING_DICE:
#if	SAME_DICE_SEQ
		m_dice[0] = 2;
		m_dice[1] = 3;
		m_dice[2] = 0;
		m_dice[3] = 0;
		updateDiceItems();
		statusBar()->showMessage("");
		m_gameState = GS_END_ROLLING_DICE;
#else
		if( ++count < 4 ) break;
		count = 0;
		updateDice();
		if( m_MonteCarlo || m_rollDiceTime.msecsTo(QTime::currentTime()) >= 500 ) {
			statusBar()->showMessage("");
			m_gameState = GS_END_ROLLING_DICE;
		}
#endif
		break;
	case GS_END_ROLLING_DICE:
		m_curTurn.m_dice1 = m_dice[0];
		m_curTurn.m_dice2 = m_dice[1];
		m_curTurnMvIX = 0;
		if( m_gameMode == GM_HUMAN_VS_COMP && isBlackTurn() || m_gameMode == GM_HUMAN_VS_HUMAN )
			m_gameState = GS_INIT_ROLLING_DICE;
		else {
			m_gameState = GS_INIT_COMP_TURN;
			break;
		}
		//m_gameState = GS_INIT_HUMAN_SRC;
		//	下にスルー
	case GS_INIT_HUMAN_SRC:		//	移動石選択待ち状態 初期化
		doOutput(QString("d1 = %1 d2 = %2\n").arg(m_dice[0]).arg(m_dice[1]));
		setupSrcMarks();
		if( !isPossibleToMove() ) {	//	移動可能着手が無い場合
L01:
			updateMoveRecord();		//	着手履歴更新
			if( m_gameMode == GM_HUMAN_VS_COMP || m_gameMode == GM_HUMAN_VS_HUMAN ) {
				//QMessageBox::information(this, qApp->applicationName(), tr("impossible to move."));
				QMessageBox mb(QMessageBox::NoIcon, qApp->applicationName(),
									isBlackTurn() ? tr("You can't move.")
													: tr("Guest can't move."));
				mb.exec();
			}
			m_gameState = GS_INIT_ROLLING_DICE;
			m_blackTurn = -m_blackTurn;
			++m_turn;
			updateNextItem();
			m_gameState = GS_INIT_ROLLING_DICE;
#if 0
			if( m_gameMode == GM_HUMAN_VS_COMP && isBlackTurn() || m_gameMode == GM_HUMAN_VS_HUMAN )
				m_gameState = GS_INIT_ROLLING_DICE;
			else
				m_gameState = GS_INIT_COMP_TURN;
#endif
			return;
		}
		if( isBlackTurn() ) {	//	黒番
			if( m_board->nBlackBar() ) {
				m_srcPoint = POINT_BLACK_START;
				setDstMarks(POINT_BLACK_START);
				m_gameState = GS_HUMAN_DST;
			} else {
				QList<Move> lst = m_board->blackMoves(m_dice[0], m_dice[1]);
				if( lst.isEmpty() )
					goto L01;	//	移動不可能な場合
				m_gameState = GS_HUMAN_SRC;
			}
		} else {	//	白番
			if( m_board->nWhiteBar() ) {
				m_srcPoint = POINT_WHITE_START;
				setDstMarks(POINT_WHITE_START);
				m_gameState = GS_HUMAN_DST;
			} else {
				QList<Move> lst = m_board->whiteMoves(m_dice[0], m_dice[1]);
				if( lst.isEmpty() )
					goto L01;	//	移動不可能な場合
				m_gameState = GS_HUMAN_SRC;
			}
		}
		break;
	case GS_HUMAN_DST:
		if( m_autoMoveDst && isForcedMoveDest() ) {
			m_dstPoint = m_dstList[0];
			Q_ASSERT( m_dstPoint < N_CELL + 2 );
			m_d = m_dList[0];
			m_curTurn.m_mvs[m_curTurnMvIX].m_src = m_srcPoint;
			m_curTurn.m_mvs[m_curTurnMvIX].m_dst = m_dstPoint;
			clearDstMarks();
			clearSrcMarks();
			m_movingHumanPiece = true;
			if( isBlackTurn() && m_dstPoint != POINT_BLACK_GOAL && m_board->white(reverseIX(m_dstPoint)) == 1
				|| !isBlackTurn() && m_dstPoint != POINT_WHITE_GOAL && m_board->black(m_dstPoint) == 1 )
			{
				m_hitItem = findPiece(m_dstPoint);
			} else {
				m_hitItem = 0;
			}
			m_gameState = GS_INIT_MOVING_PIECE;
		}
		break;
	case GS_INIT_COMP_TURN:
		if( 1 )
			updateDice();
		else {
			m_dice[0] = 2;
			m_dice[1] = 3;
			m_dice[2] = 0;
			m_dice[3] = 0;
		}
		updateDiceItems();
		m_initCompTurnTime = QTime::currentTime();		//	コンピュータ思考開始時間
		m_curTurn.m_dice1 = m_dice[0];
		m_curTurn.m_dice2 = m_dice[1];
		m_curTurnMvIX = 0;
		if( !m_dice[0] && !m_dice[1] ) {
			///updateMoveRecord();		//	着手履歴更新
			m_gameState = GS_PASSED;
			break;
		}
		doOutput(QString("d1 = %1, d2 = %2\n").arg(m_dice[0]).arg(m_dice[1]));
		m_treeSearch->setBoard(*m_board);
		m_treeSearch->m_blackTurn = isBlackTurn();
		m_treeSearch->m_searchType = isBlackTurn() ? m_comp2Type : m_compType;
		m_treeSearch->m_d1 = m_dice[0];
		m_treeSearch->m_d2 = m_dice[1];
		QMetaObject::invokeMethod(m_treeSearch, "doSearch", Qt::QueuedConnection);
		m_gameState = GS_COMP_SEARCHING;
		break;
	case GS_COMP_TURN: {
		if( m_compMove.isEmpty() ) {	//	コンピュータ着手終了
			updateMoveRecord();		//	着手履歴更新
			m_blackTurn = -m_blackTurn;
			++m_turn;
			updateNextItem();
			m_gameState = GS_INIT_ROLLING_DICE;
#if 0
			if( m_gameMode == GM_HUMAN_VS_COMP )
				m_gameState = GS_INIT_ROLLING_DICE;
			else
				m_gameState = GS_INIT_COMP_TURN;
#endif
			break;
		}
		Move mv = m_compMove[0];
		removeDice(mv.m_d);	
		//updateDiceItems();		ここでアップデートすると、目がすぐに消えてしまう
		m_compMove.pop_front();
		m_d = mv.m_d;
		m_srcPoint = mv.m_src;
		m_dstPoint = qMax(m_srcPoint - m_d, (int)POINT_BLACK_GOAL);
		if( !isBlackTurn() ) {
			m_srcPoint = reverseIX(m_srcPoint);	//	MainWindow では常に黒石座標系
			m_dstPoint = reverseIX(m_dstPoint);
		}
		Q_ASSERT( m_dstPoint < N_CELL + 2 );
		m_movingHumanPiece = false;
		if( isBlackTurn() && m_dstPoint != POINT_BLACK_GOAL && m_board->white(reverseIX(m_dstPoint)) == 1
			|| !isBlackTurn() && m_dstPoint != POINT_WHITE_GOAL && m_board->black(m_dstPoint) == 1 )
		{
			m_hitItem = findPiece(m_dstPoint);
		} else {
			m_hitItem = 0;
		}
		m_gameState = GS_INIT_MOVING_PIECE;
		break;
	}
	case GS_INIT_MOVING_PIECE:
		if( !m_movingHumanPiece
			&& m_initCompTurnTime.msecsTo(QTime::currentTime()) < 500 )		//	少なくとも0.5秒考える
		{
			return;
		}
		doOutput(QString("%1 moves %2 to %3\n")
					.arg(isBlackTurn() ? "black" : "white")
					.arg(m_srcPoint).arg(m_dstPoint));
		m_curTurn.m_mvs[m_curTurnMvIX].m_src = m_srcPoint;
		m_curTurn.m_mvs[m_curTurnMvIX].m_dst = m_dstPoint;
		indexToXY(m_srcPoint, m_srcPx, m_srcPy);
		m_srcPx += CELL_WD/2;
		if( m_srcPoint == POINT_BLACK_START || m_srcPoint == POINT_WHITE_START )
			m_srcPy = BAR_Y + CELL_WD/2;
		else
			m_srcPy += CELL_HT - CELL_WD/2;
		indexToXY(m_dstPoint, m_dstPx, m_dstPy);
		m_dstPx += CELL_WD/2;
		Q_ASSERT( m_dstPx < ORG_X + CELL_WD * (N_CELL + 2) );
		m_dstPy += CELL_HT - CELL_WD/2;
		m_movingItem = findPiece(m_srcPx, m_srcPy);
		if( m_movingItem == 0 ) {
			//	ここには来ないはず
			m_gameState = GS_INIT;
			return;
		}
		m_movingItem->setZValue(5);
		m_movingProgress = 0;
		m_gameState = GS_MOVING_PIECE;
		break;
	case GS_MOVING_PIECE:
		if( m_movingProgress >= 0 && m_movingItem != 0 ) {
			int px = m_srcPx + (m_dstPx - m_srcPx) * m_movingProgress / 100;
			int py;
			if( m_jumpMove && m_srcPy == m_dstPy ) {
				int ht = (m_movingProgress - 50) * (m_movingProgress - 50);		//	[0, 2500]
				py = m_srcPy - (CELL_WD * 3 - CELL_WD * 3 * ht / 2500);
			} else {
				py = m_srcPy + (m_dstPy - m_srcPy) * m_movingProgress / 100;
			}
			Q_ASSERT( px < ORG_X + CELL_WD * (N_CELL + 2) );
			m_movingItem->setPos(px, py);
			if( m_movingProgress >= 100 ) {		//	移動終了
				m_gameState = GS_MOVING_PIECE_FINISHED;
			} else
				m_movingProgress = qMin(m_movingProgress + m_animationSpeed, 100);
		} else {
			//	ここには来ないはず
			m_gameState = GS_INIT;
			return;
		}
		break;
	case GS_MOVING_PIECE_FINISHED: {
		if( m_movingItem != 0 )
			m_movingItem->setZValue(0);
		if( m_hitItem != 0 ) {
			//m_curTurn.m_mvs[m_curTurnMvIX].m_hit = true;
			//if( m_gameMode == GM_HUMAN_VS_COMP ) {
				if( isBlackTurn() )
					sound("humanHitComp");
				else
					sound("compHitHuman");
			//}
			m_movingItem = m_hitItem;
			m_movingItem->setZValue(5);
			m_hitItem = 0;
			m_movingProgress = 0;
			//m_srcPoint = m_dstPoint;
			m_srcPx = m_dstPx;
			m_srcPy = m_dstPy;
			int dst = isBlackTurn() ? POINT_WHITE_START : POINT_BLACK_START;
			m_dstPx = ORG_X + (IX_START - dst) * CELL_WD + CELL_WD / 2;
			m_dstPy = BAR_Y + CELL_WD / 2;
			m_gameState = GS_MOVING_PIECE;
			break;
		}
		updateDiceItems();
		++m_curTurnMvIX;
		m_movingItem = 0;
		m_movingProgress = -1;
		QString mess;
		int hr = m_humanStat.m_rating;
		m_boardHist << *m_board;
		if( isBlackTurn() ) {
			m_board->moveBlack(m_srcPoint, m_dstPoint);
			if( m_board->doesBlackWin() ) {
				onBlackWin(mess);
			}
		} else {
			m_board->moveWhite(reverseIX(m_srcPoint), reverseIX(m_dstPoint));
			if( m_board->doesWhiteWin() ) {
				onWhiteWin(mess);
			}
		}
		placePieces();
		if( !mess.isEmpty() ) {		//	ゲームオーバ の場合
			onGameOver(mess, hr);
			return;
		}
		if( m_movingHumanPiece ) {
			removeDice(m_d);	
			m_diceHist << (int)m_d;
			updateDiceItems();
			if( !m_dice[0] ) {		//	全てのダイスの目を移動し終わった場合
				updateMoveRecord();		//	着手履歴更新
				if( m_undoAvailable ) {
					statusBar()->showMessage(tr("Click to next turn."));
					m_gameState = GS_WAIT_FOR_DECIDED;
				} else {
					m_blackTurn = -m_blackTurn;
					updateNextItem();
					m_gameState = GS_INIT_ROLLING_DICE;
#if 0
					if( isBlackTurn() || m_gameMode == GM_HUMAN_VS_HUMAN )
						m_gameState = GS_INIT_ROLLING_DICE;
					else
						m_gameState = GS_INIT_COMP_TURN;	//	ダイスを振る
#endif
				}
			} else {
				if( isBlackTurn() || m_gameMode == GM_HUMAN_VS_HUMAN )
					m_gameState = GS_INIT_HUMAN_SRC;
				else
					m_gameState = GS_COMP_TURN;		//	ダイスを振り直さない
			}
		} else {
			m_gameState = GS_COMP_TURN;	//	ダイスを振り直さない
		}
		break;
	}
	case GS_WAIT_FOR_DECIDED:
		break;
	}
}
void MainWindow::onBlackWin(QString &mess, bool resign)
{
	int score = 1;		//	2 for ギャモン勝ち、3 for バックギャモン勝ち
	if( m_gameMode == GM_HUMAN_VS_COMP || m_gameMode == GM_HUMAN_VS_HUMAN ) {
		sound("humanWin");
		++m_humanStat.m_winCount;
		if( !resign && m_board->doesWhiteBackGammonLose() ) {
			mess += tr("You backgammoned Win");
			score = 3;
		} else if( !resign && m_board->doesWhiteGammonLose() ) {
			mess += tr("You gammoned Win");
			score = 2;
		} else
			mess += tr("You Win.");
		if( m_gameMode == GM_HUMAN_VS_COMP ) {
			m_humanStat.m_point += score;
			m_compStat[m_compType].m_point -= score;
			for(int i = 0; i < score; ++i)
				updateRating(m_humanStat.m_rating, m_compStat[m_compType].m_rating);
		} else {
			m_humanStat.m_point += score;
			m_guestStat.m_point -= score;
			for(int i = 0; i < score; ++i)
				updateRating(m_humanStat.m_rating, m_guestStat.m_rating);
		}
	} else {
		sound("gameOver");
		++m_compStat[m_comp2Type].m_winCount;
		if( !resign && m_board->doesWhiteBackGammonLose() ) {
			mess += tr("Computer-2 backgammoned Win");
			score = 3;
		} else if( !resign && m_board->doesWhiteGammonLose() ) {
			mess += tr("Computer-2 gammoned Win");
			score = 2;
		} else
			mess += tr("Computer-2 Win");
		if( m_compType != m_comp2Type ) {
			m_compStat[m_comp2Type].m_point += score;
			m_compStat[m_compType].m_point -= score;
			for(int i = 0; i < score; ++i)
				updateRating(m_compStat[m_comp2Type].m_rating, m_compStat[m_compType].m_rating);
		}
	}
	++m_blackWinInMatch;
	m_blackPointInMatch += score;
}
void MainWindow::onWhiteWin(QString &mess, bool resign)
{
	int score = 1;		//	2 for ギャモン勝ち、3 for バックギャモン勝ち
	if( m_gameMode == GM_HUMAN_VS_COMP ) {
		sound("compWin");
		if( !resign && m_board->doesBlackBackGammonLose() ) {
			mess += tr("I backgammoned Win");
			score = 3;
		} else if( !resign && m_board->doesBlackGammonLose() ) {
			mess += tr("I gammoned Win");
			score = 2;
		} else
			mess += tr("I Win");
		++m_compStat[m_compType].m_winCount;
		m_compStat[m_compType].m_point += score;
		m_humanStat.m_point -= score;
		for(int i = 0; i < score; ++i)
			updateRating(m_compStat[m_compType].m_rating, m_humanStat.m_rating);
	} else if( m_gameMode == GM_HUMAN_VS_HUMAN ) {
		sound("compWin");
		if( !resign && m_board->doesBlackBackGammonLose() ) {
			mess += tr("Guest backgammoned Win");
			score = 3;
		} else if( !resign && m_board->doesBlackGammonLose() ) {
			mess += tr("Guest gammoned Win");
			score = 2;
		} else
			mess += tr("Guest Win");
		++m_guestStat.m_winCount;
		m_guestStat.m_point += score;
		m_humanStat.m_point -= score;
		for(int i = 0; i < score; ++i)
			updateRating(m_guestStat.m_rating, m_humanStat.m_rating);
	} else {	//	COMP VS COMP
		sound("gameOver");
		++m_compStat[m_compType].m_winCount;
		if( !resign && m_board->doesBlackBackGammonLose() ) {
			mess += tr("Computer-1 backgammoned Win");
			score = 3;
		} else if( !resign && m_board->doesBlackGammonLose() ) {
			mess += tr("Computer-1 gammoned Win");
			score = 2;
		} else
			mess += tr("Computer-1 Win");
		if( m_compType != m_comp2Type ) {
			m_compStat[m_compType].m_point += score;
			m_compStat[m_comp2Type].m_point -= score;
			for(int i = 0; i < score; ++i)
				updateRating(m_compStat[m_compType].m_rating, m_compStat[m_comp2Type].m_rating);
		}
	}
	m_whitePointInMatch += score;
}
void MainWindow::onGameOver(QString &mess, int hr)
{
	mess = "<p>" + mess;
	updateMoveRecord();		//	着手履歴更新
	m_turn = 0;
	mess += "\n\n";
	//mess += "<p>";
	//mess += tr("Points and win/game in this Match:\n");
	mess += "<br><table>\n";
	mess += "<tr><th></th><th>Points</th><th>win/game</th></tr>\n";
	mess += "<tr><td align=right>";
	if( m_gameMode == GM_HUMAN_VS_COMP || m_gameMode == GM_HUMAN_VS_HUMAN )
		mess += tr("You:");
	else
		mess += tr("Comp-2:");
	mess += QString("</td><td>%1</td><td>(%2/%3 = %4%)</td></tr>\n")
				.arg(m_blackPointInMatch)
				.arg(m_blackWinInMatch)
				.arg(m_gameInMatch)
				.arg((m_blackWinInMatch * 1000/m_gameInMatch) / 10.0);
	mess += "<tr><td align=right>";
	if( m_gameMode == GM_HUMAN_VS_COMP || m_gameMode == GM_COMP_VS_COMP )
		mess += tr("Comp-1:");
	else
		mess += tr("Guest:");
	mess += QString("</td><td>%1</td><td>(%2/%3 = %4%)</td></tr>\n")
				.arg(m_whitePointInMatch)
				.arg(m_gameInMatch - m_blackWinInMatch)
				.arg(m_gameInMatch)
				.arg(((m_gameInMatch - m_blackWinInMatch) * 1000/m_gameInMatch) / 10.0);
	mess += "</table>\n";
	if( m_gameMode == GM_HUMAN_VS_COMP || m_gameMode == GM_HUMAN_VS_HUMAN ) {
		++m_humanStat.m_gameCount;
		//if( m_gameMode == GM_HUMAN_VS_HUMAN )
		//	++m_guestStat.m_gameCount;
		//else
		//	++m_compStat[m_compType].m_gameCount;
		mess += "\n\n";
		mess += "<p>";
		mess += tr("Your Rating = %1 (%2%3 added)\n")
					.arg(m_humanStat.m_rating)
					.arg(m_humanStat.m_rating - hr > 0 ? "+" : "")
					.arg(m_humanStat.m_rating - hr);
		mess += "<br>";
		mess += tr("total Points = %1\n").arg(m_humanStat.m_point);
		mess += "<br>";
		mess += tr("record = %1/%2 (%3%)")
					.arg(m_humanStat.m_winCount)
					.arg(m_humanStat.m_gameCount)
					.arg((double)m_humanStat.m_winCount*100/m_humanStat.m_gameCount);
		mess += "\n\n";
		if( m_gameMode == GM_HUMAN_VS_HUMAN ) {
			++m_guestStat.m_gameCount;
			mess += "<p>";
			mess += tr("opponent Rating = %1\n").arg(m_guestStat.m_rating);
			mess += "<br>";
			mess += tr("    total Points = %1").arg(m_guestStat.m_point);
		} else {
			++m_compStat[m_compType].m_gameCount;
			mess += "<p>";
			mess += tr("opponent Rating = %1\n").arg(m_compStat[m_compType].m_rating);
			mess += "<br>";
			mess += tr("    total Points = %1").arg(m_compStat[m_compType].m_point);
		}
	} else {
		++m_compStat[m_compType].m_gameCount;
		++m_compStat[m_comp2Type].m_gameCount;
		if( m_compType != m_comp2Type ) {
			mess += "\n\n";
			mess += "<p>";
			mess += tr("Comp-2 Rating = %1 total Points = %2\n")
						.arg(m_compStat[m_comp2Type].m_rating)
						.arg(m_compStat[m_comp2Type].m_point);
			mess += "<br>";
			mess += tr("Comp-1 Rating = %1 total Points = %2\n")
						.arg(m_compStat[m_compType].m_rating)
						.arg(m_compStat[m_compType].m_point);
		}
	}
	mess += "\n\n";
	mess += "<p>";
	mess += tr("one more the same settings game ?");
	writeSettings();
#if 0
	QMessageBox aDlg(QMessageBox::NoIcon, qApp->applicationName(), mess,
						QMessageBox::Yes | QMessageBox::No);
	if( aDlg.exec() == QMessageBox::Yes )
#endif
	m_gameState = GS_INIT;
	GameAgainDlg aDlg;
	connect(m_actDialog = &aDlg, SIGNAL(save()), this, SLOT(on_action_SaveRecord_triggered()));
	aDlg.setMessage(mess);
	if( aDlg.exec() == QDialog::Accepted )
	{
		clearPieces();
		newGame();
	}
	m_actDialog = 0;
}
void MainWindow::mousePressEvent ( QMouseEvent * event )
{
	QPoint pntG = m_view->mapFromGlobal(event->globalPos());
	QPointF pnt = m_view->mapToScene(pntG);
	int ix = xyToIndex(pnt.x(), pnt.y());
	int i;
	switch( m_gameState ) {
	case GS_WAIT_FOR_DECIDED:
		m_blackTurn = -m_blackTurn;
		updateNextItem();
		m_gameState = GS_INIT_ROLLING_DICE;
#if 0
		if( isBlackTurn() )
			m_gameState = GS_INIT_ROLLING_DICE;
		else
			m_gameState = GS_INIT_COMP_TURN;	//	ダイスを振る
#endif
		break;
	case GS_ROLLING_DICE:
		statusBar()->showMessage("");
		m_gameState = GS_END_ROLLING_DICE;
		break;
	case GS_HUMAN_SRC:		//	移動する石選択待ち状態
		if( ix < POINT_BLACK_START && ix > POINT_GOAL &&		//	バーに石がある場合は、移動する石は既に選択されている
			m_board->position(ix) * m_blackTurn > 0 )
		{
			if( !setDstMarks(ix) )
				break;		//	移動不可能な場合
			m_srcPoint = ix;
			m_gameState = GS_HUMAN_DST;
		}
		break;
	case GS_HUMAN_DST:
		if( (i = m_dstList.indexOf(ix)) >= 0 ) {
			m_dstPoint = ix;
			Q_ASSERT( m_dstPoint < N_CELL + 2 );
			m_d = m_dList[i];
			clearDstMarks();
			clearSrcMarks();
			m_movingHumanPiece = true;
			if( isBlackTurn() && m_dstPoint != POINT_BLACK_GOAL && m_board->white(reverseIX(m_dstPoint)) == 1
				|| !isBlackTurn() && m_dstPoint != POINT_WHITE_GOAL && m_board->black(m_dstPoint) == 1 )
			{
				m_hitItem = findPiece(m_dstPoint);
			} else {
				m_hitItem = 0;
			}
			m_gameState = GS_INIT_MOVING_PIECE;
		} else {
			clearDstMarks();
			clearSrcMarks();
			m_gameState = GS_INIT_HUMAN_SRC;
		}
		break;
	}
	//doOutput(QString("black = 0x%1 (%2), white = 0x%3 (%4)\n")
	//						.arg(m_board->black(), 0, 16).arg(m_board->nBlackBar())
	//						.arg(m_board->white(), 0, 16).arg(m_board->nWhiteBar()));
}
//	r1 が勝った場合のレーティング更新
void MainWindow::updateRating(int &r1, int &r2)
{
	int d = (int)(16 + (r2 - r1) * 0.04 + 0.5);
	d = qMax(qMin(d, 31), 1);
	r1 += d;
	r2 -= d;
}
void MainWindow::updateDice()
{
	//	サイコロの目をランダムに変化させる
	const bool noDoublet = m_turn == 1 && !m_MonteCarlo;
	do {
		QString::number(m_dice[0] = (qrand() % N_DICE));
		QString::number(m_dice[1] = (qrand() % N_DICE));
	} while( noDoublet && m_dice[0] == m_dice[1] || !m_dice[0] && !m_dice[1] );
	if( m_dice[0] == m_dice[1] )
		m_dice[3] = m_dice[2] = m_dice[0];
	else
		m_dice[3] = m_dice[2] = 0;
	updateDiceItems();
}
void MainWindow::updateDiceItems()
{
	for(int i = 0; i < 2; ++i)
		m_diceItems[i]->setText( m_dice[i] != 0 ? QString::number(m_dice[i]) : "");
	m_diceRect1->setPen(QPen(m_dice[0] != m_dice[2] ? Qt::black : Qt::red));
	m_diceRect2->setPen(QPen(!m_dice[1] || m_dice[1] != m_dice[3] ? Qt::black : Qt::red));
}
//	index は黒インデクスとする
void MainWindow::indexToXY(int ix, int &px, int &py) const
{
	Q_ASSERT( ix >= 0 && ix < N_CELL + 2 );
	px = ORG_X + reverseIX(ix) * CELL_WD;
	py = ORG_Y;
}
int MainWindow::xyToIndex(int px, int py) const
{
	if( px < ORG_X || px >= ORG_X + CELL_WD * (N_CELL + 2) ||
		py < ORG_Y || py >= ORG_Y + CELL_HT )
	{
		return -1;
	}
	return reverseIX((px - ORG_X) / CELL_WD);
}
//	移動可能で無い場合は false を返す
bool MainWindow::isPossibleToMove()
{
	if( isBlackTurn() ) {
		QList<Move> lst = m_board->blackMoves(m_dice[0], m_dice[1]);
		if( lst.isEmpty() ) {
			//qDebug() << m_board->position();
			//lst = m_board->blackMoves(m_dice[0], m_dice[1]);	//	for Debug
			return false;
		}
		//if( !m_board->canBlackMove(m_dice[0]) &&
		//		m_dice[1] != 0 && !m_board->canBlackMove(m_dice[1]) )
		//	return false;
		if( m_board->nBlackBar() != 0 ) {
			m_srcPoint = POINT_BLACK_START;
			setDstMarks(POINT_BLACK_START);
			m_gameState = GS_HUMAN_DST;
		}
	} else {
		QList<Move> lst = m_board->whiteMoves(m_dice[0], m_dice[1]);
		if( lst.isEmpty() )
			return false;
		//if( !m_board->canWhiteMove(m_dice[0]) &&
		//		m_dice[1] != 0 && !m_board->canWhiteMove(m_dice[1]) )
		//	return false;
		if( m_board->nWhiteBar() != 0 ) {
			m_srcPoint = POINT_WHITE_START;
			setDstMarks(POINT_WHITE_START);
			m_gameState = GS_HUMAN_DST;
		}
	}
	return true;
}
void MainWindow::addDice(int d)
{
	for(int ix = 0; ix < 4; ++ix) {
		if( !m_dice[ix] ) {
			m_dice[ix] = d;
			return;
		}
	}
}
void MainWindow::removeDice(int d)
{
	int ix = 0;
	while( ix < 4 && m_dice[ix] != d ) ++ix;
	while( ix < 3 ) {
		m_dice[ix] = m_dice[ix + 1];
		++ix;
	}
	m_dice[3] = 0;
}
void MainWindow::clearSrcMarks()
{
	foreach(QGraphicsRectItem *item, m_srcMarks) {
		m_scene->removeItem(item);
		delete item;
	}
	m_srcMarks.clear();
}
void MainWindow::clearDstMarks()
{
	foreach(QGraphicsRectItem *item, m_dstMarks) {
		m_scene->removeItem(item);
		delete item;
	}
	m_dstMarks.clear();
	m_dstList.clear();
	m_dList.clear();
}
//	src == 0 の場合は、黒のバーからの移動
//	src == POINT_WHITE_START  の場合は、白のバーからの移動
bool MainWindow::canMove(int src, int dst) const
{
	if( isBlackTurn() )
		return m_board->canBlackMoveSrcDst(src, dst);
	else
		return m_board->canWhiteMoveSrcDst(reverseIX(src), reverseIX(dst));

}
void MainWindow::setupSrcMarks()
{
	clearSrcMarks();
	if( isBlackTurn() ) {
		if( m_board->black(IX_START) != 0 ) return;	//	バーに石がある場合
		for(int src = IX_START - 1; src > IX_GOAL; --src) {
			if( m_board->black(src) != 0 &&
				(m_dice[0] != 0 && canMove(src, src - m_dice[0]) ||
				m_dice[1] != 0 && canMove(src, src - m_dice[1])) )
			{
				setSrcMarks(src, false);
			}
		}
	} else {
		if( m_board->white(IX_START) != 0 ) return;	//	バーに石がある場合
		for(int src = IX_START - 1; src > IX_GOAL; --src) {
			if( m_board->white(src) != 0 &&
				(m_dice[0] != 0 && canMove(reverseIX(src), reverseIX(src - m_dice[0])) ||
				m_dice[1] != 0 && canMove(reverseIX(src), reverseIX(src - m_dice[1]))) )
			{
				setSrcMarks(reverseIX(src), false);
			}
		}
	}
}
void MainWindow::setSrcMarks(int src, bool bClear)
{
	if( bClear )
		clearSrcMarks();
	int px, py;
	indexToXY(src, px, py);
	if( src == POINT_BLACK_START || src == POINT_WHITE_START )
		py -= CELL_WD;
	else
		py = PIECE_Y;
	QGraphicsRectItem *ri;
	//	左上
	ri = m_scene->addRect(	px + SEL_MARK_MARGIN,
							py + SEL_MARK_MARGIN,
							SEL_MARK_WD,
							SEL_MARK_LENGTH,
							QPen(Qt::red),
							QBrush(QColor("red")));
	m_srcMarks.push_back(ri);
	ri = m_scene->addRect(	px + SEL_MARK_MARGIN,
							py + SEL_MARK_MARGIN,
							SEL_MARK_LENGTH,
							SEL_MARK_WD,
							QPen(Qt::red),
							QBrush(QColor("red")));
	m_srcMarks.push_back(ri);
	//	左下
	ri = m_scene->addRect(	px + SEL_MARK_MARGIN,
							py + CELL_WD - SEL_MARK_MARGIN - SEL_MARK_WD,
							SEL_MARK_LENGTH,
							SEL_MARK_WD,
							QPen(Qt::red),
							QBrush(QColor("red")));
	m_srcMarks.push_back(ri);
	ri = m_scene->addRect(	px + SEL_MARK_MARGIN,
							py + CELL_WD - SEL_MARK_MARGIN - SEL_MARK_LENGTH,
							SEL_MARK_WD,
							SEL_MARK_LENGTH,
							QPen(Qt::red),
							QBrush(QColor("red")));
	m_srcMarks.push_back(ri);
	//	右上
	ri = m_scene->addRect(	px + CELL_WD - SEL_MARK_MARGIN - SEL_MARK_WD,
							py + SEL_MARK_MARGIN,
							SEL_MARK_WD,
							SEL_MARK_LENGTH,
							QPen(Qt::red),
							QBrush(QColor("red")));
	m_srcMarks.push_back(ri);
	ri = m_scene->addRect(	px + CELL_WD - SEL_MARK_LENGTH - SEL_MARK_MARGIN,
							py + SEL_MARK_MARGIN,
							SEL_MARK_LENGTH,
							SEL_MARK_WD,
							QPen(Qt::red),
							QBrush(QColor("red")));
	m_srcMarks.push_back(ri);
	//	右下
	ri = m_scene->addRect(	px + CELL_WD - SEL_MARK_MARGIN - SEL_MARK_WD,
							py + CELL_WD - SEL_MARK_MARGIN - SEL_MARK_LENGTH,
							SEL_MARK_WD,
							SEL_MARK_LENGTH,
							QPen(Qt::red),
							QBrush(QColor("red")));
	m_srcMarks.push_back(ri);
	ri = m_scene->addRect(	px + CELL_WD - SEL_MARK_LENGTH - SEL_MARK_MARGIN,
							py + CELL_WD - SEL_MARK_MARGIN - SEL_MARK_WD,
							SEL_MARK_LENGTH,
							SEL_MARK_WD,
							QPen(Qt::red),
							QBrush(QColor("red")));
	m_srcMarks.push_back(ri);
}
bool MainWindow::setDstMarks(int src)
{
	//m_dstList.clear();
	clearDstMarks();
	int px, py;
	bool bCanMove = false;
	//const bool canBearOff = isBlackTurn() ? m_board->canBlackBearOff()
	//											: m_board->canWhiteBearOff();
	for(int i = 0; i < 2; ++i) {
		int dst = src - m_dice[i] * m_blackTurn;
		if( m_dice[i] != 0 && canMove(src, dst) ) {
			if( isBlackTurn() )
				dst = qMax(dst, (int)POINT_GOAL);
			else
				dst = qMin(dst, (int)POINT_START);
			indexToXY(dst, px, py);
			py = PIECE_Y;
			QGraphicsRectItem *ri = m_scene->addRect(
									px + DST_MARK_MARGIN,
									py + DST_MARK_MARGIN,
									DST_MARK_WD,
									DST_MARK_WD,
									QPen(Qt::gray),
									QBrush(QColor("pink")));
			ri->setZValue(10);
			m_dstMarks.push_back(ri);
			m_dstList << dst;
			m_dList << m_dice[i];
			bCanMove = true;
		}
		if( m_dice[0] == m_dice[1] ) break;
	}
	if( !bCanMove ) return false;
	setSrcMarks(src);
	return true;
}
void MainWindow::on_action_Settings_triggered()
{
	///if( m_gameState != GS_INIT ) return;
	SettingsDlg aDlg;
	aDlg.setAnimationSpeed(m_animationSpeed);
	aDlg.setHumanColor(m_blackColorText);
	aDlg.setCompColor(m_whiteColorText);
	aDlg.setAutoMoveDst(m_autoMoveDst);
	aDlg.setUndoAvailable(m_undoAvailable);
	aDlg.setJumpMove(m_jumpMove);
	aDlg.setVoice(ui.action_Voice->isChecked());
	if( aDlg.exec() == QDialog::Accepted ) {
		m_animationSpeed = aDlg.animationSpeed();
		m_blackColorText = aDlg.humanColor();
		m_whiteColorText = aDlg.compColor();
		m_autoMoveDst = aDlg.autoMoveDst();
		m_undoAvailable = aDlg.undoAvailable();
		m_jumpMove = aDlg.jumpMove();
		ui.action_Voice->setChecked(aDlg.voice());
		m_blackPipsItem->setBrush(QBrush(m_blackColor = QColor(m_blackColorText)));
		m_whitePipsItem->setBrush(QBrush(m_whiteColor = QColor(m_whiteColorText)));
		writeSettings();
		placePieces();
	}
}
QStringList statRowStringList(const QString &name, const Statistics &stat)
{
	QStringList lst;
	lst << name;
	lst << QString::number(stat.m_rating);
	lst << QString::number(stat.m_point);
	lst << QString::number(stat.m_winCount);
	lst << QString::number(stat.m_gameCount);
	lst << QString::number(!stat.m_gameCount ? 0 : (stat.m_winCount*1000/stat.m_gameCount)/10.0);
	//lst << QString("%1/%2 (%3%)")
	//		.arg(stat.m_winCount)
	//		.arg(stat.m_gameCount)
	//		.arg(!stat.m_gameCount ? 0 : (stat.m_winCount*1000/stat.m_gameCount)/10.0);
	return lst;
}
void MainWindow::statisticsTextList(QList<QStringList> &lst)
{
	lst.clear();
	lst << statRowStringList(tr("You"), m_humanStat);
	lst << statRowStringList(tr("Guest"), m_guestStat);
	lst << statRowStringList(tr("Mr. Advanced"), m_compStat[3]);
	lst << statRowStringList(tr("Mr. Inermediate"), m_compStat[2]);
	lst << statRowStringList(tr("Mr. Beginner"), m_compStat[1]);
	lst << statRowStringList(tr("Mr. Random"), m_compStat[0]);
}
QString statRow(const QString &name, const Statistics &stat)
{
	return QString("<tr><td>%1</td><td><b>%2</b></td><td>%3/%4 (%5%)</td></tr>\n")
		.arg(name)
		.arg(stat.m_rating)
		.arg(stat.m_winCount)
		.arg(stat.m_gameCount)
		.arg(!stat.m_gameCount ? 0 : (stat.m_winCount*1000/stat.m_gameCount)/10.0);
}
void MainWindow::statisticsText(QString &text)
{
	text = "<table>\n";
	text += "<tr><th></th><th>Rating</th><th>win/game (percent)</th></tr>\n";
	text += statRow(tr("You"), m_humanStat);
	text += statRow(tr("Mr. Advanced"), m_compStat[3]);
	text += statRow(tr("Mr. Inermediate"), m_compStat[2]);
	text += statRow(tr("Mr. Beginner"), m_compStat[1]);
	text += statRow(tr("Mr. Random"), m_compStat[0]);
	text += "</table>";
}
void MainWindow::on_action_Statistics_triggered()
{
	//QString mess;
	//statisticsText(mess);
	//QMessageBox::information(this, tr("Statistics"), mess);
	StatisticsDlg aDlg;
	//aDlg.setText(mess);
	connect(&aDlg, SIGNAL(statisticsText(QString &)), this, SLOT(statisticsText(QString &)));
	connect(&aDlg, SIGNAL(statisticsTextList(QList<QStringList> &)), this, SLOT(statisticsTextList(QList<QStringList> &)));
	connect(&aDlg, SIGNAL(clearStatistics()), this, SLOT(clearStatistics()));
	aDlg.updateText();
	aDlg.exec();
}
void MainWindow::on_action_ClearStatistics_triggered()
{
	if( QMessageBox::question(this, qApp->applicationName(), tr("Are You Sure to Clear Statistics ?"),
								QMessageBox::Yes | QMessageBox::No)
				== QMessageBox::Yes )
	{
		clearStatistics();
	}
}
void MainWindow::clearStatistics()
{
	m_humanStat = Statistics();
	m_guestStat = Statistics();
	for(int i = 0; i < N_LEVEL; ++i)
		m_compStat[i] = Statistics();
	writeSettings();
}
//void MainWindow::on_action_Help_triggered()
void MainWindow::on_action_Rule_triggered()
{
	QString url = "http://vivi.dyndns.org/games/HalfGammon/help.html?from=";
	url += VERSION_STR;
	QDesktopServices::openUrl(QUrl(url));
}
void MainWindow::on_action_HowToPlay_triggered()
{
	QString url = "http://vivi.dyndns.org/games/HalfGammon/howToPlay.html?from=";
	url += VERSION_STR;
	QDesktopServices::openUrl(QUrl(url));
}
void MainWindow::on_action_AboutHalfGammon_triggered()
{
	QMessageBox msgBox;
	msgBox.setIconPixmap(QPixmap(":MainWindow/Resources/ayabu.png"));
	msgBox.setText(tr("<div align=center><b>HalfGammon</b> version %1</div>").arg(VERSION_STR));
#ifdef	Q_CC_MSVC
	const QString ayabu((const QChar *)L"桃色 椿");
#else
	const QString ayabu("momoshiki tsubaki");
#endif
	msgBox.setInformativeText(QString("<div align=center>Copyright (C) 2012 by N.Tsuda<br>"
								"mailto:ntsuda@master.email.ne.jp<br>"
								"<a href=\"http://vivi.dyndns.org/?from=qvi%1\">http://vivi.dyndns.org/</a><br><br>"
								"Powered by <a href=\"http://qt.nokia.com/\">Qt</a>"
								"<p>illustrated by <a href=\"http://www.pixiv.net/member.php?id=220294\">%2</a>"
								"<p>icons by <a href=\"http://www.visualpharm.com/\">VisualPharm</a></div>")
								.arg(VERSION_STR).arg(ayabu));
	msgBox.exec();
}
void MainWindow::on_action_LoadRecord_triggered()
{
	if( m_gameState != GS_INIT ) return;
	QString fileName = QFileDialog::getOpenFileName();
	QFile file(fileName);
	if( !file.open(QIODevice::ReadOnly) ) {
		QMessageBox::warning(this, qApp->applicationName(), tr("'%1' can't open.").arg(fileName));
		return;
	}
	QTextCodec *codec = QTextCodec::codecForName("UTF-8");
	QStringList lst = codec->toUnicode(file.readAll()).split(QChar('\n'));
	if( !lst[0].startsWith(qApp->applicationName() + "Record File Rev ") ) {
		QMessageBox::warning(this, qApp->applicationName(), tr("'%1' isn't a Record file.").arg(fileName));
		return;
	}
	m_gameState = GS_RECORD_LOADING;
	m_turnRecord.clear();
	m_recordView->clear();
	lst.pop_front();
	if( lst[0].startsWith("firstMove = ") ) {
		m_firstMoveBlack = lst[0][QString("firstMove = ").size()] == 'b';
		lst.pop_front();
	}
	QRegExp exp("^(\\d+): (\\d)-(\\d) ");
	QRegExp mvExp("(\\d+) to (\\d+)");
	foreach(const QString &buffer, lst) {
		int ix = exp.indexIn(buffer);
		if( ix != 0 ) continue;
		TurnRecord rec;
		QStringList t = exp.capturedTexts();
		rec.m_dice1 = t[2].toInt();
		rec.m_dice2 = t[3].toInt();
		const QStringList mvs = buffer.mid(exp.matchedLength()).split(QChar(','));
		int k = 0;
		foreach(const QString m, mvs) {
			if( (ix = mvExp.indexIn(m)) < 0 ) continue;
			QStringList t = mvExp.capturedTexts();
			rec.m_mvs[k].m_src = t[1].toInt();
			rec.m_mvs[k].m_dst = t[2].toInt();
			++k;
		}
		m_turnRecord << rec;
		appendToRecord(rec);
	}
	m_gameState = GS_INIT;
	if( !m_turnRecord.isEmpty() ) {
		m_recordView->setFocus(Qt::OtherFocusReason);
		m_recordView->setCurrentRow(0);
	}
}
void MainWindow::on_action_SaveRecord_triggered()
{
	if( m_gameState != GS_INIT ) return;
	if( m_turnRecord.isEmpty() ) {
		QMessageBox::warning(this, qApp->applicationName(), tr("No record to save."));
		if( m_actDialog ) m_actDialog->activateWindow();
		return;
	}
	QString defName = qApp->applicationName().left(1) + "G-" + QDateTime::currentDateTime().toString("yyyyMMddhhmmss") + ".qtg";
	QString fileName = QFileDialog::getSaveFileName(this, tr("SaveRecord"), defName);
	if( fileName.isEmpty() ) return;
	QFile file(fileName);
	if( !file.open(QIODevice::WriteOnly) ) {
		QMessageBox::warning(this, qApp->applicationName(), tr("'%1' can't open.").arg(fileName));
		return;
	}
	QString buffer = qApp->applicationName() + "Record File Rev 1.0\n";
	buffer += QString("firstMove = %1\n").arg(m_firstMoveBlack ? "black" : "white");
	int turn = 0;
	foreach(TurnRecord rec, m_turnRecord) {
		buffer += QString("%1: %2-%3 ").arg(++turn).arg(rec.m_dice1).arg(rec.m_dice2);
		QString text;
		for(int k = 0; k < 4; ++k) {
			const int src = rec.m_mvs[k].m_src;
			const int dst = rec.m_mvs[k].m_dst;
			if( !src && !dst ) break;
			if( !text.isEmpty() ) text += ", ";
			text += QString("%1 to %2").arg(src).arg(dst);
		}
		buffer += text;
		buffer += "\n";
	}
	QTextCodec *codec = QTextCodec::codecForName("UTF-8");
	file.write(codec->fromUnicode(buffer));
	if( m_actDialog ) m_actDialog->activateWindow();
}
void MainWindow::on_action_Start_triggered()
{
	m_recordView->setCurrentRow(0);
}
void MainWindow::on_action_End_triggered()
{
	if( m_recordView->count() != 0 )
		m_recordView->setCurrentRow(m_recordView->count() - 1);
}
void MainWindow::on_action_Backward_triggered()
{
	int row = m_recordView->currentRow();
	if( row != 0 )
		m_recordView->setCurrentRow(row - 1);
}
void MainWindow::on_action_Forward_triggered()
{
	int row = m_recordView->currentRow();
	if( row + 1 < m_recordView->count() )
		m_recordView->setCurrentRow(row + 1);
}
