//
// pixq_Matrix
// 2D matrix class
//
#pragma once

namespace _pix_plot_qt_framework {

template <class T> class pixq_Matrix
{
public:
	pixq_Matrix( void );
	~pixq_Matrix( void );

public:
   // how do you want to fill the frame area of the matrix?
   enum {
      // fill the frame area with zero
      MatrixFillFrameZero = 0,
      // fill the frame area with given constant value
      MatrixFillFrameConst,
      // fill with the last value on the edge for the all row and color outside
      MatrixFillFrameClamp,
      // return the matyrix body at the edge like mirror
      MatrixFillFrameMirror,
      // repeat the same pattern around the matrix body
      MatrixFillFrameRepeat,
   } MatrixFillFrameMethod;

private:
   // matrix height
	int _iNrows;
   
   // matrix width
	int _iNcols; 

   // height of the frame area
	int _iFrameRows;
   
   // width of the frame area
	int _iFrameCols; 

   // height of the matrix including the frame area
	int _iWholeRows;
   
   // width of the matrix including the frame area
	int _iWholeCols; 

	// size of the buffer, including frame area
   int _iSize; 

   // does this matrix have frame area?
   bool _bFrame;
   int _iFunc;

   // matrix real body

	T **_Data;

   // pointer to top of each row

   T **_pDataRow;

   // value to set to indicate the point has no data.
	T _Null;  

   // value to set to indicate the point has no data.
	bool _bNullSet;  

   // *** statistics ***
   // true if the statistic values are set below
	bool _bStatSet;
	T _vMin;
	T _vMax;
	T _vMean;
	T _vstDev;

public:
	int GetWidth( ) const { return _iNcols; }
	int GetHeight( ) const { return _iNrows; }
   
	int GetFrameWidth( ) const { return _iFrameCols; }
	int GetFrameHeight( ) const { return _iFrameRows; }

   bool bGetStatSet( ) { return _bStatSet; }
   T GetMin( ) { return _vMin; }
   T GetMax( ) { return _vMax; }
   T GetMean( ) { return _vMean; }
   T GetStDev( ) { return _vstDev; }

   void SetNull( T NullValue ) { 
      _bNullSet = true; 
      _Null = NullValue;
   }

   void UnsetNull( void ) { _bNullSet = false; }

public:
   // allocate matrix
	bool Alloc( int nRows, int nCols );
	bool Alloc( int nRows, int nCols, int nFrameRows, int nFrameCols );
	bool CheckAlloc( int nRows, int nCols );

   // release matrix
   void Free( void );
	
   // *** data handling ***
   // pixq_MatrixData.h

	// set the value at given position
   void Set( int row, int col, T Value );

   T GetFrame( int row, int col ) const;
   T GetNoFrame( int row, int col ) const;
   
   T Get( int row, int col ) const;
   //T (pixq_Matrix::*Get)( int row, int col );

   typedef T (pixq_Matrix::*GetFunc)( int row, int col ) const;

   GetFunc funcArray[2];
 
   //T (*Get)Get( double dX, double dY );

   // fill the matrix with a given constant value
   void Fill( T Value );

   // get a row of data from the matrix into a one-dimensinal buffer
   bool GetRow( int iRow, T *tBuff ) const; 
   bool GetRow( int iRow, vector<T> *tRow ) const; 
   
   // set a data into the row in the matrix from a one-dimensinal buffer
   bool SetRow( int iRow, T *tBuff ); 

   // get a data from the whole matrix into a one-dimensinal buffer
   bool GetData( T *tBuff ); 

   // set a data into the whole matrix from a one-dimensinal buffer
   bool SetData( T *tBuff ); 

   // get a range of fata from a specified row of the matrix into a one-dimensinal buffer
   bool GetLine( int iRow, int Col0, int iCol1, T *tBuff ) const; 
   bool GetLine( int iRow, int Col0, int iCol1, vector<T> *tRow ) const; 

   // get a rectangle area of data from a specified area of the matrix 
   // into a one-dimensinal buffer
   bool GetPatch( int iRow0, int iRow1, int Col0, int iCol1, T *tBuff ) const; 
   bool GetPatch( int iRow0, int iRow1, int Col0, int iCol1, vector<T> *tPatch ) const; 

   bool GetPatch( int iCol0, int iRow0, int iCol1, int iRow1, pixq_Matrix<T> *pPatch ); 
   bool GetPatch( int iCol0, int iRow0, int iCol1, int iRow1, 
      bool bFillOut, T tFill, pixq_Matrix<T> *pPatch );

   // *** utility functions ***
   // pixq_MatrixUt.h
      
   // fill the frame area around the matrix body
   void FillFrame( short sMethod, T tConst = (T)0 );
   void FillConstFrame( T tConst );
   void FillClampFrame( void );

   bool Copy( pixq_Matrix<T> *pSrc );

   // absolute difference, mtx0 - mtx1
   bool AbsoluteDiff( pixq_Matrix<T> *pMtx0, pixq_Matrix<T> *pMtx1 );

   // check the validity of the row number
   bool CheckRow( int iRow ) const;
   
   // check the validity of the column number
   bool CheckCol( int iCol ) const;

   // check the validity of the row range
   bool CheckRowRange( int iRow0, int iRow1 ) const;
   
   // check the validity of the column range
   bool CheckColRange( int iCol0, int iCol1 ) const;

   // check if the point(row/col) is inside the matrix range
   bool IsInside( int iX, int iY );
   bool IsInside( float fX, float fY );
   bool IsInside( double dX, double dY );

   // *** matrix analysis ***

   void Statistics( T *vmin, T *vmax, bool bRecalc = false );
   void Statistics( T *vmin, T *vmax, T *mean, bool bRecalc = false );
   void Statistics( T *vmin, T *vmax, T *mean, T *stdev, bool bRecalc = false );
   
   void Statistics( T *vmin, T *vmax,
      const pixq_Matrix<bool> *pMask,
      bool bRecalc = false );
   void Statistics( T *vmin, T *vmax, T *mean,
      const pixq_Matrix<bool> *pMask,
      bool bRecalc = false );
   void Statistics( T *vmin, T *vmax, T *mean, T *stdev,
      const pixq_Matrix<bool> *pMask,
      bool bRecalc = false );

   void Statistics( T *vmin, T *vmax, T *mean ) const;
   void Statistics( T *vmin, T *vmax, T *mean, T *stdev ) const;

   void Statistics( T *vmin, T *vmax, T *mean,
      const pixq_Matrix<bool> *pMask ) const;
   void Statistics( T *vmin, T *vmax, T *mean, T *stdev,
      const pixq_Matrix<bool> *pMask ) const;

   // *** linear algebra ***

   // get determinant of 3X3 matrix
   bool GetDeterminant3X3( T *tDet );

   // get inrverse matrix of of 3X3 matrix
   bool GetInverse3x3( pixq_Matrix<T> *Inv );

   // get inrverse matrix of of NXN matrix
   bool GetInverse( pixq_Matrix<T> *Inv );

   // create new matrix as a product of two square amtrix
   bool ProductSquare( pixq_Matrix<T> *pMtx0, pixq_Matrix<T> *pMtx1 );

   //
   // *** please do not use these functions Yet ***
   // *** i am currently adding some changes to these and ***
   // *** make them available as soon as possible ***
   //
   // *** Ryo Sato ***
   // *** May 8, 2009 ***
   //

	void Add( int row, int col, T Value );
	void Divide( T Value );

   // product of two matices of equal size

   bool bSqrProduct( pixq_Matrix<T> *Mtx0, pixq_Matrix<T> *Mtx1 );

   bool Crop( int nTopRow, int nBottomRow, int nLeftCol, int nRightCol, pixq_Matrix<T> *pDest ) const; 

   bool Transpose( pixq_Matrix<T> *pIn );

};

} // namespace _pix_plot_qt_framework 

namespace _pix_plot_qt_framework {

template <class T> pixq_Matrix<T>::pixq_Matrix( void )
{
	_iNrows = 0;
	_iNcols = 0;
	_iFrameRows = 0;
	_iFrameCols = 0;
	_iSize = 0;

   _bFrame = false;
   _iFunc = 0;

   _Data = (T**)NULL;	
   _pDataRow = (T**)NULL;		

   _bStatSet = false;
   _bNullSet = false;
  
   return;
}

template <class T> pixq_Matrix<T>::~pixq_Matrix( void )
{
   Free( );

   _Data = (T**)NULL;	
   _pDataRow = (T**)NULL;

   return;
}

//
// allocate matrix with no frame
//
template <class T> bool pixq_Matrix<T>::Alloc( 
   int nRows,  // [i] Height of the matrix
   int nCols ) // [i] Width of the matrix
{
	bool bRetCode = false; 

   if( !Alloc( nRows, nCols, 0, 0 ) ) {
      goto PIX_EXIT;
   }

   // --- Done ---
   bRetCode = true;
PIX_EXIT:
	return bRetCode;
}

template <class T> bool pixq_Matrix<T>::Alloc( 
   int nRows, // [i] Height of the matrix
   int nCols, // [i] Width of the matrix
   int nFrameRows, // [i] Height of the Frame, = 0
   int nFrameCols ) // [i] Height of the Width, = 0
{
	bool bRetCode = false;
   int i;
   bool bRealloc;

   if( 0 >= nRows ) goto PIX_EXIT;
   if( 0 >= nCols ) goto PIX_EXIT;
   if( 0 > nFrameRows ) goto PIX_EXIT;
   if( 0 > nFrameCols ) goto PIX_EXIT;

   // check if realloc is required
   bRealloc = false;
   if( !_Data ) bRealloc = true;

   if( !bRealloc ) {
      if( _iNrows != nRows ) bRealloc = true;
	   if( _iNcols != nCols ) bRealloc = true;
      if( _iFrameRows != nFrameRows ) bRealloc = true;
	   if( _iFrameCols != nFrameCols ) bRealloc = true;
   }

	_bStatSet = false;

   if( !bRealloc ) {
      // no need to re allocation, same as current size!
      bRetCode = true;
      goto PIX_EXIT;
   }

   Free( );

   _iNrows = nRows;
	_iNcols = nCols;
   _iFrameRows = nFrameRows;
	_iFrameCols = nFrameCols;

   _iWholeRows = _iNrows + _iFrameRows + _iFrameRows;
   _iWholeCols = _iNcols + _iFrameCols + _iFrameCols;

	_iSize = _iWholeRows * _iWholeCols;

	_Data = new T*[_iWholeRows];
   if( !_Data ) {
      goto PIX_EXIT;
   }

   for( i = 0 ; i < _iWholeRows ; i++ ) {
      _Data[i] = new T[_iWholeCols];
      if( !_Data[i] ) {
         goto PIX_EXIT;
      }
   }

   // --- set row/col top position table excluding frames  ---

	_pDataRow = new T*[_iNrows];
	if( NULL == _pDataRow ) goto PIX_EXIT;

   for( i = 0 ; i < _iNrows ; i++ ) {
      _pDataRow[i] = (T*)( _Data[i+_iFrameRows] + _iFrameCols );
   }

   // does this matrix have frame?

   if( 0 < nFrameRows || 0 < nFrameCols ) {
      _bFrame = true;
   } else {
      _bFrame = false;
   }

   // set pointer to the get func

   funcArray[0] = &pixq_Matrix<T>::GetFrame;
   funcArray[1] = &pixq_Matrix<T>::GetNoFrame;

   if( _bFrame ) {
      _iFunc = 0;
   } else {
      _iFunc = 1;
   }

   // --- Done ---
   bRetCode = true;
PIX_EXIT:
	return bRetCode;
}

//
// allocate matrix with no frame
// check if the current dimension is same as new dimension
// if identicall return as it is
// if not, release and allocate new matrices
//
template <class T> bool pixq_Matrix<T>::CheckAlloc( 
   int nRows,  // [i] Height of the matrix
   int nCols ) // [i] Width of the matrix
{
	bool bRetCode = false; 

   if( nRows == _iNrows && nCols == _iNcols ) {
      bRetCode = true;
      goto PIX_EXIT;
   }

   Free( );

   if( !Alloc( nRows, nCols, 0, 0 ) ) {
      goto PIX_EXIT;
   }

   // --- Done ---
   bRetCode = true;
PIX_EXIT:
	return bRetCode;
}

//
// release matrix
//
template <class T> void pixq_Matrix<T>::Free( void )
{
   int i;

   if( 0 >= _iNrows || 0 >= _iNcols ) {
	   _iNrows = 0;
	   _iNcols = 0;
      return;
   }

   for( i = 0 ; i < _iWholeRows ; i++ ) {
      if( _Data[i] ) {
         delete[] _Data[i];
         _Data[i] = NULL;
      }
   }

   if( _Data ) {
      delete[] _Data;
      _Data = NULL;
   }

   if( _pDataRow ) {
      delete[] _pDataRow;
      _pDataRow = (T**)NULL;
   }

   _iNrows = 0;
   _iNcols = 0;
   _iFrameRows = 0;
   _iFrameCols = 0;
   _iSize = 0;
	_bStatSet = false;

   return;
}

} // namespace _pix_plot_qt_framework 

// matrix data handling functions
#include "pixq_MatrixData.h"

// matrix analysis, statistics and histogram, etc
#include "pixq_MatrixStat.h"

// matrix processing utility
#include "pixq_MatrixUt.h"

// basic linear algebra
#include "pixq_MatrixLa.h"

