#include <pixqt_common.h>

#include <pixqtlib.h>
using namespace _pix_plot_qt_framework;

#include "hikari_common.h"


#include "hikari_window.h"

static bool DecomposeShinSingleVerse( QString strLine,
                                 int iBook,       
                                  int &iChapter, 
                                  int &iVerse, // [o] set to -1 not to record the line
                                  QString &strWords ) 
{
   bool bRetCode = false;
   bool bOK;
   int nSize, iBlank, iColon;
   QString strChapter, strCheck, strVerse;
   QChar ch;
   QChar chBlank = ' ';
   QChar chColon = ':';
   QChar chBlanket = '[';
   QChar chEndBlanket = ']';

   static int iLastChapter = -1;

//[ 10 ]
// 10:1 ɁAAnVGX́A{yƊC̓XɋۂB
// 10:2 ނ̌ЂƗECɂ邷ׂĂ̌тƁAɏd񂶂ꂽfJC̈̑傳ɂĂ̏ڍׂƂ́AfBƃyV̉̔NL̏ɂ邳Ăł͂ȂB
// 10:3 ̓_lfJCAAnVGX̎ɈʂA_l̒ł傢Ȃ҂łAނ̑̓EɌhA̖̍K߁ȂSɕałB
//
//GXeLI

//
// Psalmŝ݁AԍȂ̍s
//
// [ 3 ]
// _rf̎qAuV̂ꂽƂ̎^
// 3:1 BȂƎ̓GӂĂƂł傤Bɗ҂܂B
// 3:2 ̎҂̂܂̂ƂĂ܂Buނɐ_̋~͂ȂBvƁBZ
//
// т̉L̍s͑S
//
// ꊪ
//        w҂̂߂ɁBR̎q̃}XL[
// O      ATt̎^
// l      _̐l[Z̋F
// ܊
//

   strLine = strLine.trimmed( );
   nSize = (int)strLine.size( );
   if( 0 >= nSize ) {
      bRetCode = true;
      goto PIX_EXIT;
   }

   //
   // т''n܂s͖
   //

   if( 18 == iBook ) {
      strCheck = strLine.left( 1 );
      if( 0 == strCheck.compare( "" ) ) {
         iVerse = -1;
         bRetCode = true;
         goto PIX_EXIT;
      }
   }

   //
   // ['n܂s͏͔ԍ
   //
   ch = strLine.at( 0 );
   if( ch == chBlanket ) {
      strLine = strLine.remove( chBlanket );
      strLine = strLine.remove( chEndBlanket );
      strLine = strLine.trimmed( );
      iLastChapter = strLine.toInt( &bOK );
      if( bOK ) {
         iVerse = -1;
         bRetCode = true;
         goto PIX_EXIT;
      } else {
         goto PIX_EXIT;
      }
   }

   //
   // LȊO̍s1:1A͔ԍ:ߔԍ{uNŎn܂
   // тŏ:ߔԍ̂Ȃs͉is0jƂ
   //
   ch = strLine.at( 0 );
   if( !ch.isDigit( ) ) {
      //
      // тŏ:ߔԍ̂Ȃs͉is0jƂ
      //
      if( 18 == iBook ) { 
         iChapter = iLastChapter;
         iVerse = 0;
         strWords = strLine;
         bRetCode = true;
         goto PIX_EXIT;
      } else {
         goto PIX_EXIT;
      }
   }

   // first occurance of blank

   iBlank = strLine.indexOf( chBlank );
   if( 0 >iBlank ) {
      goto PIX_EXIT;
   }

   iColon = strLine.indexOf( chColon );
   if( 1 >iColon ) {
      goto PIX_EXIT;
   }

   strChapter = strLine.left( iColon );
   iChapter = strChapter.toInt( &bOK );
   if( !bOK ) {
      goto PIX_EXIT;
   }

   strVerse = strLine.mid( iColon+1, iBlank-iColon-1 );
   iVerse = strVerse.toInt( &bOK );
   if( !bOK ) {
      goto PIX_EXIT;
   }

   strWords = strLine.mid( iBlank+1, nSize-iBlank-1 );

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}

static bool DecomposeKjvSingleVerse( QString strLine,
                                  int &iChapter, 
                                  int &iVerse,
                                  QString &strWords ) 
{
   bool bRetCode = false;
   bool bOK;
   int iTab[3], nSize, iColon;
   QChar chTab = 0x09;
   QChar chColon = ':';
   QString strVerse, strChapter;

//_E4trans	Ru	1:8	And Naomi said to her two daughters in law, Go, return each to her mother's house. the Lord deal kindly with you, as you have dealt with the dead, and with me.
//_E4trans	Ru	1:9	The Lord grant you that you may find rest, each of you in the house of her husband. Then she kissed them. and they lifted up their voice, and wept.
//_E4trans	Ru	1:10	And they said to her. Surely we will return with you to your people.

   nSize = (int)strLine.size( );
   if( 0 >= nSize ) {
      goto PIX_EXIT;
   }

   // there should be three tabs

   iTab[0] = strLine.indexOf( chTab, 0 );
   if( 0 > iTab[0] ) {
      goto PIX_EXIT;
   }

   iTab[1] = strLine.indexOf( chTab, iTab[0]+1 );
   if( 0 > iTab[1] ) {
      goto PIX_EXIT;
   }

   iColon = strLine.indexOf( chColon );
   if( 1 >iColon ) {
      goto PIX_EXIT;
   }

   iTab[2] = strLine.indexOf( chTab, iTab[1]+1 );
   if( 0 > iTab[2] ) {
      goto PIX_EXIT;
   }

   // get header

   strChapter = strLine.mid( iTab[1]+1, iColon-iTab[1]-1 );
   iChapter = strChapter.toInt( &bOK );
   if( !bOK ) {
      goto PIX_EXIT;
   }

   strVerse = strLine.mid( iColon+1, iTab[2]-iColon-1 );
   iVerse = strVerse.toInt( &bOK );
   if( !bOK ) {
      goto PIX_EXIT;
   }

   strWords = strLine.mid( iTab[2]+1, nSize-iTab[2]-1 );
   
   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}

static bool DecomposeKougoSingleVerse( QString strLine,
                                  int &iChapter, 
                                  int &iVerse,
                                  QString &strWords ) 
{
   bool bRetCode = false;
   bool bNoWord,  bOK;
   int iTab[3], nSize, iColon;
   QChar chTab = 0x09;
   QChar chColon = ':';
   QString strVerse, strChapter;

//_J	2Ki	1:14	ȂA΂V炭āǍ܏\l̒ӂƁǍ̕܏\lƂĂs܂B킽̖Ȃ̖ڂɑ̂Ƃ݂ȂĂvB
//_J	2Ki	1:15	̎A̎g̓GɌAuނƋɉȂBނĂ͂ȂȂvBŃG͗āAނƋɉÂƂ֍sāA
//_J	2Ki	1:16	ɌAu͂܂AwȂ̓GN̐_oAE[uuɐq˂悤Ǝg҂킵A̓CXGɁǍtނׂ_Ȃ߂ł邩B䂦Ȃ́AoQ䂩~邱ƂȂAKʂł낤xvB

   nSize = (int)strLine.size( );
   if( 0 >= nSize ) {
      goto PIX_EXIT;
   }

   // there should be three tabs

   iTab[0] = strLine.indexOf( chTab, 0 );
   if( 0 > iTab[0] ) {
      goto PIX_EXIT;
   }

   iTab[1] = strLine.indexOf( chTab, iTab[0]+1 );
   if( 0 > iTab[1] ) {
      goto PIX_EXIT;
   }

   iColon = strLine.indexOf( chColon );
   if( 1 >iColon ) {
      goto PIX_EXIT;
   }

   iTab[2] = strLine.indexOf( chTab, iTab[1]+1 );
   if( 0 > iTab[2] ) {
      // some line has no word at all but  headers
      // _J	Nu	15:4
      bNoWord = true;
   } else {
      bNoWord = false;
   }

   // get header

   strChapter = strLine.mid( iTab[1]+1, iColon-iTab[1]-1 );
   iChapter = strChapter.toInt( &bOK );
   if( !bOK ) {
      goto PIX_EXIT;
   }

   if( bNoWord ) {
      strVerse = strLine.right( nSize-iColon-1 );
      iVerse = strVerse.toInt( &bOK );
      if( !bOK ) {
         goto PIX_EXIT;
      }

      strWords = "-";
   } else {
      strVerse = strLine.mid( iColon+1, iTab[2]-iColon-1 );
      iVerse = strVerse.toInt( &bOK );
      if( !bOK ) {
         goto PIX_EXIT;
      }

      strWords = strLine.mid( iTab[2]+1, nSize-iTab[2]-1 );
   }

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}

static bool DecomposeSingleVerse( int iVersion, 
                                 int iBook,               
                                  QString strLine,
                                  int &iChapter, 
                                  int &iVerse,              
                                  QString &strWords ) 
{
   bool bRetCode = false;

   switch( iVersion ) {
   case pixq_Properties::BibleShinkaiyaku2:
      if( !DecomposeShinSingleVerse( strLine, iBook, iChapter, iVerse, strWords ) ) {
         goto PIX_EXIT;
      }
      break;
      
   case pixq_Properties::BibleKougoyaku:
      if( !DecomposeKougoSingleVerse( strLine, iChapter, iVerse, strWords ) ) {
         goto PIX_EXIT;
      }
      break;
      
   case pixq_Properties::BibleKingJames:
      if( !DecomposeKjvSingleVerse( strLine, iChapter, iVerse, strWords ) ) {
         goto PIX_EXIT;
      }
      break;
   }

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}

static bool WriteBungoBookXml( BibleBook &bc,
                          int iBook,
                          QStringList strLines,
                          QXmlStreamWriter &writer )
{
   bool bRetCode = false;
   bool bFound, bFirstVerse, bVerseZero;
   int nLines, i, nSize, nChapters, nVerses, iChp, iChapterVerses, iVerse;
   int iChapterStart, iChapterEnd, iPos, iFrom, nVerseLen;
   int iVerseStart, iVerseEnd, kk;
   QString strBook, strLine, strChapter, strWords, strVerse, strCapter, strBuffer;
   QVector<BibleVerse> vrs;
   QVector<int> viChapterPos;
   QVector<int> viVersePos;
   QVector<int> viVersNumStart;
   QVector<int> viVerseStart;
   QVector<int> viVerseEnd;
   BibleVerse vr;

   QChar ch;
    
   //

   nLines = (int)strLines.size( );
   nChapters = bc.GetChapters( );

   if( nLines <= nChapters ) {
      goto PIX_EXIT;
   }

   // first line

   strBook = bc.GetBungoBook( );
   strLine = strLines.at( 0 );
   strLine = strLine.trimmed( );
   if( 0 != strBook.compare( strLine ) ) {
      goto PIX_EXIT;
   }

   // chapter number locations

   viChapterPos.resize( nChapters );
   viChapterPos.fill( -1 );

   for( iChp = 0 ; iChp < nChapters ; iChp++ ) {
      if( 18 == iBook ) {
         strCapter.sprintf( "%d", iChp + 1 );
      } else {
         strCapter.sprintf( "%d", iChp + 1 );
      }

      bFound = false;
      for( i = 1 ; !bFound && i < nLines ; i++ ) {
         strLine = strLines.at( i );
         strLine = strLine.trimmed( );

         nSize = (int)strLine.size( );
         if( 0 >= nSize ) continue;

         if( 0 == strCapter.compare( strLine ) ) {
            viChapterPos.replace( iChp, i );
            bFound = true;
            continue;
         }
      }
   
      if( !bFound ) {
         // 1ͥȂI
         goto PIX_EXIT;
      }
   }

   //

   vrs.clear( );

   for( iChp = 0 ; iChp < nChapters ; iChp++ ) {
      iChapterStart = viChapterPos.at( iChp ) + 1;
      if( iChp == nChapters-1 ) {
         iChapterEnd = nLines - 1;
      } else {
         iChapterEnd = viChapterPos.at( iChp+1 ) - 1;
      }

      // add all lines to single string

      // all one chapter in one line
      // 2:1xxxxxx2:2bbbbbbb2:3cccccc.....

      strBuffer.clear( );
      for( i = iChapterStart ; i <= iChapterEnd ; i++ ) {
         strLine = strLines.at( i );
         strLine = strLine.trimmed( );

         nSize = (int)strLine.size( );
         if( 0 >= nSize ) continue;

         strBuffer.append( strLine );
      }

      // decompose buffer into verses
      // find the position of verse numbers, 2:1 2:2 2:3 etc
      // 
      // 2:1xxxxxx2:2bbbbbbb2:3cccccc.....
      // set start position of verse numbers(for example 2:2) in viVersePos

      nVerses = bc.GetChapterVerseNumber( iChp+1 );
      viVersePos.resize( nVerses );
      viVersNumStart.resize( nVerses );
      viVerseStart.resize( nVerses );
      viVerseEnd.resize( nVerses );

      viVersePos.fill( -1 );
      viVersNumStart.fill( -1 );
      viVerseStart.fill( -1 );
      viVerseEnd.fill( -1 );

      for( iVerse = 0 ; iVerse < nVerses ; iVerse++ ) {
         // chapter verse string 1:1, find it!
         strVerse.sprintf( "%d:%d\0", iChp+1, iVerse+1 );
         nVerseLen = (int)strVerse.size( );

         iFrom = 0;
         iPos = strBuffer.indexOf( strVerse, iFrom );
         if( 0 > iPos ) {
            // verse not found!
            goto PIX_EXIT;
         }

         strVerse.replace( iVerse, iPos );
         viVersNumStart.replace( iVerse, iPos );
         viVerseStart.replace( iVerse, iPos + nVerseLen );
         if( 0 < iVerse ) {
            viVerseEnd.replace( iVerse-1, iPos-1 );
         }
      }

      nVerseLen = (int)strBuffer.size( );
      viVerseEnd.replace( nVerses-1, nVerseLen-1 );

      // Psalms 
      // verse zero may exist

      if( 18 == iBook ) {
         bVerseZero = bc.GetVerseZero( iChp+1 );

         if( bVerseZero ) {
            iVerseEnd = viVersNumStart.at( 0 );
            if( 0 >= iVerseEnd ) {
               // verse zeroȂ
               strWords = "-";
            } else {
               strWords = strBuffer.left( iVerseEnd );
               strWords = strWords.trimmed( );
            }
            vr.Set( iChp+1, 0, strWords );
         }
      }

      // decompose buffer into verses
      //
      // 2:1xxxxxx2:2bbbbbbb2:3cccccc.....
      // set [0] xxxxx, [1] bbbbb, [3] ccccc
      //

      for( iVerse = 0 ; iVerse < nVerses ; iVerse++ ) {
         iVerseStart = viVerseStart.at( iVerse );
         iVerseEnd = viVerseEnd.at( iVerse );

         strWords = strBuffer.mid( iVerseStart, iVerseEnd - iVerseStart + 1 );
         strWords = strWords.trimmed( );

         vr.Set( iChp+1, iVerse+1, strWords );
         vrs.push_back( vr );
      }

      // check the last verse and make sure it does not contain :

      iVerseStart = viVerseStart.at( nVerses-1 );
      iVerseEnd = viVerseEnd.at( nVerses-1 );
      strWords = strBuffer.mid( iVerseStart, iVerseEnd - iVerseStart + 1 );
      strWords = strWords.trimmed( );

      iPos = strWords.indexOf( ":", 0 );
         if( 0 <= iPos ) {
            // the last verse has :
            goto PIX_EXIT;
         }
   }

   //
   // Psalms
   // check whether verse 0 exists for the chapters it should exist
   // add black verse 0

   // 

   nVerses = (int)vrs.size( );

   for ( i = 0 ; i < nChapters ; i++ ) {
      iChp = i + 1;

      writer.writeStartElement( "Chapter" ); // Chapter

      strChapter.sprintf( "%d", iChp );
      writer.writeAttribute( "ID", strChapter );

      iChapterVerses = bc.GetChapterVerseNumber( iChp );
      strVerse.sprintf( "%d", iChapterVerses );
      writer.writeAttribute( "Verses", strVerse );

      //
      // Psalm, verse 0 exists?
      //
      if( 18 == iBook ) {
         bVerseZero = bc.GetVerseZero( i );
      } else {
         bVerseZero = false;
      }

      bFirstVerse = true;
      for( kk = 0 ; kk < nVerses ; kk++ ) {
         vr = vrs.at( kk );
         if( iChp != vr.GetChapter( ) ) {
            continue;
         }

         if( bFirstVerse ) {
            if( bVerseZero ) {
            
               // verse zero should exist and this verse should be zero
               iVerse = vr.GetVerse( );
               if( 0 != iVerse ) {
                  // add pseudo verse zero
                  writer.writeStartElement( "Verse" ); // Verse
                  writer.writeAttribute( "ID", "0" );
                  writer.writeAttribute( "Words", "-" ); 
                  writer.writeEndElement(); // Verse
               }
            }
            bFirstVerse = false;
         }      

         writer.writeStartElement( "Verse" ); // Verse

         iVerse = vr.GetVerse( );
         strVerse.sprintf( "%d", iVerse );
         writer.writeAttribute( "ID", strVerse );
   
         strWords = vr.GetWord( );
         //writer.writeTextElement( "Words", strWords ); 
         writer.writeAttribute( "Words", strWords ); 

         writer.writeEndElement(); // Verse

      }

      writer.writeEndElement(); // Chapter
   }

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}

static bool WriteBookXml( BibleBook &bc,
                          int iVersion,
                          int iBook,
                          QStringList strLines,
                          QXmlStreamWriter &writer )
{
   bool bRetCode = false;
   bool bFound, bFirstVerse, bVerseZero;
   int nLines, i, nSize, nChapters, iChapter, k, nVerses, nChapterVerses, iVerse;
   QString strLine, strChapter, strWords, strBook, strVerse;
   QVector<BibleVerse> vrs;
   QVector<bool> vbVerseFound;
   BibleVerse vr;

   QChar ch;

   // special handling for bungo

   if( pixq_Properties::BibleBungo == iVersion ) { 
      if( !WriteBungoBookXml( bc, iBook, strLines, writer ) ) {
         goto PIX_EXIT;
      }

      bRetCode = true;
      goto PIX_EXIT;
   }
    
   //

   strBook = bc.GetBungoBook( );
   nChapters = bc.GetChapters( );

   nLines = (int)strLines.size( );
   vrs.clear( );
   for( i = 0 ; i < nLines ; i++ ) {
      strLine = strLines.at( i );
      strLine = strLine.trimmed( );

      nSize = (int)strLine.size( );
      if( 0 >= nSize ) continue;

      if( !DecomposeSingleVerse( iVersion, iBook, strLine, iChapter, iVerse, strWords ) ) {
         goto PIX_EXIT;
      }

      if( 0 > iVerse ) {
         continue;
      }

      vr.Set( iChapter, iVerse, strWords );
      vrs.push_back( vr );
   }

   //
   // Psalms
   // check whether verse 0 exists for the chapters it should exist
   // add black verse 0

   nVerses = (int)vrs.size( );

   for ( i = 0 ; i < nChapters ; i++ ) {
      iChapter = i + 1;

      writer.writeStartElement( "Chapter" ); // Chapter

      strChapter.sprintf( "%d", iChapter );
      writer.writeAttribute( "ID", strChapter );

      nChapterVerses = bc.GetChapterVerseNumber( iChapter );
      strVerse.sprintf( "%d", nChapterVerses );
      writer.writeAttribute( "Verses", strVerse );

      // check verses
      
      vbVerseFound.resize( nChapterVerses+1 );
      vbVerseFound.fill( false );

      //
      // Psalm, verse 0 exists?
      //
      if( 18 == iBook ) {
         bVerseZero = bc.GetVerseZero( i );
         if( !bVerseZero ) {
            vbVerseFound.replace( 0, true );
         }
      } else {
         bVerseZero = false; 
         vbVerseFound.replace( 0, true );
      }


      bFirstVerse = true;
      for( k = 0 ; k < nVerses ; k++ ) {
         vr = vrs.at( k );
         if( iChapter != vr.GetChapter( ) ) {
            continue;
         }

         if( bFirstVerse ) {
            if( bVerseZero ) {
            
               // verse zero should exist and this verse should be zero
               iVerse = vr.GetVerse( );
               if( 0 != iVerse ) {
                  // add pseudo verse zero
                  writer.writeStartElement( "Verse" ); // Verse
                  writer.writeAttribute( "ID", "0" );
                  //writer.writeTextElement( "Words", "-" ); 
                  writer.writeAttribute( "Words", "-" ); 
                  writer.writeEndElement(); // Verse
                  vbVerseFound.replace( 0, true );
               }
            }
            bFirstVerse = false;
         }      

         writer.writeStartElement( "Verse" ); // Verse

         iVerse = vr.GetVerse( );
         strVerse.sprintf( "%d", iVerse );
         writer.writeAttribute( "ID", strVerse );
   
         strWords = vr.GetWord( );
         //writer.writeTextElement( "Words", strWords ); 
         writer.writeAttribute( "Words", strWords ); 

         writer.writeEndElement(); // Verse

         vbVerseFound.replace( iVerse, true );
      }

      for( k = 0 ; k <= nChapterVerses ; k++ ) {
         bFound = vbVerseFound.at( k );
         if( !bFound ) {
            // verse does not exist
            goto PIX_EXIT;
         }
      }

      writer.writeEndElement(); // Chapter
   }

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}

bool hikari_window::CovertBibleVersionToXml( int iVersion ) 
{
   bool bRetCode = false;
   bool bStatus;
   int iBook, nLines, nChapters, iBookFirst, iBookLast;
   QString strLine, strError, strVersion, strMessage, strXmlFile, strFile;
   QString strChapters, strBookNum, strBookName;

   QFile *in_file = NULL;
   QFile *out_file = NULL;
   QStringList strLines;
   QXmlStreamWriter writer;

   BibleAllBooks *pBL = GetBibleBooks( );
   BibleBook bc;

   // create output xml file

   strXmlFile = GetBibleXmlFile_PC( iVersion );
   if( strXmlFile.isEmpty( ) ) {
      strMessage.sprintf( "ERROR! Wrong version %d", iVersion );
      QMessageBox::critical( NULL, _strApplication, strMessage );
      goto PIX_EXIT;
   }

   out_file = new QFile( strXmlFile );
   if ( !out_file->open( QFile::WriteOnly | QFile::Text ) ) {
      strMessage = "ERROR! Failed to Create Temporary File:\n" + strXmlFile;
      QMessageBox::critical( NULL, _strApplication, strMessage );
      goto PIX_EXIT;
   }

   writer.setAutoFormatting(true);
   writer.setDevice( out_file );

   // write contents

   writer.writeStartDocument();
   //writer.writeDTD("<!DOCTYPE occasions>");

   strVersion = pixq_Properties::GetBibleVersionName( iVersion );
   writer.writeStartElement( "hikari" );
   writer.writeAttribute( "version", "1.0" );
   writer.writeAttribute( "data", "bibles" );
   writer.writeAttribute( "translation", strVersion );

   // read each book

   BibleAllBooks::VersionBookRange( iVersion, iBookFirst, iBookLast );

   for( iBook = iBookFirst ; iBook <= iBookLast ; iBook++ ) {

      bc = pBL->GetBook( iBook );

      // open input book file and read all lines into list
      strFile = pBL->GetVersionPath( iVersion, iBook );
      
      in_file = new QFile( strFile );
      if ( !in_file->open( QIODevice::ReadOnly | QIODevice::Text ) ) {
         strError = "ERROR! Failed to open file " + strFile + ".";
         QMessageBox::critical( NULL, _strApplication, strError );
         goto PIX_EXIT;
      }

      strLines.clear( );

      bStatus = true;
      while( bStatus ) {
         
         if( in_file->atEnd( ) ) {
			   bStatus = false;
	   	   continue;
         }

         strLine = in_file->readLine( );
         strLines.append( strLine );
         
      }

      nLines = (int)strLines.size( );

      // close in file
      in_file->close( );
      in_file = NULL;

      // write book info

      writer.writeStartElement( "Book" );

      strBookNum.sprintf( "%d", iBook + 1 );
      writer.writeAttribute( "ID", strBookNum );

      strBookName = bc.GetVersionBookName( iVersion );
      writer.writeAttribute( "Name", strBookName );

      nChapters = bc.GetChapters( );
      strChapters.sprintf( "%d", nChapters );
      writer.writeAttribute( "Chapters", strChapters );

      // write lines

      if( !WriteBookXml( bc, iVersion, iBook, strLines, writer ) ) {
         goto PIX_EXIT;
      }

      // end of the book

      writer.writeEndElement(); // Book
   }

   // end the last section

   writer.writeEndElement();

   // create output xml file

   writer.writeEndDocument();

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   if( in_file ) {
      in_file->close( );
      in_file = NULL;
   }
   if( out_file ) {
      out_file->close( );
      out_file = NULL;
   }
   return bRetCode;
}

static bool DecomposeRaguetBuffer( QString strLine,
                                 QVector<BibleVerse> &vrs ) 
{
   bool bRetCode = false;
   bool bOK;
   int iChapterStart, iVerseEnd, nColons, iChapter, iVerse, i, nSize, iPos;
   int iVerseStart;
   QString strChapter, strVerse, strWords;
   QChar ch;
   QChar chColon = ':';
   QVector<int> colonPos;
   QVector<int> chapterStart;
   QVector<int> verseEnd;
   QVector<int> chapterList;
   QVector<int> verseList;

   BibleVerse vr;

   strLine = strLine.trimmed( );
   nSize = (int)strLine.size( );
   if( 0 >= nSize ) {
      bRetCode = true;
      goto PIX_EXIT;
   }

   //
   // make a list of all occurance of :
   //
   colonPos.clear( );
   for( i = 0 ; i < nSize ; i++ ) {
      ch = strLine.at( i );
      if( ch == chColon ) {
         colonPos.push_back( i );
      }
   }

   nColons = (int)colonPos.size( );
   if( 0 >= nColons ) {
      bRetCode = true;
      goto PIX_EXIT;
   }

   // make a list of chapter:verse positions in the line

   chapterStart.clear( );
   verseEnd.clear( );
   chapterList.clear( );
   verseList.clear( );

   for( i = 0 ; i < nColons ; i++ ) {
      iPos = colonPos.at( i );

      // chapter
      strChapter.clear( );
      if( 0 == iPos ) {
         // wrong!
         goto PIX_EXIT;
      }
      if( iPos >= 3 ) {
         ch = strLine.at( iPos-3 );
         if( ch.isDigit( ) ) {
            strChapter.push_back( ch );
            iChapterStart = iPos-3;
         }
      }
      if( iPos >= 2 ) {
         ch = strLine.at( iPos-2 );
         if( ch.isDigit( ) ) {
            strChapter.push_back( ch );
            iChapterStart = iPos-2;
         }
      }
      if( iPos >= 1 ) {
         ch = strLine.at( iPos-1 );
         if( ch.isDigit( ) ) {
            strChapter.push_back( ch );
            iChapterStart = iPos-1;
         }
      }
      if( 0 >= strChapter.size( ) ) {
         goto PIX_EXIT;
      }
      iChapter = strChapter.toInt( &bOK );
      if( !bOK ) {
         goto PIX_EXIT;
      }

      // verse
      strVerse.clear( );
      if( iPos + 1 <= nSize - 1 ) {
         ch = strLine.at( iPos+1 );
         if( ch.isDigit( ) ) {
            strVerse.push_back( ch );
            iVerseEnd = iPos+1;
         }
      }
      if( iPos + 2 <= nSize - 1 ) {
         ch = strLine.at( iPos+2 );
         if( ch.isDigit( ) ) {
            strVerse.push_back( ch );
            iVerseEnd = iPos+2;
         }
      }
      if( iPos + 3 <= nSize - 1 ) {
         ch = strLine.at( iPos+3 );
         if( ch.isDigit( ) ) {
            strVerse.push_back( ch );
            iVerseEnd = iPos+3;
         }
      }
      if( 0 >= strVerse.size( ) ) {
         goto PIX_EXIT;
      }
      iVerse = strVerse.toInt( &bOK );
      if( !bOK ) {
         goto PIX_EXIT;
      }
      
      //
      chapterStart.push_back( iChapterStart );
      verseEnd.push_back( iVerseEnd );
      chapterList.push_back( iChapter );
      verseList.push_back( iVerse );
   }

   // decompose line into verses

   
   for( i = 0 ; i < nColons ; i++ ) {
      iChapter = chapterList.at( i );
      iVerse = verseList.at( i );

      iVerseStart = verseEnd.at( i )+1;
      if( i == nColons - 1 ) {
         iVerseEnd = nSize - 1;
      } else {
         iVerseEnd = chapterStart.at( i+1 ) - 1;
      }

      strWords = strLine.mid( iVerseStart, iVerseEnd-iVerseStart+1 );
      strWords = strWords.trimmed( );

      vr.Set( iChapter, iVerse, strWords );
      vrs.push_back( vr );
   }

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}


static bool WriteRaguetBookXml( BibleBook &bc,
                          int iVersion,
                          int iBook,
                          QStringList strLines,
                          QXmlStreamWriter &writer )
{
   bool bRetCode = false;
   bool bFound;
   int nLines, i, nChapters, iChapter, k, nVerses, nChapterVerses, iVerse;
   QString strValue, strLine, strWords;
   QVector<BibleVerse> vrs;
   QVector<bool> vbVerseFound;
   BibleVerse vr;

  // QChar ch;

   // write book info


   writer.writeStartElement( "Book" );

   strValue.sprintf( "%d", iBook + 1 );
   writer.writeAttribute( "ID", strValue );

   strValue = bc.GetVersionBookName( iVersion );
   writer.writeAttribute( "Name", strValue );

   nChapters = bc.GetChapters( );
   strValue.sprintf( "%d", nChapters );
   writer.writeAttribute( "Chapters", strValue );

   //

   nChapters = bc.GetChapters( );
   vrs.clear( );

   if( iBook < BibleAllBooks::_iOldBookEnd ) {
   
      // this version dosen't have old testment
   
      strWords = "**V̂**";
      for ( i = 0 ; i < nChapters ; i++ ) {
         iChapter = i + 1;
         nChapterVerses = bc.GetChapterVerseNumber( iChapter );       
         for( k = 0 ; k < nChapterVerses ; k++ ) {
             vr.Set( i+1, k+1, strWords );
             vrs.push_back( vr );
         }      
      }

   } else {

      // decompose all read lines
   
      nLines = (int)strLines.size( );
      for( i = 0 ; i < nLines ; i++ ) {
         strLine = strLines.at( i );
         if( !DecomposeRaguetBuffer( strLine, vrs ) ) {
            goto PIX_EXIT;
         }
      }
   }

   //

   nVerses = (int)vrs.size( );

   for ( i = 0 ; i < nChapters ; i++ ) {
      iChapter = i + 1;

      writer.writeStartElement( "Chapter" ); // Chapter

      strValue.sprintf( "%d", iChapter );
      writer.writeAttribute( "ID", strValue );

      nChapterVerses = bc.GetChapterVerseNumber( iChapter );
      strValue.sprintf( "%d", nChapterVerses );
      writer.writeAttribute( "Verses", strValue );

      // check verses
      
      vbVerseFound.resize( nChapterVerses+1 );
      vbVerseFound.fill( false );

      for( k = 0 ; k < nVerses ; k++ ) {
         vr = vrs.at( k );
         if( iChapter != vr.GetChapter( ) ) {
            continue;
         }

         writer.writeStartElement( "Verse" ); // Verse

         iVerse = vr.GetVerse( );
         if( iVerse > nChapterVerses ) {
            // verse number too large
            goto PIX_EXIT;
         }

         strValue.sprintf( "%d", iVerse );
         writer.writeAttribute( "ID", strValue );
   
         strWords = vr.GetWord( );
         writer.writeAttribute( "Words", strWords ); 

         writer.writeEndElement( ); // Verse

         vbVerseFound.replace( iVerse, true );
      }

      for( k = 1 ; k <= nChapterVerses ; k++ ) {
         bFound = vbVerseFound.at( k );
         if( !bFound ) {
            // verse does not exist
            goto PIX_EXIT;
         }
      }

      writer.writeEndElement(); // Chapter
   }

   // end of the book

   writer.writeEndElement(); // Book

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}

bool hikari_window::ConvertRaguetBibleXml( void ) 
{
   bool bRetCode = false;
   bool bStatus;
   int iBook;
   QString strVersion, strMessage, strFile, strError, strLine, strXmlFile;
 
   int iVersion = pixq_Properties::BibleRaguet;

   QFile *in_file = NULL;
   QFile *out_file = NULL;
   QStringList strLines;
   QXmlStreamWriter writer;

   BibleAllBooks *pBL = GetBibleBooks( );
   int nBooks = pBL->GetLength( );
   BibleBook bc;

   // create output xml file

   strXmlFile = GetBibleXmlFile_PC( iVersion );

   out_file = new QFile( strXmlFile );
   if ( !out_file->open( QFile::WriteOnly | QFile::Text ) ) {
      strMessage = "ERROR! Failed to Create Temporary File:\n" + strXmlFile;
      QMessageBox::critical( NULL, _strApplication, strMessage );
      goto PIX_EXIT;
   }

   writer.setAutoFormatting(true);
   writer.setDevice( out_file );

   // write contents

   writer.writeStartDocument();
   //writer.writeDTD("<!DOCTYPE occasions>");

   strVersion = pixq_Properties::GetBibleVersionName( iVersion );
   writer.writeStartElement( "hikari" );
   writer.writeAttribute( "version", "1.0" );
   writer.writeAttribute( "data", "bibles" );
   writer.writeAttribute( "translation", strVersion );

   // read each book

   for( iBook = 0 ; iBook < nBooks ; iBook++ ) {

      bc = pBL->GetBook( iBook );

      if( iBook >= BibleAllBooks::_iOldBookEnd ) {
         // open input book file and read all lines into list
         strFile = pBL->GetVersionPath( iVersion, iBook );
      
         in_file = new QFile( strFile );
         if ( !in_file->open( QIODevice::ReadOnly | QIODevice::Text ) ) {
            strError = "ERROR! Failed to open file " + strFile + ".";
            QMessageBox::critical( NULL, _strApplication, strError );
            goto PIX_EXIT;
         }

         strLines.clear( ); 
         bStatus = true;
         while( bStatus ) {
            if( in_file->atEnd( ) ) {
               bStatus = false;
               continue;
            }

            strLine = in_file->readLine( );
            strLines.append( strLine );
         }

         // close in file
         in_file->close( );
         in_file = NULL;
      }

      // write lines

      if( !WriteRaguetBookXml( bc, iVersion, iBook, strLines, writer ) ) {
         goto PIX_EXIT;
      }
   }

   // end the last section

   writer.writeEndElement();

   // create output xml file

   writer.writeEndDocument();

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   if( in_file ) {
      in_file->close( );
      in_file = NULL;
   }
   if( out_file ) {
      out_file->close( );
      out_file = NULL;
   }
   return bRetCode;
}

//
// convert bible text files to xml
//
// notes:
// three bibles, kougoyaku, shinkaiyaku, kjv
// 
// 
// ł3̃t@CQtHikari\FilesɈړ
// bible_shin2.xml
// bible_kougo.xml
// bible_kjv.xml
// bible_isv.xml
//
bool hikari_window::ConvertBibleToXml( void )
{
   bool bRetCode = false;
   int i, iVersion, nVersions;
   vector<int> viVersions;

   // ISV

   if( !ConvertIsvBibleXml( ) ) {
      goto PIX_EXIT;
   }

   // KingJames
   // Kougoyaku 
   // ShinKaiyaku
   // Bungo

   viVersions.clear( );
   viVersions.push_back( pixq_Properties::BibleShinkaiyaku2 );
   //viVersions.push_back( pixq_Properties::BibleKingJames );
   //viVersions.push_back( pixq_Properties::BibleKougoyaku );
   //viVersions.push_back( pixq_Properties::BibleBungo ); 

   nVersions = (int)viVersions.size( );
   for( i = 0 ; i < nVersions ; i++ ) {
      iVersion = (int)viVersions.at( i );

      if( !CovertBibleVersionToXml( iVersion ) ) {
         goto PIX_EXIT;
      }
   }

   // Raguet

   if( !ConvertRaguetBibleXml( ) ) {
      goto PIX_EXIT;
   }
   
   // V
   
   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}

//
// ׂĂ̐̃o[WŁASAS͂̐ߐĂ邩mF
//
bool hikari_window::CheckAllBiblesSingleVer( void )
{
   bool bRetCode = false;
   BibleAllBooks *pBL = GetBibleBooks( );
   int iBook0, iBook1, iBook, iChapter, nChapters, iVersion, nVersions, iContent;
   int iVer, i;
   vector<int> viVersions;
   vector<int> viContents;
   vector<int> viContentStart;
   vector<int> viContentEnd;

   pixq_Properties *pEnv = GetHikariEnv( );

   BibleBook *pBc;

   // versions list

   viVersions.clear( );
   viVersions.push_back( pixq_Properties::BibleRaguet );
   //viVersions.push_back( pixq_Properties::BibleShinkaiyaku2 );
  // viVersions.push_back( pixq_Properties::BibleKingJames );
  // viVersions.push_back( pixq_Properties::BibleKougoyaku );
  // viVersions.push_back( pixq_Properties::BibleIntlStandard );
  // viVersions.push_back( pixq_Properties::BibleBungo ); 

   nVersions = (int)viVersions.size( );

   viContents.clear( );
   viContents.push_back( pixq_Properties::ContentOldTestament );
   viContents.push_back( pixq_Properties::ContentNewTestament );

   viContentStart.clear( );
   viContentStart.push_back( BibleAllBooks::_iOldBookStart );
   viContentStart.push_back( BibleAllBooks::_iNewBookStart );
   
   viContentEnd.clear( );
   viContentEnd.push_back( BibleAllBooks::_iOldBookEnd );
   viContentEnd.push_back( BibleAllBooks::_iNewBookEnd );

   // check all - single
   
   pEnv->_HikariVieBibleNum.Set( 1 );

   for( iVer = 0 ; iVer < nVersions ; iVer++ ) {

      iVersion = viVersions.at( iVer );
      pEnv->_HikariBibleVersion.Set( iVersion );
      pEnv->SetRegEnv( _strOrganisation, _strApplication );

      for( i = 0 ; i < 2 ; i++ ) {
         iContent = viContents.at( i );
         _Content->SetContent( iContent );
         
         iBook0 = viContentStart.at( i );
         iBook1 = viContentEnd.at( i );
         
         for( iBook = iBook0 ; iBook <= iBook1 ; iBook++ ) {
            _Content->SetBook( iBook );
            pBc = pBL->pGetBook( iBook-1 );
            nChapters = pBc->GetChapters( );
            for( iChapter = 1 ; iChapter <= nChapters ; iChapter++ ) {
               _Content->SetChapter( iChapter );
               ShowContent( ); 
            }
         }
      }
   }

   // --- DONE ---
   bRetCode = true;
//PIX_EXIT:
   return bRetCode;
}


bool hikari_window::CheckAllBiblesDoubleVer( void )
{
   bool bRetCode = false;
   BibleAllBooks *pBL = GetBibleBooks( );
   int iBook0, iBook1, iBook, iChapter, nChapters, iVersion, nVersions, iContent;
   int i2Ver, i, iVer, i2Version;
   vector<int> viVersions;
   vector<int> viContents;
   vector<int> viContentStart;
   vector<int> viContentEnd;

   pixq_Properties *pEnv = GetHikariEnv( );

   BibleBook *pBc;

   // single version

   // versions list

   viVersions.clear( );
   viVersions.push_back( pixq_Properties::BibleRaguet );
   viVersions.push_back( pixq_Properties::BibleShinkaiyaku2 );
   viVersions.push_back( pixq_Properties::BibleKingJames );
   viVersions.push_back( pixq_Properties::BibleKougoyaku );
   viVersions.push_back( pixq_Properties::BibleIntlStandard );
   viVersions.push_back( pixq_Properties::BibleBungo );

   nVersions = (int)viVersions.size( );

   //

   viContents.clear( );
   viContents.push_back( pixq_Properties::ContentOldTestament );
   viContents.push_back( pixq_Properties::ContentNewTestament );

   viContentStart.clear( );
   viContentStart.push_back( BibleAllBooks::_iOldBookStart );
   viContentStart.push_back( BibleAllBooks::_iNewBookStart );
   
   viContentEnd.clear( );
   viContentEnd.push_back( BibleAllBooks::_iOldBookEnd );
   viContentEnd.push_back( BibleAllBooks::_iNewBookEnd );

   // check all - single

   pEnv->_HikariVieBibleNum.Set( 2 );

   for( iVer = 0 ; iVer < nVersions ; iVer++ ) {

      iVersion = viVersions.at( iVer );
      pEnv->_HikariBibleVersion.Set( iVersion );
   
      for( i2Ver = 0 ; i2Ver < nVersions ; i2Ver++ ) {
   
         i2Version = viVersions.at( i2Ver );
         pEnv->_HikariBibleSecondVersion.Set( i2Version );
         pEnv->SetRegEnv( _strOrganisation, _strApplication );
   
         if( iVersion == i2Version ) continue;

         for( i = 0 ; i < 2 ; i++ ) {
            iContent = viContents.at( i );
            _Content->SetContent( iContent );
         
            iBook0 = viContentStart.at( i );
            iBook1 = viContentEnd.at( i );
            
            for( iBook = iBook0 ; iBook <= iBook1 ; iBook++ ) {
               _Content->SetBook( iBook );
               pBc = pBL->pGetBook( iBook-1 );
               nChapters = pBc->GetChapters( );
               for( iChapter = 1 ; iChapter <= nChapters ; iChapter++ ) {
                  _Content->SetChapter( iChapter );
                  ShowContent( );
               }
            }
         }
      }
   }

   // --- DONE ---
   bRetCode = true;
//PIX_EXIT:
   return bRetCode;
}

static bool IsShinkai3ChapterLine( BibleBook &bc,
                                   int iBook,
                                   QString strLine, // [i] one line read from the txt file
                                   int &iChapter )
{
   bool bRetCode = false;
   bool bOK;
   int nBookSize, nSize;
   QString strLineBook, strBook, strChapter, strLast;
   QString strShou = "";
   QString strHen = "";

   // 
   //  ͔ԍ݂̂Ȃ郉C͏͔ԍ
   // nL 50
   // 
   
   nSize = (int)strLine.size( );

   // s̐擪H

   strBook = bc.GetBookJpName( );
   nBookSize = strBook.size( );

   if( nBookSize >= nSize ) {
      goto PIX_EXIT;
   }
   
   strLineBook = strLine.left( nBookSize );
   if( 0 != strLineBook.compare( strBook ) ) {
      goto PIX_EXIT;
   }

   // s̍Ō̕h́hH

   strLast = strLine.right( 1 );
   if( 18 == iBook ) {
      if( 0 != strLast.compare( strHen ) ) {
         goto PIX_EXIT;
      }
   } else {
      if( 0 != strLast.compare( strShou ) ) {
         goto PIX_EXIT;
      }
   }

   strChapter = strLine.mid( nBookSize+1, nSize - nBookSize - 2 );

   iChapter = strChapter.toInt( &bOK );
   if( !bOK ) {
      // ͔ԍs
      goto PIX_EXIT;
   }

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}


static bool DecomposeShinkai3Verse( BibleBook &bc,
                                   int iBook,
                                   QString strLine, // [i] one line read from the txt file
                                  int &iChapter,  // [o] chapter number
                                  int &iVerse, // [o] set to -1 not to record the line
                                  QString &strWords ) 
{
   bool bRetCode = false;
   bool bOK;
   int iReadChapter, nSize, nChapters, iBlank;
   QString strVerse, strLeft;

   QString strToukou = "e";
   QString strLabel = "x";
   QString str2010 = "2010N";
   QChar chBlank = ' ';

   iVerse = -1;
   strWords.clear( );

   //

   strLine = strLine.trimmed( );
   nSize = (int)strLine.size( );
   if( 0 >= nSize ) {
      bRetCode = true;
      goto PIX_EXIT;
   }

   //
   // svȍs
   //
   // e Xiong M Jarka : 12:08 0 Rg ̓eւ̃N
   // x: 001nL
   // 2010N1210j
   //
   if( 3 < nSize ) {
      strLeft = strLine.left( 3 );
      if( 0 == strLeft.compare( strToukou ) ) {         
         bRetCode = true;
         goto PIX_EXIT;
      }
      
      if( 0 == strLeft.compare( strLabel ) ) {
         bRetCode = true;
         goto PIX_EXIT;
      }
   }

   if( 5 < nSize ) {
      strLeft = strLine.left( 5 );
      if( 0 == strLeft.compare( str2010 ) ) {         
         bRetCode = true;
         goto PIX_EXIT;
      }
   }

   // 
   //  ͔ԍ݂̂Ȃ郉C͏͔ԍ
   // nL 50
   // 

   nChapters = bc.GetChapters( );
   if( IsShinkai3ChapterLine( bc, iBook, strLine, iReadChapter ) ) {
      if( 0 > iReadChapter || iReadChapter > nChapters ) {
         // wrong chapter number!
         goto PIX_EXIT;
      }

      iChapter = iReadChapter;
      iVerse = -1;
      bRetCode = true;
      goto PIX_EXIT;
   }

   // One Verse in one Line
   // first occurance of blank

   iBlank = strLine.indexOf( chBlank );
   if( 0 >iBlank ) {
      goto PIX_EXIT;
   }

   strVerse = strLine.left( iBlank );
   iVerse = strVerse.toInt( &bOK );
   if( !bOK ) {
      goto PIX_EXIT;
   }

   strWords = strLine.right( nSize-iBlank-1 );

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}


static bool WriteShinkai3BookXml( BibleBook &bc,
                          int iBook,
                          QStringList strLines,
                          QXmlStreamWriter &writer )
{
   bool bRetCode = false;
   bool bSet, bFirstVerse, bVerseZero;
   int nLines, i, nSize, nChapters, iChapter, k, nVerses, iChapterVerses, iVerse;
   int iFirstVerse, nMaxVerse;
   QString strLine, strChapter, strWords, strVerse;
   QVector<BibleVerse> vrs;
   BibleVerse vr;

   pixq_Matrix<bool> ChapterVerse;

   //QChar ch;

   nMaxVerse = -1;
   nChapters = bc.GetChapters( );
   for( i = 1 ; i <= nChapters ; i++ ) {
      nVerses = bc.GetChapterVerseNumber( i );
      nMaxVerse = max( nMaxVerse, nVerses );
   }

   // chapter verse table - initialize!

   if( !ChapterVerse.Alloc( nChapters+1, nMaxVerse+1 ) ) {
      goto PIX_EXIT;
   }

   ChapterVerse.Fill( false );
   //

   iChapter = -1;
   nLines = (int)strLines.size( );
   vrs.clear( );
   for( i = 0 ; i < nLines ; i++ ) {
      strLine = strLines.at( i );
      strLine = strLine.trimmed( );

      nSize = (int)strLine.size( );
      if( 0 >= nSize ) continue;

      if( !DecomposeShinkai3Verse( bc, iBook, strLine, iChapter, iVerse, strWords ) ) {
         goto PIX_EXIT;
      }

      if( 0 > iVerse ) {
         // chapter is set
         continue;
      }

      if( iChapter > nChapters ) {
         // something is wrong!t
         goto PIX_EXIT;
      }

      nVerses = bc.GetChapterVerseNumber( iChapter );
      if( iVerse > nVerses ) {
         // something is wrong!t
         goto PIX_EXIT;
      }

      nVerses = bc.GetChapterVerseNumber( iChapter );
      if( iVerse > nVerses ) {
         // something is wrong!t
         goto PIX_EXIT;
      }

      // is it already set?
      bSet = ChapterVerse.Get( iChapter, iVerse );
      if( bSet ) {
         // this verse is already set!
         goto PIX_EXIT;
      }

      vr.Set( iChapter, iVerse, strWords );
      vrs.push_back( vr );
      ChapterVerse.Set( iChapter, iVerse, true );
   }

   // check whether all verses exists

   for( i = 1 ; i <= nChapters ; i++ ) {
      
      if( 18 == iBook ) {
         // Psalms
         if( bc.GetVerseZero( i ) ) {
            iFirstVerse = 0;
         } else {
            iFirstVerse = 1;
         }
      } else {
         iFirstVerse = 1;
      }

      nVerses = bc.GetChapterVerseNumber( i );
      for( k = iFirstVerse ; k <= nVerses ; k++ ) { 
         bSet = ChapterVerse.Get( iChapter, iVerse );
         if( !bSet ) {
            // the verse is not set!
            goto PIX_EXIT;
         }
      }
   }

   //
   // Psalms
   // check whether verse 0 exists for the chapters it should exist
   // add black verse 0

   // 

   nVerses = (int)vrs.size( );

   for ( i = 0 ; i < nChapters ; i++ ) {
      iChapter = i + 1;

      writer.writeStartElement( "Chapter" ); // Chapter

      strChapter.sprintf( "%d", iChapter );
      writer.writeAttribute( "ID", strChapter );

      iChapterVerses = bc.GetChapterVerseNumber( iChapter );
      strVerse.sprintf( "%d", iChapterVerses );
      writer.writeAttribute( "Verses", strVerse );

      //
      // Psalm, verse 0 exists?
      //
      if( 18 == iBook ) {
         bVerseZero = bc.GetVerseZero( i );
      } else {
         bVerseZero = false;
      }

      bFirstVerse = true;
      for( k = 0 ; k < nVerses ; k++ ) {
         vr = vrs.at( k );
         if( iChapter != vr.GetChapter( ) ) {
            continue;
         }

         if( bFirstVerse ) {
            if( bVerseZero ) {
            
               // verse zero should exist and this verse should be zero
               iVerse = vr.GetVerse( );
               if( 0 != iVerse ) {
                  // add pseudo verse zero
                  writer.writeStartElement( "Verse" ); // Verse
                  writer.writeAttribute( "ID", "0" );
                  //writer.writeTextElement( "Words", "-" ); 
                  writer.writeAttribute( "Words", "-" ); 
                  writer.writeEndElement(); // Verse
               }
            }
            bFirstVerse = false;
         }      

         writer.writeStartElement( "Verse" ); // Verse

         iVerse = vr.GetVerse( );
         strVerse.sprintf( "%d", iVerse );
         writer.writeAttribute( "ID", strVerse );
   
         strWords = vr.GetWord( );
         //writer.writeTextElement( "Words", strWords ); 
         writer.writeAttribute( "Words", strWords ); 

         writer.writeEndElement(); // Verse

      }

      writer.writeEndElement(); // Chapter
   }

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}


//
// V3ŁiuOjXMLt@C
//
bool hikari_window::ConvertShinkai3BibleXml( void ) 
{
   bool bRetCode = false;
   bool bStatus;
   int iVersion, iBook, nLines, nChapters, nSize, nBooks;
  
   QString strXmlFile, strError, strMessage, strFile;
   QString strChapters, strLine, strBookNum, strBookName, strVersion;
   //QString strXmlFile;
   //QString strBookNum, strBookName;

   BibleBook bc;

   QFile *in_file = NULL;
   QFile *out_file = NULL;
   QStringList strLines;
   QXmlStreamWriter writer;

   BibleAllBooks *pBL = GetBibleBooks( );

   // create output xml file

   iVersion = pixq_Properties::BibleShinkaiyaku3;
   strXmlFile = GetBibleXmlFile_PC( iVersion );

   out_file = new QFile( strXmlFile );
   if ( !out_file->open( QFile::WriteOnly | QFile::Text ) ) {
      strMessage = "ERROR! Failed to Create Temporary File:\n" + strXmlFile;
      QMessageBox::critical( NULL, _strApplication, strMessage );
      goto PIX_EXIT;
   }

   writer.setAutoFormatting(true);
   writer.setDevice( out_file );

   // write contents

   writer.writeStartDocument();
   //writer.writeDTD("<!DOCTYPE occasions>");

   strVersion = pixq_Properties::GetBibleVersionName( iVersion );
   writer.writeStartElement( "hikari" );
   writer.writeAttribute( "version", "1.0" );
   writer.writeAttribute( "data", "bibles" );
   writer.writeAttribute( "translation", strVersion );

   // read each book

   nBooks = pBL->GetLength( );
   for( iBook = 0 ; iBook < nBooks ; iBook++ ) {

      bc = pBL->GetBook( iBook );

      // open input book file and read all lines into list
      strFile = pBL->GetVersionPath( iVersion, iBook );
      
      in_file = new QFile( strFile );
      if ( !in_file->open( QIODevice::ReadOnly | QIODevice::Text ) ) {
         strError = "ERROR! Failed to open file " + strFile + ".";
         QMessageBox::critical( NULL, _strApplication, strError );
         goto PIX_EXIT;
      }

      strLines.clear( );

      bStatus = true;
      while( bStatus ) {
         
         if( in_file->atEnd( ) ) {
			   bStatus = false;
	   	   continue;
         }

         strLine = in_file->readLine( );
         strLine = strLine.trimmed( );
         nSize = (int)strLine.size( );
         if( 0 >= nSize ) continue;

         strLines.append( strLine );
         
      }

      nLines = (int)strLines.size( );

      // close in file
      in_file->close( );
      in_file = NULL;

      // write book info

      writer.writeStartElement( "Book" );

      strBookNum.sprintf( "%d", iBook + 1 );
      writer.writeAttribute( "ID", strBookNum );

      strBookName = bc.GetVersionBookName( iVersion );
      writer.writeAttribute( "Name", strBookName );

      nChapters = bc.GetChapters( );
      strChapters.sprintf( "%d", nChapters );
      writer.writeAttribute( "Chapters", strChapters );

      // write lines

      if( !WriteShinkai3BookXml( bc, iBook, strLines, writer ) ) {
         goto PIX_EXIT;
      }

      // end of the book

      writer.writeEndElement(); // Book
   }

   // end the last section

   writer.writeEndElement();

   // create output xml file

   writer.writeEndDocument();

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   if( in_file ) {
      in_file->close( );
      in_file = NULL;
   }
   if( out_file ) {
      out_file->close( );
      out_file = NULL;
   }
   return bRetCode;
}
