#include <pixqt_common.h>
#include <piximg_common.h>

#include <pixqtlib.h>
using namespace _pix_plot_qt_framework;

#include <piximglib.h>
using namespace _pix_plot_img_framework;

#include "scope_common.h"
#include "scope_panes.h"
#include "scope_monitor.h"
#include "scope_window.h"

ChartPane::ChartPane(QWidget *parent)
    : QWidget(parent)
{
   _pMainWindow = (scope_window*)NULL;

   _bPaneInitialized = false;
   _iChartType = ChartTypeNone;
   
   createActions( );

   return;
}

ChartPane::~ChartPane()
{
   return;
}

void ChartPane::chart_menu( void )
{
    QAction *action = qobject_cast<QAction *>(sender());
    int iAction = action->data().toInt( );

   if( !_bPaneInitialized ) {
      goto PIX_EXIT;
   }

    // iAction should be one of ChartTypeToPlot

   switch( iAction ) {
   case ChartTypeNone:
      goto PIX_EXIT;
      break;
      
   case ChartTypeRgb0Histogram: 
   case ChartTypeRgb1Histogram: 
      //
   case ChartTypeRedHistogram:
   case ChartTypeGrnHistogram:
   case ChartTypeBluHistogram:   
      //
   case ChartTypeRedCorrelation:
   case ChartTypeGrnCorrelation:   
   case ChartTypeBluCorrelation:
      _iChartType = iAction;
      PlotChart( );
      break;
    }

PIX_EXIT:
    return;
}

void ChartPane::createActions( void )
{
    _rgb0_histogram_act = new QAction( tr( "First Image Histogram" ), this );
    _rgb0_histogram_act->setData( (int)ChartTypeRgb0Histogram );
    _rgb0_histogram_act->setCheckable( true );
    connect( _rgb0_histogram_act, SIGNAL(triggered()), this, SLOT(chart_menu()));

    _rgb1_histogram_act = new QAction( tr( "Second Image Histogram" ), this );
    _rgb1_histogram_act->setData( (int)ChartTypeRgb1Histogram );
    _rgb1_histogram_act->setCheckable( true );
    connect( _rgb1_histogram_act, SIGNAL(triggered()), this, SLOT(chart_menu()));
    
    _red_histogram_act = new QAction( tr( "Red Histogram" ), this );
    _red_histogram_act->setData( (int)ChartTypeRedHistogram );
    _red_histogram_act->setCheckable( true );
    connect( _red_histogram_act, SIGNAL(triggered()), this, SLOT(chart_menu()));
    
    _grn_histogram_act = new QAction( tr( "Green Histogram" ), this );
    _grn_histogram_act->setData( (int)ChartTypeGrnHistogram );
    _grn_histogram_act->setCheckable( true );
    connect( _grn_histogram_act, SIGNAL(triggered()), this, SLOT(chart_menu()));
    
    _blu_histogram_act = new QAction( tr( "Blue Histogram" ), this );
    _blu_histogram_act->setData( (int)ChartTypeBluHistogram );
    _blu_histogram_act->setCheckable( true );
    connect( _blu_histogram_act, SIGNAL(triggered()), this, SLOT(chart_menu()));

    //

    _red_correlation_act = new QAction( tr( "Red Correlation" ), this );
    _red_correlation_act->setData( (int)ChartTypeRedCorrelation );
    _red_correlation_act->setCheckable( true );
    connect( _red_correlation_act, SIGNAL(triggered()), this, SLOT(chart_menu()));
    
    _grn_correlation_act = new QAction( tr( "Green Correlation" ), this );
    _grn_correlation_act->setData( (int)ChartTypeGrnCorrelation );
    _grn_correlation_act->setCheckable( true );
    connect( _grn_correlation_act, SIGNAL(triggered()), this, SLOT(chart_menu()));
    
    _blu_correlation_act = new QAction( tr( "Blue Correlation" ), this );
    _blu_correlation_act->setData( (int)ChartTypeBluCorrelation );
    _blu_correlation_act->setCheckable( true );
    connect( _blu_correlation_act, SIGNAL(triggered()), this, SLOT(chart_menu()));

   return;
}

void ChartPane::contextMenuEvent(QContextMenuEvent *event)
{
   scope_window *pMain = (scope_window*)GetMainWindow( );
   QList<QAction*> list_actions;
   bool bValid;
   int i, nActions;

   QMenu menu(this);
   int iAction = _iChartType;

   list_actions.clear( );
   list_actions.push_back( _rgb0_histogram_act );
   list_actions.push_back( _rgb1_histogram_act );

   list_actions.push_back( _red_histogram_act );
   list_actions.push_back( _grn_histogram_act );
   list_actions.push_back( _blu_histogram_act );

   list_actions.push_back( _red_correlation_act );
   list_actions.push_back( _grn_correlation_act );
   list_actions.push_back( _blu_correlation_act );

   nActions = (int)list_actions.size( );
   for( i = 0 ; i < nActions ; i++ ) {
      menu.addAction( (QAction*)list_actions.at( i ) );
   }

   // check current

   bValid = pMain->ValidPaneDisplayMode( );

   for( i = 0 ; i < nActions ; i++ ) {
      ((QAction*)list_actions.at( i ) )->setChecked( false );
      ((QAction*)list_actions.at( i ) )->setEnabled( bValid );
   }
   
   switch( iAction ) {      
   case ChartTypeRgb0Histogram: 
      _rgb0_histogram_act->setChecked( true );
      break;

   case ChartTypeRgb1Histogram: 
      _rgb1_histogram_act->setChecked( true );
      break;

   case ChartTypeRedHistogram:
      _red_histogram_act->setChecked( true );
      break;

   case ChartTypeGrnHistogram:
      _grn_histogram_act->setChecked( true );
      break;

   case ChartTypeBluHistogram:  
      _blu_histogram_act->setChecked( true );
      break;

   case ChartTypeRedCorrelation:
      _red_correlation_act->setChecked( true );
      break;

   case ChartTypeGrnCorrelation: 
      _grn_correlation_act->setChecked( true );
      break;

   case ChartTypeBluCorrelation:
      _blu_correlation_act->setChecked( true );
      break;
   }

   menu.exec(event->globalPos());

   return;
}

void ChartPane::resizeEvent( QResizeEvent * /* eve */)
{
   if( !_bPaneInitialized ) {
      goto PIX_EXIT;
   }
   
   PlotChart( );
   
PIX_EXIT:
   return;
}

void ChartPane::paintEvent(QPaintEvent * /* event */)
{
   if( !_bPaneInitialized ) {
      goto PIX_EXIT;
   }
   
   PlotChart( );

PIX_EXIT:
   return;
}


void ChartPane::PlotNextChart( void )
{
   int iAction = _iChartType;

   switch( iAction ) {
   case ChartTypeNone:
      goto PIX_EXIT;
      break;
      
   case ChartTypeRgb0Histogram: 
      iAction = ChartTypeRgb1Histogram;
      break;

   case ChartTypeRgb1Histogram: 
      iAction = ChartTypeRedHistogram;
      break;

      //
   case ChartTypeRedHistogram:
      iAction = ChartTypeGrnHistogram;
      break;

   case ChartTypeGrnHistogram:
      iAction = ChartTypeBluHistogram;
      break;

   case ChartTypeBluHistogram:  
      iAction = ChartTypeRedCorrelation;
      break;

      //
   case ChartTypeRedCorrelation:
      iAction = ChartTypeGrnCorrelation;
      break;

   case ChartTypeGrnCorrelation: 
      iAction = ChartTypeBluCorrelation;
      break;

   case ChartTypeBluCorrelation:
      iAction = ChartTypeRgb0Histogram;
      break;             
   }

   _iChartType = iAction;
   PlotChart( );

PIX_EXIT:
   return;
}

void ChartPane::PlotPrevChart( void )
{
   int iAction = _iChartType;

   switch( iAction ) {
   case ChartTypeNone:
      goto PIX_EXIT;
      break;
      
   case ChartTypeRgb0Histogram: 
      iAction = ChartTypeBluCorrelation;
      break;

   case ChartTypeRgb1Histogram: 
      iAction = ChartTypeRgb0Histogram;
      break;

   case ChartTypeRedHistogram:
      iAction = ChartTypeRgb1Histogram;
      break;

   case ChartTypeGrnHistogram:
      iAction = ChartTypeRedHistogram;
      break;

   case ChartTypeBluHistogram:  
      iAction = ChartTypeGrnHistogram;
      break;

   case ChartTypeRedCorrelation:
      iAction = ChartTypeBluHistogram;
      break;

   case ChartTypeGrnCorrelation: 
      iAction = ChartTypeRedCorrelation;
      break;

   case ChartTypeBluCorrelation:
      iAction = ChartTypeGrnCorrelation;
      break;
    }

   _iChartType = iAction;
   PlotChart( );

PIX_EXIT:
   return;
}

void ChartPane::keyPressEvent( QKeyEvent *eve )
{
   //scope_window *pMain = (scope_window*)GetMainWindow( );

   int iKey = eve->key( );

	switch( iKey ) {
	case Qt::Key_Right:	
	case Qt::Key_Up:	
      PlotNextChart( );
		break;

	case Qt::Key_Left:
	case Qt::Key_Down:
      PlotPrevChart( );
		break;
   }

   update( );
   return;
}

void ChartPane::wheelEvent( QWheelEvent *eve )
{
   int iDelta;

   if( !ValidChartDisplayMode( ) ) {
      goto PIX_EXIT;
   }

   iDelta = eve->delta( );

PIX_EXIT:
   return;
}

//
// mouse button pressed
//
void ChartPane::mousePressEvent( QMouseEvent *eve )
{
   if( !ValidChartDisplayMode( ) ) {
      goto PIX_EXIT;
   }

   switch( eve->button( ) ) {
   case Qt::LeftButton:   
      // left click
      break;
      
   case Qt::RightButton:
      
      break;
   }

PIX_EXIT:
   return;
}

void ChartPane::mouseMoveEvent( QMouseEvent *eve )
{
   QSize rect = size( );

   //int iMouseX = eve->x();
   //int iMouseY = rect.height( ) - eve->y();

   if( ValidChartDisplayMode( ) ) {
      setCursor( Qt::ArrowCursor );
   } else {
      QWidget::mouseMoveEvent( eve );
      goto PIX_EXIT;
   }
      
PIX_EXIT:
   return;
}

void ChartPane::closeEvent( QCloseEvent * /*eve*/ )
{
   _iChartType = ChartTypeNone;
}


void ChartPane::Init( void )
{
   qreal grayValue;
   QColor back_color;
   QPalette pal;
   
   //setBackgroundRole( QPalette::Base );

   grayValue = 0.680f;
   back_color.setRgbF( grayValue, grayValue, grayValue );
   
   pal = palette();
   pal.setColor( QPalette::Background, back_color );
   setPalette( pal );

   setAutoFillBackground( true );

   //

   setSizePolicy( QSizePolicy::Ignored, QSizePolicy::Ignored );

   setMinimumSize( 50, 50 );

   setFocusPolicy( Qt::StrongFocus );
   setMouseTracking( true );

   ResetPane( );

   //
   // pane parameters
   //

   _bPaneInitialized = true;

   return;
}

void ChartPane::ResetPane( void )
{   
   _iChartType = ChartTypeNone;

   PlotChart( );

   // --- DOne ---
   return;
}

bool ChartPane::ValidChartDisplayMode( void )
{
   bool bRetCode = false;
   
   switch( _iChartType ) {

   case ChartTypeNone:
      goto PIX_EXIT;
      break;

   case ChartTypeRgb0Histogram:
   case ChartTypeRgb1Histogram:
      //
   case ChartTypeRedHistogram:
   case ChartTypeGrnHistogram:
   case ChartTypeBluHistogram:
      //
   case ChartTypeRedCorrelation:
   case ChartTypeGrnCorrelation:
   case ChartTypeBluCorrelation:
      break;

   default:
      goto PIX_EXIT;
      break;
   }

   // --- DONE ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}

void ChartPane::PlotChart( void )
{   
   switch( _iChartType ) {

   case ChartTypeNone:
      PlotBlank( );
      break;

   case ChartTypeRedCorrelation:
   case ChartTypeGrnCorrelation:
   case ChartTypeBluCorrelation:
      PlotCorrChart( );
      break;

   case ChartTypeRgb0Histogram:
   case ChartTypeRgb1Histogram:
      PlotRgbHistgram( );
      break;

   case ChartTypeRedHistogram:
   case ChartTypeGrnHistogram:
   case ChartTypeBluHistogram:
      PlotColorHistgram( );
      break;

   default:
      goto PIX_EXIT;
      break;
   }

   SetDockLabel( );

PIX_EXIT:
   return;
}

//
// black out the chart area
//
void ChartPane::PlotBlank( void )
{
   // check conditions
   if( !_bPaneInitialized ) {
      goto PIX_EXIT;
   }

PIX_EXIT:
   return;
}

void ChartPane::PlotCorrChart( void )
{
   scope_window *pMain = (scope_window*)GetMainWindow( );
   pixq_FrameImageStat *pStat = pMain->GetImageStat( );

   int iViewX, iViewY, iViewW, iViewH, iViewCenX, iViewCenY;
   int iViewFrame, i, nPoints[2], iFrameWidth, iFrameHeight, iFrameSize;
   
   double dXX0, dXX1, dYY0, dYY1, dX, dY;
   double dXScale, dYScale, dXShift, dYShift;
   vector<unsigned short> *pPixelList[2];

   QSize rect;
   QRect vp;
   QRectF win;
   QPointF *pnts = NULL;
   QColor pen_color;
   QBrush frame_brush;
   QPen frame_pen, pen;

   QPainter painter( this );

   // check conditions

   if( !_bPaneInitialized ) {
      goto PIX_EXIT;
   }

   // set up chart and draw frames

   rect = size( );

   dXX0 = 0.0;
   dXX1 = 65535.0;
   dYY0 = 0.0;
   dYY1 = 65535.0;

   win.setX( dXX0 );
   win.setY( dYY1 );
   win.setWidth( dXX1 - dXX0 );
   win.setHeight( dYY0 - dYY1 );

   //

   iViewFrame = 5;
   iFrameWidth = rect.width( ) - iViewFrame - iViewFrame;
   iFrameHeight = rect.height( ) - iViewFrame - iViewFrame;
   iFrameSize = min( iFrameWidth, iFrameHeight );
   if( 0 >= iFrameSize ) goto PIX_EXIT;

   iViewCenX = rect.width( ) / 2;
   iViewCenY = rect.height( ) / 2;

   iViewX = iViewCenX - iFrameSize / 2;
   iViewW = iFrameSize;
   iViewY = iViewCenY - iFrameSize / 2;
   iViewH = iFrameSize;

   vp.setX( iViewX );
   vp.setY( iViewY );
   vp.setWidth( iViewW );
   vp.setHeight( iViewH );
   painter.setViewport( vp );

   // scale

   painter.setWindow( vp );

   dXScale = ( (double)vp.width( ) + 1.0 ) / ( win.width( ) + 1.0 );
   dYScale = ( (double)vp.height( ) + 1.0 ) / ( win.height( ) + 1.0 );

   dXShift = (double)vp.x( ) - 0.5 - dXScale * ( win.x( ) - 0.5 );
   dYShift = (double)vp.y( ) - 0.5 - dYScale * ( win.y( ) - 0.5 );

   
   // *** draw frames ***

   frame_pen.setWidth( 0 );
   frame_pen.setStyle( Qt::SolidLine );
   frame_pen.setColor( QColor( 127, 127, 127 ) );
   painter.setPen( frame_pen );

   frame_brush.setStyle( Qt::NoBrush );
   painter.setBrush( frame_brush );

   painter.drawRect( vp );
   painter.drawLine( vp.topRight( ), vp.bottomLeft( ) );

   // *** plot chart ***

   switch( _iChartType ) {
   case ChartTypeRedCorrelation:
      pPixelList[0] = pStat->GetRedPixelList( 0 );
      pPixelList[1] = pStat->GetRedPixelList( 1 );
      pen_color.setRgbF( 1.0, 0.0, 0.0 );
      break;
   case ChartTypeGrnCorrelation:
      pPixelList[0] = pStat->GetGrnPixelList( 0 );
      pPixelList[1] = pStat->GetGrnPixelList( 1 );
      pen_color.setRgbF( 0.0, 1.0, 0.0 );
      break;
   case ChartTypeBluCorrelation:
      pPixelList[0] = pStat->GetBluPixelList( 0 );
      pPixelList[1] = pStat->GetBluPixelList( 1 );
      pen_color.setRgbF( 0.0, 0.0, 1.0 );
      break;
   default:
      goto PIX_EXIT;
      break;
   }

   nPoints[0] = (int)pPixelList[0]->size( );
   nPoints[1] = (int)pPixelList[1]->size( );

   if( 0 >= nPoints[0] ) {
      goto PIX_EXIT;
   }

   if( nPoints[0] != nPoints[1] ) {
      goto PIX_EXIT;
   }

   pnts = new QPointF[nPoints[0]];
   if( !pnts ) {
      goto PIX_EXIT;
   }

   pen.setWidth( 1 );
   pen.setStyle( Qt::SolidLine );
   pen.setJoinStyle( Qt::MiterJoin );
   pen.setCapStyle( Qt::FlatCap );
   pen.setColor( pen_color );
   painter.setPen( pen ); 

   for( i = 0 ; i < nPoints[0] ; i++ ) {
      dX = (double)pPixelList[0]->at( i );
      dY = (double)pPixelList[1]->at( i );
      
      pnts[i].setX( dXScale * dX + dXShift );
      pnts[i].setY( dYScale * dY + dYShift );
   } 

   painter.drawPoints( pnts, nPoints[0] );


   // --- draw frame ---
PIX_EXIT:
   if( pnts ) {
      delete[] pnts;
      pnts = NULL;
   }
   return;
}

void ChartPane::SetDockLabel( void )
{   
   scope_window *pMain = (scope_window*)GetMainWindow( );
   scope_dock *pDock = pMain->GetDockChart( );
   QString strLabel;

   switch( _iChartType ) {

   case ChartTypeNone:
      goto PIX_EXIT;
      break;

   case ChartTypeRedCorrelation:
      strLabel = "Correlation : Red";
      break;
   case ChartTypeGrnCorrelation:
      strLabel = "Correlation : Green";
      break;
   case ChartTypeBluCorrelation:
      strLabel = "Correlation : Blue";
      break;

   case ChartTypeRgb0Histogram:
      strLabel = "Histogram : First Image";
      break;
   case ChartTypeRgb1Histogram:
      strLabel = "Histogram : Second Image";
      break;

   case ChartTypeRedHistogram:
      strLabel = "Histogram : Red Channel";
      break;
   case ChartTypeGrnHistogram:
      strLabel = "Histogram : Green Channel";
      break;
   case ChartTypeBluHistogram:
      strLabel = "Histogram : Blue Channel";
      break;

   default:
      goto PIX_EXIT;
      break;
   }

   pDock->setWindowTitle( strLabel );

PIX_EXIT:
   return;
}

void ChartPane::DrawColorBar(int iViewX,
                             int iViewW,
                             int iViewY,
                             int iViewH )
{
   QPainter painter( this );
   QPen pen;
   QBrush brush;
   QRect vp, win;
   QLinearGradient linearGradient;

   // check conditions

   if( !_bPaneInitialized ) {
      goto PIX_EXIT;
   }

   // *** draw color bar chart

   vp.setX( iViewX );
   vp.setY( iViewY );
   vp.setWidth( iViewW );
   vp.setHeight( iViewH );
   painter.setViewport( vp ); 
   painter.setWindow( vp );

   // *** draw color chart ***

   pen.setWidth( 1 );
   pen.setStyle( Qt::SolidLine );
   pen.setColor( QColor( 63, 63, 63 ) );
   painter.setPen( pen );

   linearGradient.setStart( iViewX, iViewY );
   linearGradient.setFinalStop( iViewX + iViewW, iViewY + iViewH );
   linearGradient.setColorAt( 0.0, Qt::black );
   linearGradient.setColorAt( 1.0, Qt::white );
   brush = linearGradient;
   painter.setBrush( brush );
   painter.drawRect( vp );

PIX_EXIT:
   return;
}

void ChartPane::DrawHistogram( double dMaxHist,
                              const vector<double> &pBinLow,
                              const vector<double> &pBinHigh,
                              int nHists, // [i] 2 or 3
                              QColor pen_color0,
                              QColor pen_color1,
                              QColor pen_color2,
                              const vector<int> *pHist0,
                              const vector<int> *pHist1,
                              const vector<int> *pHist2 )
{
   int i, nHistogramBins, iViewFrame, iViewX, iViewY0, iViewY1, iViewW, iViewH;
   int k, iGrayChartHeight;
   
   double dXX0, dXX1, dYY0, dYY1, dX0, dX1, dY0, dY1;
   double dXFrame, dYFrame;
   double dFreq, dCen, dVFreq, dVCen, dLow, dHigh;

   QColor pen_colors[3];
   QPainter painter( this );
   QSize sz;
   QRect vp;
   QRectF win;
   QPointF *pnts = NULL;

   QPen frame_pen, pen;
   QBrush frame_brush, back_brush;
   QLinearGradient grad;

   double dXScale, dYScale, dXShift, dYShift;
   const vector<int> *pHist[3];

   // set up chart and draw frames
   
   if( !_bPaneInitialized ){
      goto PIX_EXIT;
   }

   // set viewport

   sz = size( );
   nHistogramBins = (int)pBinLow.size( );

   iGrayChartHeight = 12;
   iViewFrame = 5;
   iViewX = iViewFrame;
   iViewW = sz.width( ) - iViewFrame - iViewFrame;

   iViewY0 = iViewFrame;
   iViewH = sz.height( ) - iViewFrame - iViewFrame - iGrayChartHeight;
   iViewY1 = iViewY0 + iViewH - 1;
   
   vp.setX( iViewX );
   vp.setY( iViewY0 );
   vp.setWidth( iViewW );
   vp.setHeight( iViewH );
   painter.setViewport( vp );

   //

   dX0 = pBinLow[ 0 ];
   dX1 = pBinHigh[ nHistogramBins-1 ];
   dY0 = 0;
   dY1 = dMaxHist;

   dXFrame = 2.0;
   dYFrame = dXFrame * dMaxHist / 65535.0;

   dXX0 = dX0 - dXFrame;
   dXX1 = dX1 + dXFrame;
   dYY0 = dY0 - dYFrame;
   dYY1 = dY1 + dYFrame;

   win.setX( dXX0 );
   win.setY( dYY1 );
   win.setWidth( dXX1 - dXX0 );
   win.setHeight( dYY0 - dYY1 );
   
   // do not use setWindows for very large area
   // it makes ugly picture!
   //painter.setWindow( win );
   
   painter.setWindow( vp );

   dXScale = ( (double)vp.width( ) + 1.0 ) / ( win.width( ) + 1.0 );
   dYScale = ( (double)vp.height( ) + 1.0 ) / ( win.height( ) + 1.0 );

   dXShift = (double)vp.x( ) - 0.5 - dXScale * ( win.x( ) - 0.5 );
   dYShift = (double)vp.y( ) - 0.5 - dYScale * ( win.y( ) - 0.5 );

   // *** draw chart ***

   if( 0 >= nHistogramBins ) {
      goto PIX_EXIT;
   }
   
   // first image

   nHistogramBins = (int)pBinLow.size( );

   pHist[0] = pHist0;
   pHist[1] = pHist1;
   pHist[2] = pHist2;

   pen_colors[0] = pen_color0;
   pen_colors[1] = pen_color1;
   pen_colors[2] = pen_color2;

   pnts = new QPointF[nHistogramBins];
   if( !pnts ) {
      goto PIX_EXIT;
   }

   pen.setWidth( 1 );
   pen.setStyle( Qt::SolidLine );
   pen.setJoinStyle( Qt::MiterJoin );
   pen.setCapStyle( Qt::FlatCap );

   for( k = 0 ; k < nHists ; k++ ) {
      
      pen.setColor( pen_colors[k] );
      painter.setPen( pen ); 
      
      for( i = 0 ; i < nHistogramBins ; i++ ) {

         dLow = pBinLow[i];
         dHigh = pBinHigh[i];
         dCen = ( dLow + dHigh ) / 2.0;

         dFreq = (double)pHist[k]->at( i );

         dVCen = dXScale * dCen + dXShift;
         dVFreq = dYScale * dFreq + dYShift;

         pnts[i].setX( dVCen );
         pnts[i].setY( dVFreq );
      }

      painter.drawPolyline( pnts, nHistogramBins );
   }    

   // *** draw frames ***

   frame_pen.setWidth( 0 );
   frame_pen.setStyle( Qt::SolidLine );
   frame_pen.setColor( QColor( 127, 127, 127 ) );
   painter.setPen( frame_pen );

   frame_brush.setStyle( Qt::NoBrush );
   painter.setBrush( frame_brush );
   painter.drawRect( vp );
   //painter.drawRect( vp.x( ), vp.y( ), vp.width( )-1, vp.height( )-1  );

   // *** draw color bar chart

   DrawColorBar( iViewX, iViewW, iViewY1, iGrayChartHeight );

PIX_EXIT:
   if( pnts ) {
      delete[] pnts;
      pnts = NULL;
   }
   return;
}

void ChartPane::PlotRgbHistgram( void )
{
   scope_window *pMain = (scope_window*)GetMainWindow( );
   pixq_FrameImageStat *pStat = pMain->GetImageStat( );
   
   int iChan;
   double dMaxHist;
   QColor pen_colors[3];
   vector<double> *pBinLow = NULL;
   vector<double> *pBinHigh = NULL;
   vector<int> *pColorsHist0 = NULL;
   vector<int> *pColorsHist1 = NULL;
   vector<int> *pColorsHist2 = NULL;

   // check conditions

   if( !_bPaneInitialized ) {
      goto PIX_EXIT;
   }

   switch( _iChartType ) {
   case ChartTypeRgb0Histogram:
      iChan = 0;
      break;
   case ChartTypeRgb1Histogram:
      iChan = 1;
      break;
   default:
      goto PIX_EXIT;
      break;
   }

   dMaxHist = (double)pStat->GetSideMaxFrequency( iChan );
   if( 0 >= dMaxHist ) {
      dMaxHist = 100.0;
   } else {
      dMaxHist = dMaxHist * 1.05;
   }

   pBinLow = pStat->GetBinLowValue( );
   pBinHigh = pStat->GetBinHighValue( );

   // *** select histogram ***

   pColorsHist0 = pStat->GetRedHistogram( iChan );
   pColorsHist1 = pStat->GetGrnHistogram( iChan );
   pColorsHist2 = pStat->GetBluHistogram( iChan );

   // draw it!

   pen_colors[0] = Qt::red;
   pen_colors[1] = Qt::green;
   pen_colors[2] = Qt::blue;

   DrawHistogram( dMaxHist, *pBinLow, *pBinHigh, 
      3,
      pen_colors[0], pen_colors[1], pen_colors[2],
      pColorsHist0, pColorsHist1, pColorsHist2 );

   // --- Done ---
PIX_EXIT:
   return;
}

//
// plot histogram for channel
//
void ChartPane::PlotColorHistgram( void )
{
   scope_window *pMain = (scope_window*)GetMainWindow( );
   pixq_FrameImageStat *pStat = pMain->GetImageStat( );
   qreal col1, col2;
   int iChan;

   QColor pen_colors[3];
   double dMaxHist;
   vector<double> *pBinLow = NULL;
   vector<double> *pBinHigh = NULL;
   vector<int> *pForeHist = NULL;
   vector<int> *pBackHist = NULL;

   // check conditions

   if( !_bPaneInitialized ) {
      goto PIX_EXIT;
   }

   // *** select histogram ***

   col1 = 1.0;
   col2 = 0.6;

   switch( _iChartType ) {
   case ChartTypeRedHistogram:
      iChan = 0;
      pForeHist = pStat->GetRedHistogram( 0 );
      pBackHist = pStat->GetRedHistogram( 1 );
      pen_colors[0].setRgbF( col1, 0.0, 0.0 );
      pen_colors[1].setRgbF( col1, col2, col2 );
      pen_colors[2].setRgbF( 0.0, 0.0, 0.0 );
      break;
   case ChartTypeGrnHistogram:
      iChan = 1;
      pForeHist = pStat->GetGrnHistogram( 0 );
      pBackHist = pStat->GetGrnHistogram( 1 );
      pen_colors[0].setRgbF( 0.0, col1, 0.0 );
      pen_colors[1].setRgbF( col2, col1, col2 );
      pen_colors[2].setRgbF( 0.0, 0.0, 0.0 );
      break;
   case ChartTypeBluHistogram:
      iChan = 2;
      pForeHist = pStat->GetBluHistogram( 0 );
      pBackHist = pStat->GetBluHistogram( 1 );
      pen_colors[0].setRgbF( 0.0, 0.0, col1 );
      pen_colors[1].setRgbF( col2, col2, col1 );
      pen_colors[2].setRgbF( 0.0, 0.0, 0.0 );
      break;
   }
   
   dMaxHist = (double)pStat->GetColorMaxFrequency( iChan );
   if( 0 >= dMaxHist ) {
      dMaxHist = 100.0;
   } else {
      dMaxHist = dMaxHist * 1.05;
   }

   pBinLow = pStat->GetBinLowValue( );
   pBinHigh = pStat->GetBinHighValue( );

   // draw it!

   DrawHistogram( dMaxHist, *pBinLow, *pBinHigh, 
      2,
      pen_colors[0], pen_colors[1], pen_colors[2],
      pForeHist, pBackHist, (vector<int>*)NULL );

   // --- Done ---
PIX_EXIT:
   return;
}
