#include <pixqt_common.h>
#include <piximg_common.h>

#include <pixqtlib.h>
using namespace _pix_plot_qt_framework;

#include <piximglib.h>
using namespace _pix_plot_img_framework;

#include "sampler_common.h"
#include "sampler_panes.h"
#include "sampler_monitor.h"
#include "sampler_window.h"
#include "ui_sampler_window.h"

//
// apply one dimension DCT 
//
static bool bApply1DDct( int nBlockSize, 
                        float *f1DDct, // [i/o] 1-D DCT coeficient
                        bool bInverse, 
                        pixq_Matrix<float> *mfKitei ) // [o] kiteinfunction
{
   bool bRetCode = false;
   int i, k, n2;
   float *fBuffer = NULL;

   n2 = nBlockSize * 2;
   fBuffer = new float[n2];
   if( !fBuffer ) {
      goto PIX_EXIT;
   }

   if( bInverse ) {
      // inverse treanformation
      for( i = 0 ; i < nBlockSize ; i++ ) {
         fBuffer[i] = 0.0f;
         for( k = 0 ; k < nBlockSize ; k++ ) {
            fBuffer[i] = fBuffer[i] + f1DDct[i] * mfKitei->Get( k, i );
         }
      }
      // copy the conversion results
      for( i = 0 ; i < nBlockSize ; i++ ) {
         f1DDct[i] = fBuffer[i];
      }
   } else {
      // copy input data
      for( i = 0 ; i < nBlockSize ; i++ ) {
         fBuffer[i] = f1DDct[i];
      }
      //
      for( k = 0 ; k < nBlockSize ; k++ ) {
         f1DDct[k] = 0.0f;
         for( i = 0 ; i < nBlockSize ; i++ ) {
            f1DDct[i] = f1DDct[i] + fBuffer[i] * mfKitei->Get( k, i );
         }        
      }
   }

   // --- Done ---
   bRetCode = true;
PIX_EXIT:
   if( fBuffer ) {
      delete[] fBuffer;
      fBuffer = NULL;
   }
   return bRetCode;
}

//
// calculate basic function for the DCT
//
// history:
// 2009, taken from "digital gazou shori nyuumon"
//
static void CreateDctBaseKitei( int nBlockSize,
                               pixq_Matrix<float> *fPhi ) // [o] must be allocated for nBlockSize * nBlockSize
{
   short i, k;
   float fA, fB, fBase;
   double dI, dK, dSize, dSize2;

   dSize = (double)nBlockSize;
   dSize2 = (double)nBlockSize * 2.0;
   fA = (float)sqrt( 1.0 / dSize );
   fB = (float)sqrt( 2.0 / dSize );

   // k = 0
   k = 0;
   for( i = 0 ; i < nBlockSize ; i++ ) {
      fPhi->Set( k, i, fA );
   }

   // k > 0
   for( k = 1 ; k < nBlockSize ; k++ ) { // row
      dK = (double)k;
      for( i = 0 ; i < nBlockSize ; i++ ) {
         dI = (double)i;
         fBase = fB * (float)cos( ( 2.0 * dI + 1.0 ) * dK * M_PI / dSize2 );
         fPhi->Set( k, i, fBase );
      }
   }

   // --- Done ---
   return;
}

float sampler_window::fGetRed( int iRow, int iCol )
{
   float fValue = (float)_ImageInMtx[0].Get( iRow, iCol );

   return fValue;
}

float sampler_window::fGetGreen( int iRow, int iCol )
{
   float fValue = (float)_ImageInMtx[1].Get( iRow, iCol );

   return fValue;
}

float sampler_window::fGetBlue( int iRow, int iCol )
{
   float fValue = (float)_ImageInMtx[2].Get( iRow, iCol );

   return fValue;
}

//
// create gray scale image from rgb matrix
// in float matrix
//
bool sampler_window::CreateGrayMatrix( pixq_Matrix<float> *fGray )
{
   bool bRetCode = false;
   int i, j, nWidth, nHeight;
   float fRed, fGrn, fBlu, fValue;

   nWidth = _ImageInMtx[0].GetWidth( );
   nHeight = _ImageInMtx[0].GetHeight( );

   if( 0 >= nWidth || 0 >= nHeight ) goto PIX_EXIT;

   if( !fGray->Alloc( nHeight, nWidth ) ) {
      goto PIX_EXIT;
   }

   // create grayscale matrix

   for( i = 0 ; i < nHeight ; i++ ) {
      for( j = 0 ; j < nWidth ; j++ ) {
         fRed = (float)fGetRed( i, j );
         fGrn = (float)fGetGreen( i, j );
         fBlu = (float)fGetBlue( i, j );
         fValue = GrayFromRgb_PI( fRed, fGrn, fBlu );
         fGray->Set( i, j, fValue );
      }
   }
   
   // --- Done ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}

//
// create grayscale image from (any) float matrix
// in RGB image three matrices
//
bool sampler_window::FloatMatrixGrayImage( pixq_Matrix<float> *fFft )
{
   bool bRetCode = false;
   int i, j, nWidth, nHeight;
   float fNorm, fValue;
   pixq_Matrix<float> fTmp;
   float fDet, fShift, fMin, fMax, fMean, fStdev, fRange;

   nWidth = _ImageInMtx[0].GetWidth( );
   nHeight = _ImageInMtx[0].GetHeight( );

   if( 0 >= nWidth || 0 >= nHeight ) {
      goto PIX_EXIT;
   }

   // conver amplitutde to gray scale image

   fFft->Statistics( &fMin, &fMax, &fMean, &fStdev );

   fRange = 2.5 * fStdev;
   fMin = fMean - fRange;
   fMax = fMean + fRange;
 
   if( fMin == fMax ) {
      goto PIX_EXIT;
   }

   fDet = 65536.0f / ( fMax - fMin );
   fShift = 0.5 / fDet + fMin;

   if( !fTmp.Alloc( nHeight, nWidth ) ) {
      goto PIX_EXIT;
   }

   for( i = 0 ; i < nHeight ; i++ ) {
      // copy one scanline of the matrix to buffer
      for( j = 0 ; j < nWidth ; j++ ) {
         fNorm = fFft->Get( i, j );
         fValue = fDet * ( fNorm - fShift );
         fTmp.Set( i, j, fValue );
      }
   }

   // set the gray matrix to output image matrices 
   
   if( !bSetAll( &fTmp ) ) {
      goto PIX_EXIT;
   }

   // --- Done ---
   bRetCode = true;
PIX_EXIT:
   fTmp.Free( );
   return bRetCode;
}

//
// copy contents of a matrix to all three image matrices
//
bool sampler_window::bSetAll( pixq_Matrix<float> *fPix )
{
   bool bRetCode = false;

   int iStart = 0;
   int iEnd = 2;

   int nPixWidth, nPixHeight, i, j, k, nWidth, nHeight;
   float fMax, fValue;
   unsigned short usValue;
   pix_ImageMatrix *pMtx[3];
   
   nPixWidth = fPix->GetWidth( );
   nPixHeight = fPix->GetHeight( );
   if( 0 >= nPixWidth || 0 >= nPixHeight ) goto PIX_EXIT;

   // allocate target matrix

   pMtx[0] = &_ImageDrvMtx[0];
   pMtx[1] = &_ImageDrvMtx[1];
   pMtx[2] = &_ImageDrvMtx[2];

   for( k = iStart ; k <= iEnd ; k++ ) {
      nWidth = pMtx[k]->GetWidth( );
      nHeight = pMtx[k]->GetHeight( );
      if( nWidth == nPixWidth && nHeight == nPixHeight ) continue;

      pMtx[k]->Free( );
      
      if( !pMtx[k]->Alloc( nPixHeight, nPixWidth ) ) {
         goto PIX_EXIT;
      }
   }

   // conver DFT result to gray scale image

   fMax = (float)0xFFFF;
   for( i = 0 ; i < nPixHeight ; i++ ) {
      for( j = 0 ; j < nPixWidth ; j++ ) {

         fValue = fPix->Get( i, j );
         if( 0 > fValue ) {
            usValue = 0;
         } else if( fMax < fValue ) {
            usValue = 0xFFFF;
         } else {
            usValue = (unsigned short)floor( 0.5 + fValue );
         }
         
         for( k = iStart ; k <= iEnd ; k++ ) {
            pMtx[k]->Set( i, j, usValue );
         }
      }
   }

   // --- Done ---
   bRetCode = true;
PIX_EXIT:
   return bRetCode;
}

//
// create filter image with 0 and 1 according to the filter setting in the properties
//
bool sampler_window::ApplyFilterImage( pixq_Matrix<float> *fReal,
                                    pixq_Matrix<float> *fImag,
                                    pixq_Matrix<float> *fFilter )
{
   bool bRetCode = false;
   int iFilter, i, j, nWidth, nHeight;
   float fRadius, fFliterVal, fRealVal, fImagVal;
   float fBandLow, fBandHigh, fBand;
   float *fVerPos = NULL;
   float *fHorPos = NULL;
   pixq_Properties *pEnv = GetEnv( );

   nWidth = fReal->GetWidth( );
   nHeight = fReal->GetHeight( );

   if( 0 >= nWidth || 0 >= nHeight ) {
      goto PIX_EXIT;
   }

   // check filter shape
   
   if( !fFilter->Alloc( nHeight, nWidth ) ) {
      goto PIX_EXIT;
   }

   fVerPos = new float[nHeight];
   if( !fVerPos ) {
      goto PIX_EXIT;
   }

   fHorPos = new float[nWidth];
   if( !fHorPos ) {
      goto PIX_EXIT;
   }

   for( i = 0 ; i < nHeight ; i++ ) {
      fVerPos[i] = (float)i - (float)nHeight / 2.0f;
   }

   for( j = 0 ; j < nWidth ; j++ ) {
      fHorPos[j] = (float)j - (float)nWidth / 2.0f;
   }

   // create filter according to the filter shape

   iFilter = pEnv->_DftFilterDigitalFiltersShape.GetInt( );

   switch( iFilter ) {
   case pixq_Properties::DigitalFilterLowPass:
      // low-pass filter    
      fBand = (float)pEnv->_DftFilterHighLowPassBoundary.GetDouble( );
      for( i = 0 ; i < nHeight ; i++ ) {
            for( j = 0 ; j < nWidth ; j++ ) {
             fRadius = sqrt( fHorPos[j] * fHorPos[j] + fVerPos[i] * fVerPos[i] );
   
            if( fRadius <= fBand ) {
               fFilter->Set( i, j, 1.0f );
            } else {
               fFilter->Set( i, j, 0.0f );
            }
         }
      }
      break;

   case pixq_Properties::DigitalFilterHighPass:
      // high-pass filter  
      fBand = (float)pEnv->_DftFilterHighLowPassBoundary.GetDouble( );
      for( i = 0 ; i < nHeight ; i++ ) {
            for( j = 0 ; j < nWidth ; j++ ) {
             fRadius = sqrt( fHorPos[j] * fHorPos[j] + fVerPos[i] * fVerPos[i] );
   
            if( fRadius >= fBand ) {
               fFilter->Set( i, j, 1.0f );
            } else {
               fFilter->Set( i, j, 0.0f );
            }
         }
      }
      break;

   case pixq_Properties::DigitalFilterBandPass:
      // band-pass filter     
      fBandLow = (float)pEnv->_DftFilterBandPassLowBoundary.GetDouble( );
      fBandHigh = (float)pEnv->_DftFilterBandPassHighBoundary.GetDouble( );
      for( i = 0 ; i < nHeight ; i++ ) {
            for( j = 0 ; j < nWidth ; j++ ) {
             fRadius = sqrt( fHorPos[j] * fHorPos[j] + fVerPos[i] * fVerPos[i] );
   
            if( fRadius >= fBandLow && fRadius <= fBandHigh ) {
               fFilter->Set( i, j, 1.0f );
            } else {
               fFilter->Set( i, j, 0.0f );
            }
         }
      }
      break;

   case pixq_Properties::DigitalFilterNotch:
      // notch filter  
      fBandLow = (float)pEnv->_DftFilterBandPassLowBoundary.GetDouble( );
      fBandHigh = (float)pEnv->_DftFilterBandPassHighBoundary.GetDouble( );  
      for( i = 0 ; i < nHeight ; i++ ) {
         for( j = 0 ; j < nWidth ; j++ ) {
            fRadius = sqrt( fHorPos[j] * fHorPos[j] + fVerPos[i] * fVerPos[i] );
   
            if( fRadius >= fBandHigh || fRadius <= fBandLow ) {
               fFilter->Set( i, j, 1.0f );
            } else {
               fFilter->Set( i, j, 0.0f );
            }
         }
      }
      break;
  
   default:
      goto PIX_EXIT;
   }

   // apply filter

   for( i = 0 ; i < nHeight ; i++ ) {
      for( j = 0 ; j < nWidth ; j++ ) {
         fFliterVal = fFilter->Get( i, j );

         fRealVal = fFliterVal * fReal->Get( i, j );
         fImagVal = fFliterVal * fImag->Get( i, j );

         fReal->Set( i, j, fRealVal );
         fImag->Set( i, j, fImagVal );
      }
   }
   
   // --- Done ---
   bRetCode = true;
PIX_EXIT:
   if( fVerPos ) {
      delete[] fVerPos;
      fVerPos = NULL;
   }   
   if( fHorPos ) {
      delete[] fHorPos;
      fHorPos = NULL;
   }
   return bRetCode;
}

//
// Image processing in matrix:
// fourier transform
// fft image of the first image in the second image
//
bool sampler_window::CreateSecondDftImage( void )
{
   bool bRetCode = false;
   pixq_Matrix<float> fGray;
   pixq_Matrix<float> fReal;
   pixq_Matrix<float> fImag; 
   pixq_Matrix<float> fDft;
   int i, j, nWidth, nHeight;
   float fR;
   pixq_ImageOpDft OpDft;
   
   SamplerMonitor *pMonitor = (SamplerMonitor*)GetMonitorPane( );
   QString strMessage;

   nWidth = _ImageInMtx[0].GetWidth( );
   nHeight = _ImageInMtx[0].GetHeight( );

   // check image dimension must be power of two

   if( !CheckImagePowerOfTwo( ) ) {
      pMonitor->DisplayMessage( "Image Size must be a power of two." );
      goto PIX_EXIT;
   }

   // allocate and create grayscale image in float matrix

   if( !CreateGrayMatrix( &fGray ) ) {
      goto PIX_EXIT;
   }
   
   if( !fReal.Alloc( nHeight, nWidth ) ) {
      goto PIX_EXIT;
   }
   
   if( !fImag.Alloc( nHeight, nWidth ) ) {
      goto PIX_EXIT;
   }

   for( i = 0 ; i < nHeight ; i++ ) {
      for( j = 0 ; j < nWidth ; j++ ) {
         fR = fGray.Get( i, j );
         fReal.Set( i, j, fR );
         fImag.Set( i, j, 0.0f );
      }
   }

   // get fft matrix
   
   if( !Get2DImageFft_PI( &fReal, &fImag, false ) ) {
      goto PIX_EXIT;
   }

   // conver DFT result to gray scale image

   if( !DftAmplitudeImage_PC( &fReal, &fImag, &fDft ) ) {
      goto PIX_EXIT;
   }

   // create gray scale image in second texture matrices

   if( !FloatMatrixGrayImage( &fDft ) ) {
      goto PIX_EXIT;
   }

/*
   OpDft.StoreIn( _ImageInMtx[0] );
   OpDft.StoreOut( _ImageDrvMtx[0] );

   if( !OpDft.Proc( ) ) {
      QMessageBox::warning( this, _strApplication, "ERROR! DFT Failed." );
      goto PIX_EXIT;
   }
*/
   // --- Done ---
   bRetCode = true;
PIX_EXIT: 
   fGray.Free( );
   fReal.Free( );
   fImag.Free( );
   fDft.Free( );
   return bRetCode;
}

bool sampler_window::CreateSecondDftShape( void )
{
   bool bRetCode = false;
   pixq_ImageOpDft OpDft;
   pixq_Matrix<float> fGray;
   pixq_Matrix<float> fReal;
   pixq_Matrix<float> fImag; 
   pixq_Matrix<float> fDft;
   pixq_Matrix<float> fFilter; 
   int i, j, nWidth, nHeight;
   float fVal;

   SamplerMonitor *pMonitor = (SamplerMonitor*)GetMonitorPane( );
   QString strMessage;

   nWidth = _ImageInMtx[0].GetWidth( );
   nHeight = _ImageInMtx[0].GetHeight( );

   // check image dimension must be power of two

   if( !CheckImagePowerOfTwo( ) ) {
      pMonitor->DisplayMessage( "Image Size must be a power of two." );
      goto PIX_EXIT;
   }

   // allocate and create grayscale image in float matrix

   if( !CreateGrayMatrix( &fGray ) ) {
      goto PIX_EXIT;
   }
   
   if( !fReal.Alloc( nHeight, nWidth ) ) {
      goto PIX_EXIT;
   }
   
   if( !fImag.Alloc( nHeight, nWidth ) ) {
      goto PIX_EXIT;
   }

   for( i = 0 ; i < nHeight ; i++ ) {
      for( j = 0 ; j < nWidth ; j++ ) {
         fVal = fGray.Get( i, j );
         fReal.Set( i, j, fVal );
         fImag.Set( i, j, 0.0f );
      }
   }

   // get fft matrix
   
   if( !Get2DImageFft_PI( &fReal, &fImag, false ) ) {
      goto PIX_EXIT;
   }

   // check filter shape
   // *** apply filter to the fft ***

   if( !ApplyFilterImage( &fReal, &fImag, &fFilter ) ) {
      goto PIX_EXIT;
   }

   // conver DFT result to amplitude

   if( !DftAmplitudeImage_PC( &fReal, &fImag, &fDft ) ) {
      goto PIX_EXIT;
   }

   // create gray scale image in second texture matrices

   if( !FloatMatrixGrayImage( &fDft ) ) {
      goto PIX_EXIT;
   }

   // --- Done ---
   bRetCode = true;
PIX_EXIT: 
   fGray.Free( );
   fReal.Free( );
   fImag.Free( );
   fDft.Free( );
   fFilter.Free( );
   return bRetCode;
}

//
// Image processing in matrix:
// visualize dct base
//
bool sampler_window::CreateSecondDftFiltered( void )
{
   bool bRetCode = false;
   pixq_ImageOpDft OpDft;
   pixq_Matrix<float> mfGray;
   int i, j, nWidth, nHeight, ii, jj, nBlockXSize, nBlockYSize;
   int iRow, jCol, iBlock, jBlock;
   float fGray, fB, fValue, fRange;
   
   int nBlockSize;
   pixq_Matrix<float> mfKitei;
   pixq_Matrix<float> mf2DDct; // 2-D DCT coeficient
   float *f1DDct = NULL; // 1-D DCT coeficient
   
   SamplerMonitor *pMonitor = (SamplerMonitor*)GetMonitorPane( );
   QString strMessage;

   nWidth = _ImageInMtx[0].GetWidth( );
   nHeight = _ImageInMtx[0].GetHeight( );

   // check image dimension must be power of two

   if( !CheckImagePowerOfTwo( ) ) {
      pMonitor->DisplayMessage( "Image Size must be a power of two." );
      goto PIX_EXIT;
   }

   // use standard block size of DCT, = 8

   nBlockSize = 8;
   nBlockXSize = (int)ceil( (double)nWidth / (double)nBlockSize );
   nBlockYSize = (int)ceil( (double)nHeight / (double)nBlockSize );

   f1DDct = new float[nBlockSize];
   if( !f1DDct ) {
      goto PIX_EXIT;
   }

   if( !mfKitei.Alloc( nBlockSize, nBlockSize ) ) {
      goto PIX_EXIT;
   }

   if( !mf2DDct.Alloc( nBlockSize, nBlockSize ) ) {
      goto PIX_EXIT;
   }

   if( !mfGray.Alloc( nHeight, nWidth ) ) {
      goto PIX_EXIT;
   }

   // calculate DCT Basic Function

   CreateDctBaseKitei( nBlockSize, &mfKitei );

   // allocate and create grayscale image in float matrix

   if( !CreateGrayMatrix( &mfGray ) ) {
      goto PIX_EXIT;
   }
   
   // Visualized DCT Image

   for( iBlock = 0 ; iBlock < nBlockYSize ; iBlock++ ) {
      iRow = iBlock * nBlockSize;
      for( jBlock = 0 ; jBlock < nBlockXSize ; jBlock++ ) {
         jCol =  jBlock * nBlockSize;

         // in-block horizontal DCT
         for( ii = 0 ; ii < nBlockSize ; ii++ ) {
            i = iRow + ii;
            for( jj = 0 ; jj < nBlockSize ; jj++ ) {
               j = jCol + jj;
               f1DDct[jj] = mfGray.Get( i, j );
            }
            // 1D DCT
            bApply1DDct( nBlockSize, f1DDct, false, &mfKitei );
            // store the 2D DCT coeficient
            for( jj = 0 ; jj < nBlockSize ; jj++ ) {
               mf2DDct.Set( ii, jj, f1DDct[jj] );
            }
         }

         // in-block verticL DCT
         for( jj = 0 ; jj < nBlockSize ; jj++ ) {
            for( ii = 0 ; ii < nBlockSize ; ii++ ) {            
               f1DDct[ii] = mf2DDct.Get( ii, jj );
            }
            // 1D DCT
            bApply1DDct( nBlockSize, f1DDct, false, &mfKitei );
            // store the 2D DCT coeficient
            for( ii = 0 ; ii < nBlockSize ; ii++ ) {  
               mf2DDct.Set( jj, ii, f1DDct[ii] );
            }
         }

         // visualize based on spectol
         fRange = 80.0f;
         for( ii = 0 ; ii < nBlockSize ; ii++ ) {
            i = iRow + ii;
            for( jj = 0 ; jj < nBlockSize ; jj++ ) {
               j = jCol + jj;
               //
               fValue = mf2DDct.Get( ii, jj );
               if( 0.0f == fValue ) {
                  fB = 0.0f;
               } else {
                  fB = 20.0f * (float)log10( fabs( fValue ) / ( 8.0f * 255.0f ) ) + fRange;
               }
               //
               if( 0.0f > fB ) {
                  fB = 0.0f;
               }
               // 
               fGray = 255.0f * fB / fRange;
               if( 255.0f < fGray ) fGray = 255.0f;
               if( 0.0f > fGray ) fGray = 0.0f;
               mfGray.Set( i, j, fGray );
            }
         }

      } // jBlock
   } // iBlock

   // create gray scale image in second texture matrices

   // set the gray matrix to output image matrices 
   
   if( !bSetAll( &mfGray ) ) {
      goto PIX_EXIT;
   }

   // --- Done ---
   bRetCode = true;
PIX_EXIT: 
   mfGray.Free( );
   mfKitei.Free( );
   mf2DDct.Free( );
   if( f1DDct ) {
      delete[] f1DDct;
      f1DDct = NULL;
   }
   return bRetCode;
}