/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.export.conf.monitor;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.StringListComparator;
import jp.co.intellilink.hinemos.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ConvertValueConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.snmp.bean.MonitorSnmpInfo;
import com.clustercontrol.snmp.ejb.session.MonitorSnmpController;

/**
 * SNMP監視情報を取得するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ExportMonitorSnmp extends ExportMonitorNumeric {

	// ログ出力
	private static Log log = LogFactory.getLog(ExportMonitorSnmp.class);

	public ExportMonitorSnmp() {
		super(ExportMonitorNumeric.INPUT_VERIFICATION_REAL_NUMBER);
	}

	/**
	 * SNMP監視情報をExportする。<br>
	 */
	public void exportSnmp() {
		log.info(Messages.getMsg("Snmp.Start", new String[]{Messages.getMsg("ExportTool.Export")}));

		//CSV出力配列を用意
		ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();

		//監視項目リスト取得
		ArrayList idList = getSnmpList();

		if(idList instanceof ArrayList){
			Iterator itr = idList.iterator();
			while(itr.hasNext()){
				ArrayList line = (ArrayList)itr.next();

				int monitorType = MonitorTypeConstant.stringToType((String)line.get(3));
				if(monitorType == MonitorTypeConstant.TYPE_NUMERIC){
					Object[] args = {(String)line.get(0)};
					log.info(Messages.getMsg("MonitorID", args));

					MonitorInfo info = getSnmp((String)line.get(0), monitorType);
					csvOutput.add(createCsvFormat(info));
				}
			}

			//監視項目IDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
		}

		String timestamp = "";
		if(Config.getConfig("file.timestamp").equalsIgnoreCase("on")){
			SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
			timestamp = "_" + format.format(new Date());
		}
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/SNMP" + timestamp + ".csv", 
				csvOutput);

		log.info(Messages.getMsg("Snmp.End", new String[]{Messages.getMsg("ExportTool.Export")}));
	}

	/**
	 * SNMP監視情報をExportする。<br>
	 * 
	 * @param monitorInfo SNMP監視情報
	 * @return CSVフォーマット文字列
	 */
	@Override
	public ArrayList<String> createMonitorInfo(MonitorInfo monitorInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		MonitorSnmpInfo snmpInfo = (MonitorSnmpInfo)monitorInfo.getCheckInfo();
		if(snmpInfo == null){
			snmpInfo = new MonitorSnmpInfo();
		}
		//コミュニティ名
		if (snmpInfo.getCommunityName() != null)
			csvFormat.add(snmpInfo.getCommunityName());
		else
			csvFormat.add("");
		//ポート番号
		if (snmpInfo.getSnmpPort() != null)
			csvFormat.add(snmpInfo.getSnmpPort().toString());
		else
			csvFormat.add("");
		//OID
		if (snmpInfo.getSnmpOid() != null)
			csvFormat.add(snmpInfo.getSnmpOid());
		else
			csvFormat.add("");
		//バージョン
		if (snmpInfo.getSnmpVersion() != null)
			csvFormat.add(snmpInfo.getSnmpVersion());
		else
			csvFormat.add("");
		//計算方法     
		if (snmpInfo.getConvertFlg() != null)
			csvFormat.add(ConvertValueConstant.typeToString(snmpInfo.getConvertFlg().intValue()));
		else
			csvFormat.add("");

		return csvFormat;
	}

	/**
	 * SNMP監視情報を取得する。<br>
	 * 
	 * @param monitorId 取得対象の監視項目ID
	 * @return SNMP監視情報
	 */
	protected MonitorInfo getSnmp(String monitorId, int monitorType) {

		MonitorSnmpController snmp = EjbConnectionManager.getConnectionManager().getMonitorSnmpController();

		MonitorInfo info = null;
		try {
			info = snmp.getSnmp(monitorId, monitorType);
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return info;
	}

	/**
	 * 監視項目リストを取得する。<br>
	 * 
	 * @return 監視項目一覧
	 */
	protected ArrayList getSnmpList() {

		MonitorSnmpController snmp = EjbConnectionManager.getConnectionManager().getMonitorSnmpController();

		ArrayList records = null;
		try {
			records = snmp.getSnmpList();
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}