/*
 
Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.export.conf.performance;

import java.util.HashMap;
import java.util.List;

import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.performance.bean.CollectorItemInfo;
import com.clustercontrol.performance.ejb.session.CollectorController;
import com.clustercontrol.performance.util.code.CollectorItemTreeItem;

/**
 * 収集項目コードの情報を生成するファクトリクラス
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class CollectorItemCodeFactory {
	private static HashMap<String, CollectorItemTreeItem> m_codeTable = null;

	// ログ出力
	private static Log log = LogFactory.getLog(CollectorItemCodeFactory.class);

	/**
	 * コンストラクター
	 */
	private CollectorItemCodeFactory() {
	}

	/**
	 * 初期化を行います。
	 */
	public static void init() {
		m_codeTable = getItemCodeTreeMap();
	}

	/**
	 * 指定収集項目コードの項目名を返します。
	 * 
	 * @param itemCode
	 * @return 指定収集項目コードの項目名
	 */
	public static String getItemName(String itemCode) {
		if(m_codeTable == null){
			init();
		}

		return m_codeTable.get(itemCode).getItemName();
	}

	/**
	 * 指定収集項目コードの項目名をデバイス名も含んだ形で返します。
	 * 
	 * @param itemCode 収集項目コード
	 * @param deviceName デバイス情報
	 * @return 指定収集項目コードの項目名
	 */
	public static String getFullItemName(String itemCode, String deviceName) {
		if(m_codeTable == null){
			init();
		}

		String name;
		if (deviceName == null || deviceName.equals("none") || deviceName.equals("")) {
			name = m_codeTable.get(itemCode).getItemName();
		} else {
			name = m_codeTable.get(itemCode).getItemName()
			+ "[" + deviceName + "]";
		}

		return name;
	}

	/**
	 * 指定収集項目コードの項目名をデバイス名も含んだ形で返します。
	 * 
	 * @param collectorItemInfo 収集項目情報
	 * @return 指定収集項目コードの項目名
	 */
	public static String getFullItemName(CollectorItemInfo collectorItemInfo) {
		if(m_codeTable == null){
			init();
		}

		String itemCode = collectorItemInfo.getItemCode();

		// デバイス別の収集を行う項目か否かで出力を変更
		if(CollectorItemCodeFactory.isDeviceSupport(itemCode)){
			// デバイス名も含めた項目名を返す
			return getFullItemName(
					collectorItemInfo.getItemCode(), 
					collectorItemInfo.getDeviceName()
			);
		} else {   	
			return getItemName(itemCode);
		}
	}

	/**
	 * デバイス別の性能値か否かを返します。
	 * 
	 * @return true デバイス別の性能値、false デバイス別の性能値ではない
	 */
	public static boolean isDeviceSupport(String itemCode) {
		if(m_codeTable == null){
			init();
		}

		return m_codeTable.get(itemCode).isDeviceFlag();
	}

	/**
	 * 指定のファシリティのデバイス情報を元に、収集項目ツリーを生成して返します。
	 * (収集IDにはnullが設定されます)
	 * 
	 * @param facilityId ファシリティID
	 * @return 指定のファシリティで設定可能な収集項目の一覧(収集IDはnull)
	 */
	public static List<CollectorItemInfo> getAvailableCollectorItemList(String facilityId){
		List<CollectorItemInfo> ret = null;

		CollectorController collector = EjbConnectionManager.getConnectionManager().getCollectorController();

		try {
			ret = collector.getAvailableCollectorItemList(facilityId);
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		} finally {
			// EJBオブジェクトを破棄する。
			if (collector != null) {
				try {
					collector.remove();
				} catch (Exception e) {
				}
			}
		}

		return ret;
	}

	/**
	 * 収集項目コード情報を取得します。
	 * 
	 * @return 収集項目コード情報
	 */
	public static HashMap<String, CollectorItemTreeItem> getItemCodeTreeMap() {
		HashMap<String, CollectorItemTreeItem> codeTable = null;

		CollectorController collector = EjbConnectionManager.getConnectionManager().getCollectorController();
		try {
			codeTable = collector.getItemCodeMap();
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		} finally {
			// EJBオブジェクトを破棄する。
			if (collector != null) {
				try {
					collector.remove();
				} catch (Exception e) {
				}
			}
		}

		return codeTable;
	}
	
    /**
     * 指定収集項目コードの性能値の単位を返します。
     * 
     * @return 性能値の単位
     */
    public static String getMeasure(String itemCode) {
    	if(m_codeTable == null){
    		init();
    	}
    	
        return m_codeTable.get(itemCode).getMeasure();
    }
}