/*
 
Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.export.conf.monitor;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.util.CheckString;
import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.StringListComparator;
import jp.co.intellilink.hinemos.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapOidInfo;
import com.clustercontrol.snmptrap.ejb.session.MonitorSnmpTrapController;

/**
 * SNMPTRAP監視情報を取得するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ExportMonitorSnmpTrap extends ExportMonitor {

	// ログ出力
	private static Log log = LogFactory.getLog(ExportMonitorSnmpTrap.class);
	
	/**
     * SNMPTRAP監視情報をExportする。<br>
     */
    public void exportSnmpTrap() {
    	log.info(Messages.getMsg("SnmpTrap.Start", new String[]{Messages.getMsg("ExportTool.Export")}));
    	
    	//CSV出力配列を用意
    	ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();
    	
    	//監視項目リスト取得
    	ArrayList idList = getSnmpTrapList();
    	
    	if(idList instanceof ArrayList){
			Iterator itr = idList.iterator();
			while(itr.hasNext()){
				ArrayList line = (ArrayList)itr.next();
				
				Object[] args = {(String)line.get(0)};
				log.info(Messages.getMsg("MonitorID", args));
				
				MonitorInfo info = getSnmpTrap((String)line.get(0));
				csvOutput.add(createCsvFormat(info));
			}
			
			//監視項目IDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
    	}
    	
    	String timestamp = "";
    	if(Config.getConfig("file.timestamp").equalsIgnoreCase("on")){
    		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
    		timestamp = "_" + format.format(new Date());
    	}
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/TRAP" + timestamp + ".csv", 
				csvOutput);
		
		log.info(Messages.getMsg("SnmpTrap.End", new String[]{Messages.getMsg("ExportTool.Export")}));
    }
    
	/**
	 * SNMPTRAP監視情報をExportする。<br>
	 * 
	 * @param monitorInfo HTTP監視情報
	 * @return CSVフォーマット文字列
	 * 
     * @see jp.co.intellilink.hinemos.export.conf.monitor.ExportMonitorNumeric#createCsvFormat(com.clustercontrol.monitor.run.bean.MonitorInfo)
     */
    @Override
	public ArrayList<String> createCsvFormat(MonitorInfo monitorInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		//監視基本情報
		csvFormat.addAll(createMonitorBase(monitorInfo));
		
		//カレンダID
		if (monitorInfo.getCalendarId() != null)
			csvFormat.add(monitorInfo.getCalendarId());
		else
			csvFormat.add("");
		
		//監視情報
		csvFormat.addAll(createMonitorInfo(monitorInfo));
		
		//監視通知情報
		csvFormat.addAll(createMonitorNotify(monitorInfo));

        //アプリケーション
        if (monitorInfo.getApplication() != null)
        	csvFormat.add(monitorInfo.getApplication());
        else
        	csvFormat.add("");
        
        //有効/無効
        csvFormat.add(ValidConstant.typeToString(monitorInfo.getValid()));
		
		return csvFormat;
	}
    
    /**
	 * SNMPTRAP監視情報をExportする。<br>
	 * 
	 * @param monitorInfo SNMPTRAP監視情報
	 * @return CSVフォーマット文字列
	 */
    @Override
	public ArrayList<String> createMonitorInfo(MonitorInfo monitorInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		MonitorSnmpTrapInfo trapInfo = (MonitorSnmpTrapInfo)monitorInfo;
        //コミュニティ名
        if (trapInfo.getCommunityName() != null)
        	csvFormat.add(trapInfo.getCommunityName());
        else
        	csvFormat.add("");
        //監視対象
		switch(trapInfo.getCheckMode()){
		case 1:
			csvFormat.add(com.clustercontrol.util.Messages.getString("OidListComposite.AllMaterOid"));
			break;
		case 2:
			csvFormat.add(com.clustercontrol.util.Messages.getString("OidListComposite.NoMasterOid"));
			break;
		default:
		case 0:
			csvFormat.add(com.clustercontrol.util.Messages.getString("OidListComposite.SpecificOid"));
			break;
		}
    	//OID
        if (trapInfo.getOidInfos() != null){
        	ArrayList<MonitorSnmpTrapOidInfo> oidInfos = trapInfo.getOidInfos();
        	for(MonitorSnmpTrapOidInfo oidInfo : oidInfos){
        		//トラップ名
                if (oidInfo.getTrapName() != null)
                	csvFormat.add(oidInfo.getTrapName());
                else
                	csvFormat.add("");
        		//OID
                if (oidInfo.getTrapOid() != null)
                	csvFormat.add(oidInfo.getTrapOid());
                else
                	csvFormat.add("");
        		//generic_id
                csvFormat.add(String.valueOf(oidInfo.getGenericId()));
        		//specific_id
                csvFormat.add(String.valueOf(oidInfo.getSpecificId()));
        		//有効・無効
                csvFormat.add(ValidConstant.typeToString(oidInfo.getValidFlg()));
        		//重要度
                csvFormat.add(PriorityConstant.typeToString(oidInfo.getPriority()));
        		//メッセージ
                if (oidInfo.getLogmsg() != null)
                	csvFormat.add(CheckString.checkReturn(oidInfo.getLogmsg()));
                else
                	csvFormat.add("");
        		//詳細メッセージ
                if (oidInfo.getDescr() != null)
                	csvFormat.add(CheckString.checkReturn(oidInfo.getDescr()));
                else
                	csvFormat.add("");
        		break;
        	}
        }
        else{
        	csvFormat.add("");
        	csvFormat.add("");
        	csvFormat.add("");
        	csvFormat.add("");
        	csvFormat.add("");
        	csvFormat.add("");
        	csvFormat.add("");
        	csvFormat.add("");
        }
        
		return csvFormat;
	}

    /**
     * SNMPTRAP監視情報を取得する。<br>
     * 
     * @param monitorId 取得対象の監視項目ID
     * @return SNMPTRAP監視情報
     */
    protected MonitorInfo getSnmpTrap(String monitorId) {

    	MonitorSnmpTrapController snmpTrap = EjbConnectionManager.getConnectionManager().getMonitorSnmpTrapController();

        MonitorInfo info = null;
        try {
        	info = snmpTrap.getMonitorInfo(monitorId);
        } catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
        }
        return info;
    }
    
    /**
     * 監視項目リストを取得する。<br>
     * 
     * @return 監視項目一覧
     */
    protected ArrayList getSnmpTrapList() {

    	MonitorSnmpTrapController snmpTrap = EjbConnectionManager.getConnectionManager().getMonitorSnmpTrapController();

    	ArrayList records = null;
    	try {
    		records = snmpTrap.getMonitorList();
        } catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
        }
        return records;
    }
}