/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.export.conf.monitor;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.StringListComparator;
import jp.co.intellilink.hinemos.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.http.bean.HttpCheckInfo;
import com.clustercontrol.http.ejb.session.MonitorHttpController;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;

/**
 * HTTP監視情報を取得するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ExportMonitorHttp extends ExportMonitorNumeric {

	// ログ出力
	private static Log log = LogFactory.getLog(ExportMonitorHttp.class);

	public ExportMonitorHttp() {
		super(ExportMonitorNumeric.INPUT_VERIFICATION_POSITIVE_INTEGER);
	}

	/**
	 * HTTP監視情報をExportする。<br>
	 */
	public void exportHttp() {
		log.info(Messages.getMsg("Http.Start", new String[]{Messages.getMsg("ExportTool.Export")}));

		//CSV出力配列を用意
		ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();

		//監視項目リスト取得
		ArrayList idList = getHttpList();

		if(idList instanceof ArrayList){
			Iterator itr = idList.iterator();
			while(itr.hasNext()){
				MonitorInfo info = (MonitorInfo)itr.next();
				if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
					Object[] args = {info.getMonitorId()};
					log.info(Messages.getMsg("MonitorID", args));
					csvOutput.add(createCsvFormat(info));
				}
			}

			//監視項目IDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
		}

		String timestamp = "";
		if(Config.getConfig("file.timestamp").equalsIgnoreCase("on")){
			SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
			timestamp = "_" + format.format(new Date());
		}
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/HTTP" + timestamp + ".csv", 
				csvOutput);

		log.info(Messages.getMsg("Http.End", new String[]{Messages.getMsg("ExportTool.Export")}));
	}

	/**
	 * HTTP監視情報をExportする。<br>
	 * 
	 * @param monitorInfo HTTP監視情報
	 * @return CSVフォーマット文字列
	 * 
	 * @see jp.co.intellilink.hinemos.export.conf.monitor.ExportMonitorNumeric#createCsvFormat(com.clustercontrol.monitor.run.bean.MonitorInfo)
	 */
	@Override
	public ArrayList<String> createCsvFormat(MonitorInfo monitorInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		//監視基本情報
		csvFormat.addAll(createMonitorBase(monitorInfo));

		//監視条件情報
		csvFormat.addAll(createMonitorRule(monitorInfo));

		//監視情報
		csvFormat.addAll(createMonitorInfo(monitorInfo));

		//監視判定情報
		csvFormat.addAll(createMonitorJudgement(monitorInfo));

		//監視通知情報
		csvFormat.addAll(createMonitorNotify(monitorInfo));

		//アプリケーション
		if (monitorInfo.getApplication() != null)
			csvFormat.add(monitorInfo.getApplication());
		else
			csvFormat.add("");

		//有効/無効
		csvFormat.add(ValidConstant.typeToString(monitorInfo.getValid()));

		return csvFormat;
	}

	/**
	 * HTTP監視情報をExportする。<br>
	 * 
	 * @param monitorInfo HTTP監視情報
	 * @return CSVフォーマット文字列
	 * 
	 * @see jp.co.intellilink.hinemos.export.conf.monitor.ExportMonitor#createMonitorInfo(com.clustercontrol.monitor.run.bean.MonitorInfo)
	 */
	@Override
	public ArrayList<String> createMonitorInfo(MonitorInfo monitorInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		HttpCheckInfo httpInfo = (HttpCheckInfo)monitorInfo.getHttpCheckInfo();
		if(httpInfo == null){
			httpInfo = new HttpCheckInfo();
		}
		//URL
		if (httpInfo.getRequestUrl() != null)
			csvFormat.add(httpInfo.getRequestUrl());
		else
			csvFormat.add("");
		//タイムアウト
		csvFormat.add(String.valueOf(httpInfo.getTimeout()));

		return csvFormat;
	}

	/**
	 * 監視項目リストを取得する。<br>
	 * 
	 * @return 監視項目一覧
	 */
	protected ArrayList getHttpList() {

		MonitorHttpController http = EjbConnectionManager.getConnectionManager().getMonitorHttpController();

		ArrayList records = null;
		try {
			records = http.getHttpList();
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}