/*

Copyright (C) 2008 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.test.monitor;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.test.util.Config;
import jp.co.intellilink.hinemos.test.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.test.util.Messages;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.MonitorBlockConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.RenotifyTypeConstant;
import com.clustercontrol.bean.StatusExpirationConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.monitor.bean.ReportEventInfo;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.bean.NotifyTypeConstant;
import com.clustercontrol.notify.ejb.entity.MonitorStatusData;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryData;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfoData;
import com.clustercontrol.repository.ejb.session.RepositoryController;

/**
 * 監視結果をチェックするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class CheckResultMonitor {
	private Date fromDate;
	private Date toDate;

	// ログ出力
	private static Log log = LogFactory.getLog(CheckResultMonitor.class);

	public CheckResultMonitor(Date fromDate, Date toDate){
		this.fromDate = fromDate;
		this.toDate = toDate;
	}

	/**
	 * 監視結果をチェックしCSVフォーマット文字列を作成する。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<ArrayList<String>> createCheckResult(MonitorInfo monitorInfo) {
		ArrayList<ArrayList<String>> csvFormat = new ArrayList<ArrayList<String>>();
		ArrayList<String> facilityIdList = new ArrayList<String>();

		//監視結果のファシリティIDリストを取得
		facilityIdList.addAll(getFacilityIdList(monitorInfo));

		for(String facilityId : facilityIdList){
			ArrayList<String> csvLine = new ArrayList<String>();
			//監視項目ID
			csvLine.add(monitorInfo.getMonitorId());
			//ファシリティID
			csvLine.add(facilityId);

			//ステータスチェック
			csvLine.addAll(checkStatus(monitorInfo, facilityId));

			//イベントチェック（重要度別）
			csvLine.addAll(checkEvent(monitorInfo, facilityId));

			//抑制チェック
			csvLine.addAll(checkNotify(monitorInfo, facilityId));

			csvFormat.add(csvLine);
		}

		return csvFormat;
	}

	/**
	 * 監視情報を基にファシリティIDリストを取得する。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @return ファシリティIDリスト
	 */
	protected ArrayList<String> getFacilityIdList(MonitorInfo monitorInfo) {
		ArrayList<String> facilityIdList = new ArrayList<String>();

		//単位から監視結果のファシリティIDリストを取得
		if(monitorInfo.getMonitorBlock() == MonitorBlockConstant.TYPE_NODE)
			facilityIdList.addAll(getNodeList(monitorInfo.getFacilityId()));
		else
			facilityIdList.add(monitorInfo.getFacilityId());

		return facilityIdList;
	}

	/**
	 * ステータス出力結果をチェックする。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkStatus(MonitorInfo monitorInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		String result = "-";

		//監視が無効の場合
		if(monitorInfo.getValid() == YesNoConstant.TYPE_NO){
			csvFormat.add(result);
			return csvFormat;
		}

		String notifyId = null;
		if(monitorInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = monitorInfo.getNotifyId();

			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS){
					notifyId = relationInfo.getNotifyId();
					break;
				}
			}
		}

		if(notifyId != null){
			result = "NG";

			Collection ct = 
				EjbConnectionManager.getConnectionManager().getStatus(facilityId);
			Iterator itr = ct.iterator();
			while (itr.hasNext()) {
				StatusInfoData status = (StatusInfoData) itr.next();
				if(status.getMonitorId().compareTo(monitorInfo.getMonitorId()) == 0){
					result = DateFormat.getDateTimeInstance().format(status.getGenerationDate());
					if(status.getExpirationFlg() != null && status.getExpirationFlg().intValue() != StatusExpirationConstant.TYPE_EXPIRATION)
						result = DateFormat.getDateTimeInstance().format(status.getGenerationDate());
					break;
				}
			}
		}
		else{
			result = "-";
		}

		csvFormat.add(result);

		return csvFormat;
	}

	/**
	 * イベント出力結果をチェックする。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkEvent(MonitorInfo monitorInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		String info = "-";
		String warning = "-";
		String critical = "-";
		String unknown = "-";

		//監視が無効の場合
		if(monitorInfo.getValid() == YesNoConstant.TYPE_NO){
			csvFormat.add(info);
			csvFormat.add(warning);
			csvFormat.add(critical);
			csvFormat.add(unknown);
			return csvFormat;
		}

		String notifyId = null;
		if(monitorInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = monitorInfo.getNotifyId();

			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT){
					notifyId = relationInfo.getNotifyId();
					break;
				}
			}
		}

		if(notifyId != null){
			//通知
			info = checkEventByPriority(monitorInfo, facilityId, PriorityConstant.TYPE_INFO, fromDate, toDate);
			//警告
			warning = checkEventByPriority(monitorInfo, facilityId, PriorityConstant.TYPE_WARNING, fromDate, toDate);
			//危険
			critical = checkEventByPriority(monitorInfo, facilityId, PriorityConstant.TYPE_CRITICAL, fromDate, toDate);
			//不明
			unknown = checkEventByPriority(monitorInfo, facilityId, PriorityConstant.TYPE_UNKNOWN, fromDate, toDate);
		}

		csvFormat.add(info);
		csvFormat.add(warning);
		csvFormat.add(critical);
		csvFormat.add(unknown);

		return csvFormat;
	}

	/**
	 * イベント出力結果をチェックする。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @param priority 重要度
	 * @param fromDate 出力日時（開始）
	 * @return CSVフォーマット文字列
	 */
	protected String checkEventByPriority(MonitorInfo monitorInfo, String facilityId, int priority, Date fromDate, Date toDate) {
		String result = "";

		ArrayList<ReportEventInfo> eventList = 
			EjbConnectionManager.getConnectionManager().getEvent(facilityId, fromDate, toDate);
		Date newDate = null;
		if (eventList == null)
			return result;
		for (ReportEventInfo event : eventList) {
			String monitorId = event.getMonitorId();
			int eventPriority = PriorityConstant.stringToType(event.getPriority());
			if(monitorId.compareTo(monitorInfo.getMonitorId()) == 0 && 
					eventPriority == priority){
				if(newDate instanceof Date){
					if(newDate.before(event.getGenerationDate()))
						newDate = event.getGenerationDate();
				}
				else{
					newDate = event.getGenerationDate();
				}
			}
		}
		if (newDate instanceof Date) {
			result = DateFormat.getDateTimeInstance().format(newDate);
		}

		return result;
	}

	/**
	 * 通知抑制の結果をチェックする。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkNotify(MonitorInfo monitorInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		String status = "-";
		String event = "-";
		String mail = "-";
		String job = "-";
		String syslog = "-";

		//監視が無効の場合
		if(monitorInfo.getValid() == YesNoConstant.TYPE_NO){
			csvFormat.add(status);
			csvFormat.add(event);
			csvFormat.add(mail);
			csvFormat.add(job);
			csvFormat.add(syslog);
			return csvFormat;
		}

		if(monitorInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = monitorInfo.getNotifyId();

			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS)
					status = checkNotifyInhibit(monitorInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT)
					event = checkNotifyInhibit(monitorInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_MAIL)
					mail = checkNotifyInhibit(monitorInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_JOB)
					job = checkNotifyInhibit(monitorInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_LOG_ESCALATE)
					syslog = checkNotifyInhibit(monitorInfo, relationInfo.getNotifyId(), facilityId);
			}
		}

		csvFormat.add(status);
		csvFormat.add(event);
		csvFormat.add(mail);
		csvFormat.add(job);
		csvFormat.add(syslog);

		return csvFormat;
	}

	/**
	 * 通知IDの抑制の結果をチェックする<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @param notifyId 通知ID
	 * @param facilityId ファシリティID
	 * @return 結果文字列
	 */
	protected String checkNotifyInhibit(MonitorInfo monitorInfo, String notifyId, String facilityId) {
		String result = "NG";

		NotifyInfo notify = getNotify(notifyId);
		if(notify.getValidFlg().intValue() == ValidConstant.TYPE_VALID){

			MonitorStatusData monitorStatus = getMonitorStatus(monitorInfo, facilityId);
			if(monitorStatus instanceof MonitorStatusData){
				// 監視状態が存在する
				
				if(monitorStatus.getCounter() >= notify.getInitialCount()){
					// カウンタが条件を満たしている
					
					NotifyHistoryData notifyHistory = getNotifyHistory(monitorInfo, notifyId, facilityId);
					if(notifyHistory instanceof NotifyHistoryData){
						// 通知履歴が存在する
						
						if(Config.getConfig("notify.control.mode") == "0"){							
							if(notify.getRenotifyType() == RenotifyTypeConstant.TYPE_NO_NOTIFY){
								// 通知しない
								if(monitorStatus.getLastUpdate().getTime() >= notifyHistory.getLastNotify().getTime()){
									// 通知履歴の最終通知時刻より、監視状態の最終更新時刻が新しい場合、OKとする。
									result = "OK";
								}
							} else if(notify.getRenotifyType() == RenotifyTypeConstant.TYPE_ALWAYS_NOTIFY){
								// 常に通知する
								if(monitorStatus.getLastUpdate().getTime() >= notifyHistory.getLastNotify().getTime()){
									// 通知履歴の最終通知時刻より、監視状態の最終更新時刻が新しい場合、OKとする。
									result = "OK";
								}
							} else {
								// 前回通知から一定時間通知しない
								Long diff = monitorStatus.getLastUpdate().getTime() - notifyHistory.getLastNotify().getTime();
								Long period = (notify.getRenotifyPeriod() * 60 * 1000l);
								if(diff <= period && diff >= -period){
									// 通知履歴の最終通知時刻と監視状態の最終更新時刻の差が通知抑制期間以内ならば、OKとする。
									result = "OK";
								}
							}
						}
						else
							result = "OK";
					}
				}
			}
			else
				result = "OK";
		}
		else
			result = "OK";

		return result;
	}
	
	/**
	 * 監視状態を取得する<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @return 監視状態
	 */
	protected MonitorStatusData getMonitorStatus(MonitorInfo monitorInfo, String facilityId) {
		
		try{
			Class.forName("org.postgresql.Driver");
		}catch(ClassNotFoundException e){
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		Connection connection = null;
		try{
			connection=DriverManager.getConnection(
					Config.getConfig("DB.URL"),
					Config.getConfig("DB.USER"),
					Config.getConfig("DB.PASSWORD"));
		}catch(SQLException e){
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
			
		try{
			//SQL文のセット
			PreparedStatement stmt = connection.prepareStatement("SELECT * FROM cc_monitor_status as a WHERE a.facility_id = ? and a.monitor_id = ?");
			stmt.setString(1, facilityId);
			stmt.setString(2, monitorInfo.getMonitorId());
			ResultSet result = stmt.executeQuery();
			while(result.next()){
				MonitorStatusData monitorStatus = new MonitorStatusData(
						result.getString("facility_id"),
						result.getString("plugin_id"),
						result.getString("monitor_id"),
						result.getInt("priority"),
						result.getTimestamp("last_update"),
						result.getLong("counter"));
				return monitorStatus;
			}
			return null;
		}catch(SQLException e){
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}finally{
			if (connection != null)
				try{
					connection.close();
				}catch(SQLException e){
				}
		}
		return null;
	}
	
	/**
	 * 通知履歴を取得する<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @param notifyId 通知ID
	 * @param facilityId ファシリティID
	 * @return 通知履歴
	 */
	protected NotifyHistoryData getNotifyHistory(MonitorInfo monitorInfo, String notifyId, String facilityId) {
		
		try{
			Class.forName("org.postgresql.Driver");
		}catch(ClassNotFoundException e){
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		Connection connection = null;
		try{
			connection=DriverManager.getConnection(
					Config.getConfig("DB.URL"),
					Config.getConfig("DB.USER"),
					Config.getConfig("DB.PASSWORD"));
		}catch(SQLException e){
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
			
		try{
			//SQL文のセット
			PreparedStatement stmt = connection.prepareStatement("SELECT * FROM cc_notify_history as a WHERE a.facility_id = ? and a.notify_id = ? and a.monitor_id = ?");
			stmt.setString(1, facilityId);
			stmt.setString(2, notifyId);
			stmt.setString(3, monitorInfo.getMonitorId());
			ResultSet result = stmt.executeQuery();
			while(result.next()){
				NotifyHistoryData notifyHistory = new NotifyHistoryData(
						result.getString("facility_id"),
						result.getString("plugin_id"),
						result.getString("monitor_id"),
						result.getString("notify_id"),
						result.getTimestamp("last_notify"));
				return notifyHistory;
			}
			return null;
		}catch(SQLException e){
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}finally{
			if (connection != null)
				try{
					connection.close();
				}catch(SQLException e){
				}
		}
		return null;
	}

	/**
	 * 通知情報を取得する。<br>
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return 通知情報
	 */
	protected NotifyInfo getNotify(String notifyId) {
		return EjbConnectionManager.getConnectionManager().getNotifyId(notifyId);
	}

	/**
	 * ノードのファシリティIDを取得する。<br>
	 */
	protected ArrayList<String> getNodeList(String facilityId) {

		RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

		ArrayList<String> records = new ArrayList<String>();
		try {
			if(repository.isNode(facilityId)){
				records.add(facilityId);
			}
			else{
				records.addAll(repository.getExecTargetFacilityIdList(facilityId));
			}
		} catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}