(ns test.math
  (:use clojure.contrib.test-is))

(require 'src.math)
(alias 'mt 'src.math)

(deftest test-swap
  (is (= '(d b c a)
         (mt/swap 0 3 '(a b c d))))
  (is (= '(b a c d)
         (mt/swap -3 0 '(a b c d))))
  (is (= '(a b c d)
         (mt/swap 1 1 '(a b c d)))))

(deftest test-search-non-zero
  (let [m [[0 0 1 2]
           [0 1 0 1]
           [0 0 0 3]]]
    (is (= (mt/search-non-zero m 0)
           [1 1]))
    (is (= (mt/search-non-zero m 2)
           nil))))

(deftest test-sweep1
  (let [m [[2 1 1 0]
           [1 2 4 6]
           [4 0 3 2]]]
    (is (= (mt/sweep1 m 0)
           [[1 1/2 1/2 0]
            [0 3/2 7/2 6]
	    [0  -2   1 2]]))
    (is (= (mt/sweep1 m 1)
           [[3/2 0 -1 -3]
            [1/2 1  2  3]
            [  4 0  3  2]]))))

(deftest test-lin-solve
  (let [m [[2 1 1 5]
           [1 2 1 4]
           [1 1 2 3]]]
    (is (= (mt/lin-solve m)
           [[1 0 0 2]
            [0 1 0 1]
            [0 0 1 0]
            ]))))

(deftest test-m*v
  (is (= (mt/m*v [[1 2] [3 4]] [5 6])
         [17 39]))
  (is (= (mt/m*v [[1 0] [0 1]] [3 4])
         [3 4]))
  (is (= (mt/m*v [[1 0 2] [2 1 3]] [1 1 2])
         [5 9])))

(deftest test-s*m
  (is (= (mt/s*m 2 [[1 2] [3 4]])
         [[2 4] [6 8]])))

(deftest test-m-m
  (is (= (mt/m-m [[1 2] [3 4]] [[5 6] [7 8]])
         [[-4 -4] [-4 -4]]))
  (is (= (mt/m-m [[1 0] [0 1]] [[1 2] [3 4]])
         [[0 -2] [-3 -3]])))

(deftest test-i-mat
  (is (= (mt/i-mat 0) []))
  (is (= (mt/i-mat 1) [[1]]))
  (is (= (mt/i-mat 2) [[1 0] [0 1]]))
  (is (= (mt/i-mat 3)
         [[1 0 0]
          [0 1 0]
          [0 0 1]])))

(deftest test-inv-mat
  (is (= (mt/inv-mat [[1 0 0]
                      [0 1 0]
                      [0 0 1]])
         [[1 0 0]
          [0 1 0]
          [0 0 1]
          ]))
  (is (= (mt/inv-mat [[1 2]
                      [3 4]])
         [[ -2    1]
	  [3/2 -1/2]
          ])))

(deftest test-tref
  (is (= (mt/tref '[up 0 1 2])
         '[up 0 1 2]
         ))
  (is (= (mt/tref '[up 0 [up 1 2] [up 3 4]]
                  2)
         '[up 3 4]))
  (is (= (mt/tref '[up 0 [up 1 2] [up 3 4]]
                  2 1)
         4)))

(deftest test-tfassoc
  (is (= (mt/tfassoc 2 '[] #(* % 3))
         6))
  (is (= (mt/tfassoc '[up 0 [up 1 2] [down 3 4]]
                     '[2 1]
                     #(+ % 4))
         '[up 0 [up 1 2] [down 3 8]]
         )))

(deftest test-t+2
  (is (= (mt/t+2 1 2) 3))
  (is (= (mt/t+2 '[up 1 2 3]
                 '[up 4 5 6])
         '[up 5 7 9]
         ))
  (is (= (mt/t+2 '[down 0 [up 1 2] [down 3 4]]
                 '[down 5 [up 6 7] [down 8 9]])
         '[down 5 [up 7 9] [down 11 13]]
         ))
  (is (= (mt/t+2 '[up 1 2] '[down 3 4])
         nil
         )))

(deftest test-t+
  (is (= (mt/t+ '[down 0 1 2])
         '[down 0 1 2]
         ))
  (is (= (mt/t+ 1 2) 3))
  (is (= (mt/t+ '[up 1 2 3]
                '[up 4 5 6])
         '[up 5 7 9]
         ))
  (is (= (mt/t+ '[up 1 2 3]
                '[up 4 5 6]
                '[up 7 8 9])
         '[up 12 15 18]
         )))

(deftest test-t-2
  (is (= (mt/t-2 1 3) -2))
  (is (= (mt/t-2 '[up 1  2 3]
                 '[up 4 -4 5])
         '[up -3 6 -2]
         ))
  (is (= (mt/t-2 '[down  0 [up  1  2] [down  3  4]]
                 '[down -1 [up -3 -5] [down -7 -9]])
         '[down 1 [up 4 7] [down 10 13]]
         ))
  (is (= (mt/t-2 '[up 1 2] '[down 3 4])
         nil
         )))

(deftest test-t-
  (is (= (mt/t- '[down 3 [up 1 2]])
         '[down -3 [up -1 -2]]
         ))
  (is (= (mt/t- 1 -3) 4))
  (is (= (mt/t- '[up 1 2 3]
                '[up 3 5 7])
         '[up -2 -3 -4]
         ))
  (is (= (mt/t- '[up 1 2 3]
                '[up 3 5 7]
                '[up 2 5 8])
         '[up -4 -8 -12]
         )))

(deftest test-t*2
  (is (= (mt/t*2 '[up 1 2] '[down 3 4])
         11))
  (is (= (mt/t*2 2 '[up 2 3])
         '[up 4 6]
         ))
  (is (= (mt/t*2 '[down 3 4] 5)
         '[down 15 20]
         ))
  (is (= (mt/t*2 '[up 1 2]
                 '[up [down 2 3] [down 3 4]])
         '[up 8 11]
         )))

(deftest test-t*
  (is (= (mt/t* '[up 2 3])
         '[up 2 3]
         ))
  (is (= (mt/t* 2 '[up 2 3])
         '[up 4 6]
         ))
  (is (= (mt/t* 2
                '[up 1 2]
                '[up [down 2 3] [down 3 4]])
         '[up 16 22]
         )))

(deftest test-tinv
  (is (= (mt/tinv '[up [up 1 2] [up 3 4]])
         '[down [down -2 1] [down 3/2 -1/2]]))
  (is (= (mt/tinv '[up [down 1 2] [down 3 4]])
         '[up [down -2 1] [down 3/2 -1/2]]))
  (is (= (mt/tinv '[down [up 1 2] [up 3 4]])
         '[down [up -2 1] [up 3/2 -1/2]]))
  (is (= (mt/tinv '[down [down 1 2] [down 3 4]])
         '[up [up -2 1] [up 3/2 -1/2]])))

(deftest test-partial-val
  (is (= (binding [mt/*dx* 0.5]
           (mt/partial-val (fn [x] (* 4 x))
                           []
                           3
                           ))
         4
         ))
  (is (= (binding [mt/*dx* 0.5]
           (mt/partial-val (fn [[_ x y]]
                             ['up
                              (* 2 x y)
                              (+ (* 5 x) (* 3 y))
                              ])
                           [0]
                           ['up 1 2]
                           ))
         ['up 4 5]
         ))
  (is (= (binding [mt/*dx* 0.5]
           (mt/partial-val (fn [[_ x y]]
                             ['up
                              (* 2 x y)
                              (+ (* 5 x) (* 3 y))
                              ])
                           []
                           ['up 1 2]
                           ))
         ['down ['up 4 5]
                ['up 2 3]
                ]))
  (is (= (binding [mt/*dx* 0.5]
           (mt/partial-val (fn [[_ x [_ y]]]
                             ['up (* 2 x y)
                                  ['up (+ (* 5 x) (* 3 y))]
                                  ])
                           []
                           ['down 1 ['down 2]]
                           ))
         ['up ['up 4 ['up 5]]
              ['up ['up 2 ['up 3]]]
              ]))
  )

(run-tests)

