package com.namaspider.iota.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * 文字列操作ユーティリティクラス。
 * @author Akado Takatoshi
 *
 */
public class StringUtil {

	/**
	 * str が null または長さ 0 の文字列である場合 true を返す。
	 * @param str String
	 * @return str が null または長さ 0 の文字列である場合 true、それ以外の場合 false
	 */
	public static boolean isEmpty(String str) {
		return str == null || str.equals("");
	}

	/**
	 * str が null でなく、かつ長さ 0 の文字列でない場合 true を返す。
	 * @param str String
	 * @return str が null でなく、かつ長さ 0 の文字列でない場合 true、それ以外の場合 false
	 */
	public static boolean isNotEmpty(String str) {
		return str != null && !str.equals("");
	}

	/**
	 * str を改行（\r または \n）で分割する。
	 *
	 * @param str String
	 * @return 改行毎に分割した文字列リスト
	 */
	public static List<String> getStringLineList(String str) {
	    if (str == null) {
	        return null;
	    } else {
	        String[] lines = str.split("\\n|\\r");
	        List<String> list = new ArrayList<String>();
	        for (String line : lines) {
	            list.add(line);
	        }
	        return list;
	    }
	}

	/**
	 * 1文字目を大文字に変換する。
	 * @param str String
	 * @return 変換した文字列
	 */
	public static String toUpperCaseInitial(String str) {
		if (isEmpty(str)) {
			return str;
		} else {
			return str.substring(0, 1).toUpperCase().concat(str.substring(1));
		}
	}

	/**
	 * 1文字目を小文字に変換する。
	 * @param str String
	 * @return 変換した文字列
	 */
	public static String toLowerCaseInitial(String str) {
		if (isEmpty(str)) {
			return str;
		} else {
			return str.substring(0, 1).toLowerCase().concat(str.substring(1));
		}
	}

	/**
	 * 文字列を先頭が小文字のキャメルケースに変換する。
	 *
	 * LowerCamelCase => lowerCamelCase
	 * lower_camel_case => lowerCamelCase
	 *
	 * @param str String
	 * @return 変換した文字列
	 */
	public static String toCamelCase(String str) {
		StringBuilder sb = new StringBuilder();
		String[] terms = toLowerCaseInitial(str).split("_");
		sb.append(terms[0]);
		for (int i = 1, len = terms.length; i < len; i++) {
			sb.append(toUpperCaseInitial(terms[i]));
		}
		return sb.toString();
	}

	/**
	 * 文字列を先頭が大文字のキャメルケースに変換する。
	 *
	 * upperCamelCase => UpperCamelCase
	 * upper_camel_case => UpperCamelCase
	 *
	 * @param str String
	 * @return 変換した文字列
	 */
	public static String toPascalCase(String str) {
		StringBuilder sb = new StringBuilder();
		String[] terms = toUpperCaseInitial(str).split("_");
		sb.append(terms[0]);
		for (int i = 1, len = terms.length; i < len; i++) {
			sb.append(toUpperCaseInitial(terms[i]));
		}
		return sb.toString();
	}

	/**
	 * 文字列をスネークケースに変換する。
	 *
	 * SnakeCase => snake_case
	 * snakeCase => snake_case
	 *
	 * @param str String
	 * @return 変換した文字列
	 */
	public static String toSnakeCase(String str) {
		StringBuilder sb = new StringBuilder();
		char[] buf = toLowerCaseInitial(str).toCharArray();
		sb.append(buf[0]);
		for (int i = 1; i < buf.length; i++) {
			if ('A' <= buf[i] && buf[i] <= 'Z') {
				sb.append('_');
				sb.append((char)(buf[i] + 0x20));
			} else {
				sb.append(buf[i]);
			}
		}
		return sb.toString();
	}

	/**
	 * map に含まれるすべてのエントリのリスト文字列を生成する。
	 * 生成文字列は toString() に従う。順序は iterator() の返す順である。
	 *
	 * @param map Map
	 * @param delim リスト文字列の区切り文字
	 * @param entryDelim キーと要素の区切り文字
	 * @return リスト文字列
	 */
	public static String getListString(Map<?, ?> map, String delim, String entryDelim) {
		StringBuilder sb = new StringBuilder();
		Iterator<?> i = map.keySet().iterator();
		Iterator<?> j = map.values().iterator();
		sb.append(getListString(entryDelim, i.next(), j.next()));
		while (i.hasNext()) {
			sb.append(delim);
			sb.append(getListString(entryDelim, i.next(), j.next()));
		}
		return sb.toString();
	}

	/**
	 * iterable に含まれるすべての要素のリスト文字列を生成する。
	 * 生成文字列は toString() に従う。順序は iterator() の返す順である。
	 * このメソッドは
	 * <pre>
	 *     getListString(iterable.iterator(), delim)
	 * </pre>
	 * と同一の動作である。
	 *
	 * @param iterable Iterable
	 * @param delim リスト文字列の区切り文字
	 * @return リスト文字列
	 */
	public static String getListString(Iterable<?> iterable, String delim) {
		return getListString(iterable.iterator(), delim);
	}

	/**
	 * i に含まれるすべてのエントリのリスト文字列を生成する。
	 * 生成文字列は toString() に従う。順序は iterator() の返す順である。
	 *
	 * @param i Iterator
	 * @param delim リスト文字列の区切り文字
	 * @return リスト文字列
	 */
	public static String getListString(Iterator<?> i, String delim) {
		StringBuilder sb = new StringBuilder();
		sb.append(i.next());
		while (i.hasNext()) {
			sb.append(delim);
			sb.append(i.next());
		}
		return sb.toString();
	}

	/**
	 * obj の n 回繰り返しリスト文字列を生成する。
	 * 生成文字列は toString() に従う。
	 *
	 * @param obj Object
	 * @param delim リスト文字列の区切り文字
	 * @param n 繰り返し数
	 * @return リスト文字列
	 */
	public static String getListString(Object obj, String delim, int n) {
		if (n <= 0) {
			return "";
		} else {
			StringBuilder sb = new StringBuilder();
			sb.append(obj);
			for (int i = 1; i < n; i++) {
				sb.append(delim);
				sb.append(obj);
			}
			return sb.toString();
		}
	}

	/**
	 * 配列 obj のリスト文字列を生成する。
	 * 生成文字列は toString() に従う。
	 *
	 * @param delim リスト文字列の区切り文字
	 * @param obj Object[]
	 * @return リスト文字列
	 */
	private static String getListString(String delim, Object... obj) {
		StringBuilder sb = new StringBuilder();
		sb.append(obj[0]);
		for (int i = 1, len = obj.length; i < len; i++) {
			sb.append(delim);
			sb.append(obj[i]);
		}
		return sb.toString();
	}

}
