#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sched.h>
#include <arpa/inet.h>

#include "crc32.h"
#include "ringbuffer.h"
#include "status.h"

inline int get_pid(unsigned char *this) {
    return ((this[1] & 0x1f) << 8) + this[2];
}

unsigned char *find_packet(buffer_t *this, int find_pid) {
    int i, cur_pid;
    for(i = 0; i <= this->bottom; i++) {
        cur_pid = get_pid(this->data[i].ts);
        if(cur_pid == find_pid)
            return this->data[i].ts;
    }
    return NULL;
}

unsigned char *find_pmt(buffer_t *this, int find_pid) {
    int i, cur_pid;
    unsigned char *pmt;
    
    for(i = 0; i <= this->bottom; i++) {
        cur_pid = get_pid(this->data[i].ts);
        if(cur_pid == find_pid) {
            pmt = this->data[i].ts;
            if(pmt[1] & 0x40) {
                return pmt;
            }
        }
    }
    return NULL;
}

void dump_packet(unsigned char *ts) {
    int i;
    for(i = 0; i < 188; i++) {
        printf("%02x ", ts[i]);
        if(i % 16 == 0)
            printf("\n");
    }
    printf("\n");
}

void edit_pmt(buffer_t *this, int top, int n) {
    int i, adof = 0;
    unsigned int pid, crc32;
    unsigned char *ts;
    unsigned char pmt_buffer[188*2];
    int multiPMT = 0;
    for(i = 0; i < n; i++) {
        ts = this->data[(top+i) % BUFFER_SIZE].ts;
        pid = get_pid(ts);
        if(pid == this->pmt) {
            if(ts[1] & 0x40) {
                //PMTの先頭packet
                if(ts[5] == 0x02) {
                    if(ts[3] & 0x20)
                        adof = ts[4]+1;
                    if(ts[17+adof] == 0x09 && ts[16+adof] != 0) {
                        ts[17+adof] = 0xE1;
                        int size = ts[7+adof];
                        if(size < (188-4-2)) {
                            crc32 = htonl(gf_m2ts_crc32((char *)ts+adof+5,ts[7+adof]-1));
                            memcpy(ts+4+adof+ts[7+adof],&crc32,4);
                        } else {
                            multiPMT = 1;
                        }
                    }
                }
                memcpy(pmt_buffer, ts, 188);
            } else {
                //PMTの非先頭packet
                if(multiPMT) {
                    memcpy(pmt_buffer + 188, ts + 4, 184);
                    crc32 = htonl(gf_m2ts_crc32((char *)pmt_buffer+adof+5, pmt_buffer[7+adof]-1));
                    memcpy(ts+4+4+adof+pmt_buffer[7+adof]-188, &crc32, 4);
                }
                multiPMT = 1;
            }
        } else if(pid == 0x14) {
            //何のための処理か忘れた
            ts[1] = 0x00;
            ts[2] = 0x14;
        }
    }
}


int initBuffer(buffer_t *this, FILE *target) {
    pthread_rwlock_init(&this->rwlock, NULL);
    this->data = (cell_t *)malloc(sizeof(cell_t) * BUFFER_SIZE);
    this->top = 0;
    this->bottom = fread(this->data, 188, BUFFER_SIZE, target) - 1;
    this->end = -1;
    this->fp = target;
    unsigned char *pat, *pmt;
    int pmt_pid = -1, id, i, offset = 0, adof = 0;
    pat = find_packet(this, 0); // find PAT
    if(pat != NULL) {
        if((pat[3] & 0x20)) { //exist adaptation field
            offset = pat[4] + 1;
            adof=offset - 2+2;
            printf("adap hakken offset = %d\n", offset);
        }
        pmt_pid = -1;
        i = 0;
        do {
            id = (pat[offset+13+4*i] << 8)+ pat[offset+14+4*i];
            if(id != 0) {
                pmt_pid = ((pat[offset+15+4*i] & 0x1f) << 8)+ pat[offset+16+4*i];
                printf("pmt_pid = 0x%02x\n", pmt_pid);
            }
            i++;
        } while(pmt_pid < 0 && (offset+16+4*i) < 188);
        if(pmt_pid > 0) {
            pmt = find_pmt(this, pmt_pid);
            offset = 4;
            if(pmt != NULL){
                this->pmt = pmt_pid;
                if((pmt[3] & 0x20)) { //exist adaptation field
                    offset += pmt[4] + 1;
                    printf("adap hakken offset = %d\n", offset);
                }
                i =  get_pid(pmt+offset+8);
                this->pcr = i;
            }
        }
    }
    edit_pmt(this, 0, this->bottom + 1);
    printf("bottom = %d\n", this->bottom);
    return this->pcr;
}

void *fillBuffer(void *buf) {
    buffer_t *this = (buffer_t *)buf;
    int l_top, l_bottom, i, aim;
    int counter = 5;

    while(1) {
        sleep(1);

        l_bottom = this->bottom;
        pthread_rwlock_rdlock(&(this->rwlock));
        l_top = this->top;
        pthread_rwlock_unlock(&(this->rwlock));
        //printf("top = %d, bottom = %d\n",l_top,l_bottom);

        if(l_top <= l_bottom) {
            i = fread(this->data+l_bottom+1, 188, aim = (BUFFER_SIZE-l_bottom-1),this->fp);
            i += fread(this->data, 188, l_top,this->fp); aim += l_top;
        } else {
            i = fread(this->data+l_bottom+1, 188,aim=(l_top - l_bottom - 1),this->fp);
        }
        
        edit_pmt(this, l_bottom + 1, i);

        if((i == 0) && (aim != 0))
            counter--;

        l_bottom = (l_bottom + i) % BUFFER_SIZE;

        pthread_rwlock_wrlock(&(this->rwlock));
        this->bottom = l_bottom;
        pthread_rwlock_unlock(&(this->rwlock));

        if(!counter)
            break;
        if(!isPlaying())
            break;
    }
    printf("finish fill buffer\n");
    
    return NULL;
}

int readBuffer(unsigned char *data, int n_packets,
                unsigned int dropped, void *callback_data) {
    int l_end, l_top;
    buffer_t *this = callback_data;

    int forward;
    if((forward = getForward())) {
        l_top = (this->top + 53000 * forward) % BUFFER_SIZE;
        unsetForward();
        printf("forward = %d\n\n\n", forward);
    } else {
        l_top = this->top;
    }

    memcpy(data, this->data[l_top].ts, 188);
    pthread_rwlock_wrlock(&this->rwlock);
    this->top = (l_top + 1) % BUFFER_SIZE;
    l_end = this->bottom;
    pthread_rwlock_unlock(&this->rwlock);

    if(isPaused()) {
        pthread_mutex_t *pause_control;
        pause_control = getPauseControl();

        pthread_mutex_lock(pause_control);
        pthread_mutex_unlock(pause_control);
        if(!isPlaying()){
            return -1;
        }
    }

    if(this->top == l_end) {
        unsetPlaying();
        return -1;
    }

    return 0;
}

void freeBuffer(buffer_t *this) {
    free(this->data);
    pthread_rwlock_destroy(&this->rwlock);
    fclose(this->fp);
}

/*static u64
ts_get_pcr( unsigned char *ts )
{
    u64 pcr;
    
    pcr = ts->pcr[ 0 ] << 25;
    pcr += ts->pcr[ 1 ] << 17;
    pcr += ts->pcr[ 2 ] << 9;
    pcr += ts->pcr[ 3 ] << 1;
    pcr += ( ts->pcr[ 4 ] & 0x80 ) >> 7;
    pcr *= 300;
    pcr += ( ts->pcr[ 4 ] & 0x1 ) << 8;
    pcr += ts->pcr[ 5 ];
    
    return pcr;
}
*/
