/* テスト */
/*
 * File: cmd3.c
 * Purpose: Miscellaneous queries
 *
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke
 *
 * This work is free software; you can redistribute it and/or modify it
 * under the terms of either:
 *
 * a) the GNU General Public License as published by the Free Software
 *    Foundation, version 2, or
 *
 * b) the "Angband licence":
 *    This software may be copied and distributed for educational, research,
 *    and not for profit purposes provided that this copyright and statement
 *    are included in all such copies.  Other copyrights may also apply.
 */
#include "angband.h"
#include "object/tvalsval.h"

#include "cmds.h"

/*
 * Display inventory
 */
void do_cmd_inven(void)
{
	/* Hack -- Start in "inventory" mode */
	p_ptr->command_wrk = (USE_INVEN);

	/* Save screen */
	screen_save();

	/* Hack -- show empty slots */
	item_tester_full = TRUE;

	/* Display the inventory */
	show_inven();

	/* Hack -- hide empty slots */
	item_tester_full = FALSE;

	/* Prompt for a command */
	prt(format(L"(Inventory) Burden %d.%dlb (%d%% capacity). Command: ",
	    p_ptr->total_weight / 10, p_ptr->total_weight % 10,
	    (10 * p_ptr->total_weight) / (6 * adj_str_wgt[p_ptr->state.stat_ind[A_STR]])), 0, 0);

	/* Hack -- Get a new command */
	p_ptr->command_new = inkey();

	/* Load screen */
	screen_load();


	/* Hack -- Process "Escape" */
	if (p_ptr->command_new == ESCAPE)
	{
		/* Reset stuff */
		p_ptr->command_new = 0;
	}
}

/*
 * Display equipment
 */
void do_cmd_equip(void)
{
	/* Hack -- Start in "equipment" mode */
	p_ptr->command_wrk = (USE_EQUIP);

	/* Save screen */
	screen_save();

	/* Hack -- show empty slots */
	item_tester_full = TRUE;

	/* Display the equipment */
	show_equip();

	/* Hack -- undo the hack above */
	item_tester_full = FALSE;

	/* Prompt for a command */
	prt(L"(Equipment) Command: ", 0, 0);

	/* Hack -- Get a new command */
	p_ptr->command_new = inkey();

	/* Load screen */
	screen_load();

	/* Hack -- Process "Escape" */
	if (p_ptr->command_new == ESCAPE)
	{
		/* Reset stuff */
		p_ptr->command_new = 0;
	}
}

/*
 * Wield or wear a single item from the pack or floor
 */
void wield_item(object_type *o_ptr, int item)
{
	object_type object_type_body;
	object_type *i_ptr = &object_type_body;

	const _TCHAR *act;
	_TCHAR o_name[80];

	int slot = wield_slot(o_ptr);

	/* Take a turn */
	p_ptr->energy_use = 100;

	/* Obtain local object */
	object_copy(i_ptr, o_ptr);

	/* Modify quantity */
	i_ptr->number = 1;

	/* Decrease the item (from the pack) */
	if (item >= 0)
	{
		inven_item_increase(item, -1);
		inven_item_optimize(item);
	}
	else /* Decrease the item (from the floor) */
	{
		floor_item_increase(0 - item, -1);
		floor_item_optimize(0 - item);
	}
	/* Get the wield slot */
	o_ptr = &inventory[slot];

	/* Take off existing item */
	if (o_ptr->k_idx)
	{
		/* Take off existing item */
		(void)inven_takeoff(slot, 255);
	}
	/* Wear the new stuff */
	object_copy(o_ptr, i_ptr);

	/* Increase the weight */
	p_ptr->total_weight += i_ptr->weight;

	/* Increment the equip counter by hand */
	p_ptr->equip_cnt++;

	/* Do any ID-on-wield */
	object_tried(o_ptr);
	object_notice_on_wield(o_ptr);

	/* Where is the item now */
	if (slot == INVEN_WIELD)
		act = L"You are wielding";
	else if (slot == INVEN_BOW)
		act = L"You are shooting with";
	else if (slot == INVEN_LITE)
		act = L"Your light source is";
	else
		act = L"You are wearing";

	/* Describe the result */
	object_desc(o_name, _countof(o_name), o_ptr, TRUE, ODESC_FULL);

	/* Message */
	sound(MSG_WIELD);
	msg_format(L"%s %s (%c).", act, o_name, index_to_label(slot));

	/* Cursed! */
	if (cursed_p(o_ptr))
	{
		/* Warn the player */
		sound(MSG_CURSED);
		msg_print(L"Oops! It feels deathly cold!");

		/* Sense the object */
		object_notice_curses(o_ptr);
	}

	/* Recalculate bonuses, torch, mana */
	p_ptr->update |= (PU_BONUS | PU_TORCH | PU_MANA);
	p_ptr->redraw |= (PR_INVEN | PR_EQUIP);
}

void do_cmd_destroy(cmd_code code, cmd_arg args[]) 
{ 
	int item, amt; 

	object_type *o_ptr; 

	object_type destroyed_obj;

	_TCHAR o_name[120]; 

	item = args[0].item; 
	amt = args[1].number; 

	/* Destroying squelched items is easy. */ 
	if (item == ALL_SQUELCHED) 
	{ 
		squelch_items(); 
		return; 
	} 
	if (!item_is_available(item, NULL, USE_INVEN | USE_EQUIP | USE_FLOOR)) 
	{ 
		msg_print(__T("You do not have that item to destroy it.")); 
		return; 
	}
	o_ptr = object_from_item_idx(item);

	/* Can't destroy cursed items we're wielding. */
	if ((item >= INVEN_WIELD) && cursed_p(o_ptr))
	{
		msg_print(__T("You cannot destroy the cursed item."));
		return;
	}
	/* Describe the destroyed object by taking a copy with the right "amt" */ 
	object_copy_amt(&destroyed_obj, o_ptr, amt); 
	object_desc(o_name, _countof(o_name), &destroyed_obj, TRUE, ODESC_FULL);

	/* Artifacts cannot be destroyed */
	if (artifact_p(o_ptr))
	{
		/* Message */
		msg_format(L"You cannot destroy %s.", o_name);
		o_ptr->ident |= IDENT_INDESTRUCT;

		/* Combine the pack */
		p_ptr->notice |= (PN_COMBINE);

		/* Redraw stuff */
		p_ptr->redraw |= (PR_INVEN | PR_EQUIP);

		/* Done */
		return;
	}
	/* Message */
	message_format(MSG_DESTROY, 0, L"You destroy %s.", o_name);

	/* Reduce the charges of rods/wands/staves */
	reduce_charges(o_ptr, amt);

	/* Eliminate the item (from the pack) */
	if (item >= 0)
	{
		inven_item_increase(item, -amt);
		inven_item_describe(item);
		inven_item_optimize(item);
	}
	else /* Eliminate the item (from the floor) */
	{
		floor_item_increase(0 - item, -amt);
		floor_item_describe(0 - item);
		floor_item_optimize(0 - item);
	}
} 

/*
 * Destroy an item
 */
void textui_cmd_destroy(void)
{
	int item, amt;
	object_type *o_ptr;
	object_type obj_to_destroy;

	_TCHAR o_name[120];
	_TCHAR out_val[160];

	const _TCHAR *q; 
	const _TCHAR *s;

	/* Get an item */
	q = L"Destroy which item? ";
	s = L"You have nothing to destroy.";
	if (!get_item(&item, q, s, (USE_INVEN | USE_EQUIP | USE_FLOOR | CAN_SQUELCH))) return;

	/* Deal with squelched items */
	if (item == ALL_SQUELCHED)
	{
		cmd_insert(CMD_DESTROY, item, 0);
		return;
	}
	o_ptr = object_from_item_idx(item);

	/* Get a quantity */
	amt = get_quantity(NULL, o_ptr->number);
	if (amt <= 0) return;

	/* Describe the destroyed object by taking a copy with the right "amt" */ 
	object_copy_amt(&obj_to_destroy, o_ptr, amt); 
	object_desc(o_name, _countof(o_name), &obj_to_destroy, TRUE, ODESC_FULL);

	/* Verify destruction */
	strnfmt(out_val, _countof(out_val), L"Really destroy %s? ", o_name);
	if (!get_check(out_val)) return;

	/* Tell the game to destroy the item. */
	cmd_insert(CMD_DESTROY, item, amt);

	/* Check for squelching */
	if (squelch_tval(o_ptr->tval))
	{
		_TCHAR sval_name[50];

		/* Obtain plural form without a quantity */
		object_desc(sval_name, sizeof sval_name, o_ptr, FALSE,
				ODESC_BASE | ODESC_PLURAL);
		strnfmt(out_val, sizeof out_val, L"Ignore %s in future? ",
				sval_name);

		if (get_check(out_val))
		{
			/* squelch_set_squelch(tval, sval); */
			k_info[o_ptr->k_idx].squelch = TRUE;
			p_ptr->notice |= PN_SQUELCH;
			msg_format(L"Ignoring %s from now on.", sval_name);
		}		
	}
}

void refill_lamp(object_type *j_ptr, object_type *o_ptr, int item)
{
	/* Refuel */
	j_ptr->timeout += o_ptr->timeout ? o_ptr->timeout : o_ptr->pval;

	/* Message */
	msg_print(L"You fuel your lamp.");

	/* Comment */
	if (j_ptr->timeout >= FUEL_LAMP)
	{
		j_ptr->timeout = FUEL_LAMP;
		msg_print(L"Your lamp is full.");
	}
	/* Refilled from a lantern */
	if (o_ptr->sval == SV_LITE_LANTERN)
	{
		/* Unstack if necessary */
		if (o_ptr->number > 1)
		{
			object_type *i_ptr;
			object_type object_type_body;

			/* Get local object */
			i_ptr = &object_type_body;

			/* Obtain a local object */
			object_copy(i_ptr, o_ptr);

			/* Modify quantity */
			i_ptr->number = 1;

			/* Remove fuel */
			i_ptr->timeout = 0;

			/* Unstack the used item */
			o_ptr->number--;
			p_ptr->total_weight -= i_ptr->weight;

			/* Carry or drop */
			if (item >= 0)
				item = inven_carry(i_ptr);
			else
				drop_near(i_ptr, 0, p_ptr->py, p_ptr->px);
		}
		else /* Empty a single lantern */
		{
			/* No more fuel */
			o_ptr->timeout = 0;
		}
		/* Combine / Reorder the pack (later) */
		p_ptr->notice |= (PN_COMBINE | PN_REORDER);

		/* Redraw stuff */
		p_ptr->redraw |= (PR_INVEN);
	}
	/* Refilled from a flask */
	else
	{
		/* Decrease the item (from the pack) */
		if (item >= 0)
		{
			inven_item_increase(item, -1);
			inven_item_describe(item);
			inven_item_optimize(item);
		}
		else /* Decrease the item (from the floor) */
		{
			floor_item_increase(0 - item, -1);
			floor_item_describe(0 - item);
			floor_item_optimize(0 - item);
		}
	}
	/* Recalculate torch */
	p_ptr->update |= (PU_TORCH);

	/* Redraw stuff */
	p_ptr->redraw |= (PR_EQUIP);
}

void refuel_torch(object_type *j_ptr, object_type *o_ptr, int item)
{
	/* Refuel */
	j_ptr->timeout += o_ptr->timeout + 5;

	/* Message */
	msg_print(L"You combine the torches.");

	/* Over-fuel message */
	if (j_ptr->timeout >= FUEL_TORCH)
	{
		j_ptr->timeout = FUEL_TORCH;
		msg_print(L"Your torch is fully fueled.");
	}
	else /* Refuel message */
	{
		msg_print(L"Your torch glows more brightly.");
	}
	/* Decrease the item (from the pack) */
	if (item >= 0)
	{
		inven_item_increase(item, -1);
		inven_item_describe(item);
		inven_item_optimize(item);
	}
	else /* Decrease the item (from the floor) */
	{
		floor_item_increase(0 - item, -1);
		floor_item_describe(0 - item);
		floor_item_optimize(0 - item);
	}
	/* Recalculate torch */
	p_ptr->update |= (PU_TORCH);

	/* Redraw stuff */
	p_ptr->redraw |= (PR_EQUIP);
}

/*
 * Target command
 */
void do_cmd_target(void)
{
	/* Target set */
	if (target_set_interactive(TARGET_KILL, -1, -1))
	{
		msg_print(L"Target Selected.");
	}
	else /* Target aborted */
	{
		msg_print(L"Target Aborted.");
	}
}

/*
 * Look command
 */
void do_cmd_look(void)
{
	/* Look around */
	if (target_set_interactive(TARGET_LOOK, -1, -1))
	{
		msg_print(L"Target Selected.");
	}
}

/*
 * Allow the player to examine other sectors on the map
 */
void do_cmd_locate(void)
{
	int dir, y1, x1, y2, x2;

	_TCHAR tmp_val[80];
	_TCHAR out_val[160];

	/* Start at current panel */
	y1 = Term->offset_y;
	x1 = Term->offset_x;

	/* Show panels until done */
	while (1)
	{
		/* Get the current panel */
		y2 = Term->offset_y;
		x2 = Term->offset_x;
		
		/* Describe the location */
		if ((y2 == y1) && (x2 == x1))
		{
			tmp_val[0] = 0;
		}
		else
		{
			strnfmt(tmp_val, _countof(tmp_val), L"%s%s of",
			        ((y2 < y1) ? L" north" : (y2 > y1) ? L" south" : __T("")),
			        ((x2 < x1) ? L" west" : (x2 > x1) ? L" east" : __T("")));
		}
		/* Prepare to ask which way to look */
		strnfmt(out_val, _countof(out_val),
		        L"Map sector [%d,%d], which is%s your sector.  Direction?",
		        (y2 / PANEL_HGT), (x2 / PANEL_WID), tmp_val);

		/* More detail */
		if (OPT(center_player))
		{
			strnfmt(out_val, _countof(out_val),
		        	L"Map sector [%d(%02d),%d(%02d)], which is%s your sector.  Direction?",
		        	(y2 / PANEL_HGT), (y2 % PANEL_HGT),
		        	(x2 / PANEL_WID), (x2 % PANEL_WID), tmp_val);
		}
		/* Assume no direction */
		dir = 0;

		/* Get a direction */
		while (!dir)
		{
			_TCHAR command;

			/* Get a command (or Cancel) */
			if (!get_com(out_val, &command)) break;

			/* Extract direction */
			dir = target_dir(command);

			/* Error */
			if (!dir) 
				bell(L"Illegal direction for locate!");
		}
		/* No direction */
		if (!dir) break;

		/* Apply the motion */
		change_panel(dir);

		/* Handle stuff */
		handle_stuff();
	}
	/* Verify panel */
	verify_panel();
}

/*
 * The table of "symbol info" -- each entry is a string of the form
 * "X:desc" where "X" is the trigger, and "desc" is the "info".
 */
static const _TCHAR *ident_info[] =
{
	L" :A dark grid",
	L"!:A potion (or oil)",
	L"\":An amulet (or necklace)",
	L"#:A wall (or secret door)",
	L"$:Treasure (gold or gems)",
	L"%:A vein (magma or quartz)",
	/* "&:unused", */
	L"':An open door",
	L"(:Soft armor",
	L"):A shield",
	L"*:A vein with treasure",
	L"+:A closed door",
	L",:Food (or mushroom patch)",
	L"-:A wand (or rod)",
	L".:Floor",
	L"/:A polearm (Axe/Pike/etc)",
	/* "0:unused", */
	L"1:Entrance to General Store",
	L"2:Entrance to Armory",
	L"3:Entrance to Weaponsmith",
	L"4:Entrance to Temple",
	L"5:Entrance to Alchemy shop",
	L"6:Entrance to Magic store",
	L"7:Entrance to Black Market",
	L"8:Entrance to your home",
	/* "9:unused", */
	L"::Rubble",
	L";:A glyph of warding",
	L"<:An up staircase",
	L"=:A ring",
	L">:A down staircase",
	L"?:A scroll",
	L"@:You",
	L"A:Angel",
	L"B:Bird",
	L"C:Canine",
	L"D:Ancient Dragon/Wyrm",
	L"E:Elemental",
	L"F:Dragon Fly",
	L"G:Ghost",
	L"H:Hybrid",
	L"I:Insect",
	L"J:Snake",
	L"K:Killer Beetle",
	L"L:Lich",
	L"M:Multi-Headed Reptile",
	/* "N:unused", */
	L"O:Ogre",
	L"P:Giant Humanoid",
	L"Q:Quylthulg (Pulsing Flesh Mound)",
	L"R:Reptile/Amphibian",
	L"S:Spider/Scorpion/Tick",
	L"T:Troll",
	L"U:Major Demon",
	L"V:Vampire",
	L"W:Wight/Wraith/etc",
	L"X:Xorn/Xaren/etc",
	L"Y:Yeti",
	L"Z:Zephyr Hound",
	L"[:Hard armor",
	L"\\:A hafted weapon (mace/whip/etc)",
	L"]:Misc. armor",
	L"^:A trap",
	L"_:A staff",
	/* "`:unused", */
	L"a:Ant",
	L"b:Bat",
	L"c:Centipede",
	L"d:Dragon",
	L"e:Floating Eye",
	L"f:Feline",
	L"g:Golem",
	L"h:Hobbit/Elf/Dwarf",
	L"i:Icky Thing",
	L"j:Jelly",
	L"k:Kobold",
	L"l:Louse",
	L"m:Mold",
	L"n:Naga",
	L"o:Orc",
	L"p:Person/Human",
	L"q:Quadruped",
	L"r:Rodent",
	L"s:Skeleton",
	L"t:Townsperson",
	L"u:Minor Demon",
	L"v:Vortex",
	L"w:Worm/Worm-Mass",
	/* "x:unused", */
	L"y:Yeek",
	L"z:Zombie/Mummy",
	L"{:A missile (arrow/bolt/shot)",
	L"|:An edged weapon (sword/dagger/etc)",
	L"}:A launcher (bow/crossbow/sling)",
	L"~:A tool (or miscellaneous item)",
	NULL
};

/*
 * Sorting hook -- Comp function -- see below
 *
 * We use "u" to point to array of monster indexes,
 * and "v" to select the type of sorting to perform on "u".
 */
bool ang_sort_comp_hook(const void *u, const void *v, int a, int b)
{
	const u16b *who = (const u16b *)(u);
	const u16b *why = (const u16b *)(v);

	int w1 = who[a];
	int w2 = who[b];

	int z1, z2;


	/* Sort by player kills */
	if (*why >= 4)
	{
		/* Extract player kills */
		z1 = l_list[w1].pkills;
		z2 = l_list[w2].pkills;

		/* Compare player kills */
		if (z1 < z2) return (TRUE);
		if (z1 > z2) return (FALSE);
	}
	/* Sort by total kills */
	if (*why >= 3)
	{
		/* Extract total kills */
		z1 = l_list[w1].tkills;
		z2 = l_list[w2].tkills;

		/* Compare total kills */
		if (z1 < z2) return (TRUE);
		if (z1 > z2) return (FALSE);
	}
	/* Sort by monster level */
	if (*why >= 2)
	{
		/* Extract levels */
		z1 = r_info[w1].level;
		z2 = r_info[w2].level;

		/* Compare levels */
		if (z1 < z2) return (TRUE);
		if (z1 > z2) return (FALSE);
	}
	/* Sort by monster experience */
	if (*why >= 1)
	{
		/* Extract experience */
		z1 = r_info[w1].mexp;
		z2 = r_info[w2].mexp;

		/* Compare experience */
		if (z1 < z2) return (TRUE);
		if (z1 > z2) return (FALSE);
	}

	/* Compare indexes */
	return (w1 <= w2);
}

/*
 * Sorting hook -- Swap function -- see below
 *
 * We use "u" to point to array of monster indexes,
 * and "v" to select the type of sorting to perform.
 */
void ang_sort_swap_hook(void *u, void *v, int a, int b)
{
	u16b *who = (u16b*)(u);
	u16b holder;

	UNREFERENCED_PARAMETER(v);

	/* Swap */
	holder = who[a];
	who[a] = who[b];
	who[b] = holder;
}

/*
 * Identify a character, allow recall of monsters
 *
 * Several "special" responses recall "multiple" monsters:
 *   ^A (all monsters)
 *   ^U (all unique monsters)
 *   ^N (all non-unique monsters)
 *
 * The responses may be sorted in several ways, see below.
 *
 * Note that the player ghosts are ignored, since they do not exist.
 */
void do_cmd_query_symbol(void)
{
	int i, n, r_idx;
	_TCHAR sym;
	_TCHAR buf[128];

	ui_event_data query;

	bool all = FALSE;
	bool uniq = FALSE;
	bool norm = FALSE;

	bool recall = FALSE;

	u16b why = 0;
	u16b *who;

	/* Get a character, or abort */
	if (!get_com(L"Enter character to be identified, or control+[ANU]: ", &sym)) return;

	/* Find that character info, and describe it */
	for (i = 0; ident_info[i]; ++i)
	{
		if (sym == ident_info[i][0]) break;
	}
	/* Describe */
	if (sym == KTRL('A'))
	{
		all = TRUE;
		_tcscpy_s(buf, _countof(buf), L"Full monster list.");
	}
	else if (sym == KTRL('U'))
	{
		all = uniq = TRUE;
		_tcscpy_s(buf, _countof(buf), L"Unique monster list.");
	}
	else if (sym == KTRL('N'))
	{
		all = norm = TRUE;
		_tcscpy_s(buf, _countof(buf), L"Non-unique monster list.");
	}
	else if (ident_info[i])
	{
		strnfmt(buf, _countof(buf), L"%c - %s.", sym, ident_info[i] + 2);
	}
	else
	{
		strnfmt(buf, _countof(buf), L"%c - %s.", sym, L"Unknown Symbol");
	}
	/* Display the result */
	prt(buf, 0, 0);

	/* Allocate the "who" array */
	who = C_ZNEW(z_info->r_max, u16b);

	/* Collect matching monsters */
	for (n = 0, i = 1; i < z_info->r_max - 1; i++)
	{
		monster_race *r_ptr = &r_info[i];
		monster_lore *l_ptr = &l_list[i];

		/* Nothing to recall */
		if (!OPT(cheat_know) && !l_ptr->sights) continue;

		/* Require non-unique monsters if needed */
		if (norm && (r_ptr->flags[0] & (RF0_UNIQUE))) continue;

		/* Require unique monsters if needed */
		if (uniq && !(r_ptr->flags[0] & (RF0_UNIQUE))) continue;

		/* Collect "appropriate" monsters */
		if (all || (r_ptr->d_char == sym)) who[n++] = i;
	}
	/* Nothing to recall */
	if (!n)
	{
		/* XXX XXX Free the "who" array */
		FREE(who);

		return;
	}
	/* Buttons */
	button_add(L"[y]", 'y');
	button_add(L"[k]", 'k');
	/* Don't collide with the repeat button */
	button_add(L"[n]", 'q'); 
	redraw_stuff();

	/* Prompt */
	put_str(L"Recall details? (y/k/n): ", 0, 40);

	/* Query */
	query = inkey_ex();

	/* Restore */
	prt(buf, 0, 0);

	/* Buttons */
	button_kill('y');
	button_kill('k');
	button_kill('q');
	redraw_stuff();

	/* Interpret the response */
	if (query.key == 'k')
	{
		/* Sort by kills (and level) */
		why = 4;
	}
	else if (query.key == 'y' || query.key == 'p')
	{
		/* Sort by level; accept 'p' as legacy */
		why = 2;
	}
	else
	{
		/* Any unsupported response is "nope, no history please" */
	
		/* XXX XXX Free the "who" array */
		FREE(who);

		return;
	}

	/* Select the sort method */
	ang_sort_comp = ang_sort_comp_hook;
	ang_sort_swap = ang_sort_swap_hook;

	/* Sort the array */
	ang_sort(who, &why, n);

	/* Start at the end */
	i = n - 1;

	/* Button */
	button_add(L"[r]", 'r');
	button_add(L"[-]", '-');
	button_add(L"[+]", '+');
	redraw_stuff();

	/* Scan the monster memory */
	while (1)
	{
		/* Extract a race */
		r_idx = who[i];

		/* Hack -- Auto-recall */
		monster_race_track(r_idx);

		/* Hack -- Handle stuff */
		handle_stuff();

		/* Hack -- Begin the prompt */
		roff_top(r_idx);

		/* Hack -- Complete the prompt */
		Term_addstr(-1, TERM_WHITE, L" [(r)ecall, ESC]");

		/* Interact */
		while (1)
		{
			/* Recall */
			if (recall)
			{
				/* Save screen */
				screen_save();

				/* Recall on screen */
				screen_roff(who[i]);

				/* Hack -- Complete the prompt (again) */
				Term_addstr(-1, TERM_WHITE, L" [(r)ecall, ESC]");
			}
			/* Command */
			query = inkey_ex();

			/* Unrecall */
			if (recall)
			{
				/* Load screen */
				screen_load();
			}

			/* Normal commands */
			if (query.key != 'r') break;

			/* Toggle recall */
			recall = !recall;
		}

		/* Stop scanning */
		if (query.key == ESCAPE) break;

		/* Move to "prev" monster */
		if (query.key == '-')
		{
			if (++i == n)
				i = 0;
		}
		else /* Move to "next" monster */
		{
			if (i-- == 0)
				i = n - 1;
		}
	}
	/* Button */
	button_kill('r');
	button_kill('-');
	button_kill('+');
	redraw_stuff();

	/* Re-display the identity */
	prt(buf, 0, 0);

	/* Free the "who" array */
	FREE(who);
}
