/* テスト */
/*
 * Debug Status: Not checked.
 * - Fixed __T on this page.
 */
/*
 * File: score.c
 * Purpose: Highscore handling for Angband
 *
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke
 *
 * This work is free software; you can redistribute it and/or modify it
 * under the terms of either:
 *
 * a) the GNU General Public License as published by the Free Software
 *    Foundation, version 2, or
 *
 * b) the "Angband licence":
 *    This software may be copied and distributed for educational, research,
 *    and not for profit purposes provided that this copyright and statement
 *    are included in all such copies.  Other copyrights may also apply.
 */
extern "C"
{
#include "angband.h"
}
extern void message_flush(void);
extern void screen_save(void);
extern void screen_load(void);
extern void msg_print(const _TCHAR *msg);
extern void prt(const _TCHAR *str, int row, int col);
extern void c_put_str(byte attr, const _TCHAR *str, int row, int col);
extern void put_str(const _TCHAR *str, int row, int col);
extern size_t strnfmt(_TCHAR *buf, size_t max, const _TCHAR *fmt, ...);

/*
 * Semi-Portable High Score List Entry (128 bytes)
 *
 * All fields listed below are null terminated ascii strings.
 *
 * In addition, the "number" fields are right justified, and
 * space padded, to the full available length (minus the "null").
 *
 * Note that "string comparisons" are thus valid on "pts".
 */
typedef struct
{
	_TCHAR what[8];		/* Version info (string) */

	_TCHAR pts[10];		/* Total Score (number) */

	_TCHAR gold[10];		/* Total Gold (number) */

	_TCHAR turns[10];		/* Turns Taken (number) */

	_TCHAR day[10];		/* Time stamp (string) */

	_TCHAR who[16];		/* Player Name (string) */

	_TCHAR uid[8];		/* Player UID (number) */

	_TCHAR sex[2];		/* Player Sex (string) */
	_TCHAR p_r[3];		/* Player Race (number) */
	_TCHAR p_c[3];		/* Player Class (number) */

	_TCHAR cur_lev[4];		/* Current Player Level (number) */
	_TCHAR cur_dun[4];		/* Current Dungeon Level (number) */
	_TCHAR max_lev[4];		/* Max Player Level (number) */
	_TCHAR max_dun[4];		/* Max Dungeon Level (number) */

	_TCHAR how[32];		/* Method of death (string) */
} high_score;

/*
 * Hack -- Calculates the total number of points earned
 */
static long total_points(void)
{
	return (p_ptr->max_exp + (100 * p_ptr->max_depth));
}

/*
 * Read in a highscore file.
 */
static size_t highscore_read(high_score scores[], size_t sz)
{
	_TCHAR fname[1024];
	ang_file *scorefile;
	size_t i;
	size_t n_read = 0;

	/* Wipe current scores */
	C_WIPE(scores, sz, high_score);

	path_build(fname, _countof(fname), ANGBAND_DIR_APEX, __T("scores.raw"));
	scorefile = file_open(fname, MODE_READ_BIN, -1);

	if (!scorefile) 
		return TRUE;

	/* TODO Check this works */
	for (i = 0; i < sz && file_read(scorefile, (_TCHAR *)&scores[i], sizeof(high_score), &n_read) == 0; i++)
		;

	file_close(scorefile);

	return i;
}

/*
 * Just determine where a new score *would* be placed
 * Return the location (0 is best) or -1 on failure
 */
static size_t highscore_where(const high_score *entry, const high_score scores[], size_t sz)
{
	size_t i;

	/* Read until we get to a higher score */
	for (i = 0; i < sz; i++)
	{
		long entry_pts = _tcstoul(entry->pts, NULL, 0);
		long score_pts = _tcstoul(scores[i].pts, NULL, 0);

		if (entry_pts >= score_pts)
			return i;

		if (scores[i].what[0] == 0)
			return i;
	}

	/* The last entry is always usable */
	return sz - 1;
}

static size_t highscore_add(const high_score *entry, high_score scores[], size_t sz)
{
	size_t slot = highscore_where(entry, scores, sz);

	memmove(&scores[slot+1], &scores[slot], sizeof(high_score) * (sz - 1 - slot));
	memcpy(&scores[slot], entry, sizeof(high_score));

	return slot;
}

static size_t highscore_count(const high_score scores[], size_t sz)
{
	size_t i;
	for (i = 0; i < sz; i++)
	{
		if (scores[i].what[0] == 0) break;
	}

	return i;
}

/*
 * Actually place an entry into the high score file
 * Return the location (0 is best) or -1 on "failure"
 */
static void highscore_write(const high_score scores[], size_t sz)
{
	size_t n;

	ang_file *lok;
	ang_file *scorefile;

	_TCHAR old_name[1024];
	_TCHAR cur_name[1024];
	_TCHAR new_name[1024];
	_TCHAR lok_name[1024];

	path_build(old_name, _countof(old_name), ANGBAND_DIR_APEX, __T("scores.old"));
	path_build(cur_name, _countof(cur_name), ANGBAND_DIR_APEX, __T("scores.raw"));
	path_build(new_name, _countof(new_name), ANGBAND_DIR_APEX, __T("scores.new"));
	path_build(lok_name, _countof(lok_name), ANGBAND_DIR_APEX, __T("scores.lok"));

	/* Read in and add new score */
	n = highscore_count(scores, sz);

	/*** Lock scores ***/

	if (_file_exists(lok_name))
	{
		msg_print(__T("Lock file in place for scorefile; not writing."));
		return;
	}

	safe_setuid_grab();
	lok = file_open(lok_name, MODE_WRITE_BIN, FTYPE_RAW);
	file_lock(lok);
	safe_setuid_drop();

	if (!lok)
	{
		msg_print(__T("Failed to create lock for scorefile; not writing."));
		return;
	}

	/*** Open the new file for writing ***/

	safe_setuid_grab();
	scorefile = file_open(new_name, MODE_WRITE_BIN, FTYPE_RAW);
	safe_setuid_drop();

	if (!scorefile)
	{
		msg_print(__T("Failed to open new scorefile for writing."));

		file_close(lok);
		file_delete(lok_name);
		return;
	}

	file_write(scorefile, (const char*)scores, sizeof(high_score)*n);
	file_close(scorefile);

	/*** Now move things around ***/

	safe_setuid_grab();

	if (_file_exists(old_name) && !file_delete(old_name))
		msg_print(__T("Couldn't delete old scorefile"));

	if (_file_exists(cur_name) && !file_move(cur_name, old_name))
		msg_print(__T("Couldn't move old scores.raw out of the way"));

	if (!file_move(new_name, cur_name))
		msg_print(__T("Couldn't rename new scorefile to scores.raw"));

	/* Remove the lock */
	file_close(lok);
	file_delete(lok_name);

	safe_setuid_drop();
}

/*
 * Display the scores in a given range.
 */
static void display_scores_aux(const high_score scores[], int from, int to, int highlight)
{
	_TCHAR ch;

	int j, k, n, place;
	int count;

	/* Assume we will show the first 10 */
	if (from < 0)
		from = 0;
	if (to < 0)
		to = 10;
	if (to > MAX_HISCORES)
		to = MAX_HISCORES;

	/* Hack -- Count the high scores */
	for (count = 0; count < MAX_HISCORES; count++)
	{
		if (!scores[count].what[0])
			break;
	}
	/* Forget about the last entries */
	if (count > to)
		count = to;

	/* Show 5 per page, until "done" */
	for (k = from, j = from, place = k+1; k < count; k += 5)
	{
		_TCHAR out_val[160];
		_TCHAR tmp_val[160];

		/* Clear screen */
		Term_clear();

		/* Title */
		put_str(format(__T("%s Hall of Fame"), VERSION_NAME), 0, 26);

		/* Indicate non-top scores */
		if (k > 0)
			put_str(format(__T("(from position %d)"), place), 0, 40);

		/* Dump 5 entries */
		for (n = 0; j < count && n < 5; place++, j++, n++)
		{
			const high_score *score = &scores[j];

			byte attr;

			int pr, pc, clev, mlev, cdun, mdun;
			const _TCHAR *user;
			const _TCHAR *gold;
			const _TCHAR *when;
			const _TCHAR *aged;

			/* Hack -- indicate death in yellow */
			attr = (j == highlight) ? TERM_L_GREEN : TERM_WHITE;

			/* Extract the race/class */
			pr = _tstoi(score->p_r);
			pc = _tstoi(score->p_c);

			/* Extract the level info */
			clev = _tstoi(score->cur_lev);
			mlev = _tstoi(score->max_lev);
			cdun = _tstoi(score->cur_dun);
			mdun = _tstoi(score->max_dun);

			/* Hack -- extract the gold and such */
			for (user = score->uid; _istspace(*user); user++) /* loop */;
				for (when = score->day; _istspace(*when); when++) /* loop */;
					for (gold = score->gold; _istspace(*gold); gold++) /* loop */;
						for (aged = score->turns; _istspace(*aged); aged++) /* loop */;

			/* Dump some info */
			strnfmt(out_val, _countof(out_val),
			        __T("%3d.%9s  %s the %s %s, Level %d"),
			        place, score->pts, score->who,
			        p_name + p_info[pr].name, c_name + c_info[pc].name,
			        clev);

			/* Append a "maximum level" */
			if (mlev > clev)
				_tcscat_s(out_val, _countof(out_val), format(__T(" (Max %d)"), mlev));

			/* Dump the first line */
			c_put_str(attr, out_val, n*4 + 2, 0);

			/* Died where? */
			if (!cdun)
				strnfmt(out_val, _countof(out_val), __T("Killed by %s in the town"), score->how);
			else
				strnfmt(out_val, _countof(out_val), __T("Killed by %s on dungeon level %d"), score->how, cdun);

			/* Append a "maximum level" */
			if (mdun > cdun)
				_tcscat_s(out_val, _countof(out_val), format(__T(" (Max %d)"), mdun));

			/* Dump the info */
			c_put_str(attr, out_val, n*4 + 3, 15);

			/* Clean up standard encoded form of "when" */
			if ((*when == '@') && _tcslen(when) == 9)
			{
				strnfmt(tmp_val, _countof(tmp_val), __T("%.4s-%.2s-%.2s"), when + 1, when + 5, when + 7);
				when = tmp_val;
			}

			/* And still another line of info */
			strnfmt(out_val, _countof(out_val), __T("(User %s, Date %s, Gold %s, Turn %s)."), user, when, gold, aged);
			c_put_str(attr, out_val, n*4 + 4, 15);
		}

		/* Wait for response */
		prt(__T("[Press ESC to exit, any other key to continue.]"), 23, 17);
		ch = inkey();
		prt(__T(""), 23, 0);

		/* Hack -- notice Escape */
		if (ch == ESCAPE) break;
	}

	return;
}

static void build_score(high_score *entry, const _TCHAR *died_from, time_t *death_time)
{
	struct tm * timeinfo=NULL;

	WIPE(entry, high_score);

	/* Save the version */
	strnfmt(entry->what, _countof(entry->what), __T("%s"), VERSION_STRING);

	/* Calculate and save the points */
	strnfmt(entry->pts, _countof(entry->pts), __T("%9lu"), (long)total_points());

	/* Save the current gold */
	strnfmt(entry->gold, _countof(entry->gold), __T("%9lu"), (long)p_ptr->au);

	/* Save the current turn */
	strnfmt(entry->turns, _countof(entry->turns), __T("%9lu"), (long)turn);

	/* Time of death */
	if (death_time)
	{
#if defined(_MSC_VER) && (_MSC_VER >= 1400)
		timeinfo = RNEW(struct tm);
		localtime_s(timeinfo, death_time);
#else
		timeinfo = localtime(death_time);
#endif
		_tcsftime(entry->day, _countof(entry->day), __T("@%Y%m%d"), timeinfo);
	}
	else
		_tcscpy_s(entry->day, _countof(entry->day), __T("TODAY"));

	/* Save the player name (15 chars) */
	strnfmt(entry->who, _countof(entry->who), __T("%-.15s"), op_ptr->full_name);

	/* Save the player info XXX XXX XXX */
	strnfmt(entry->uid, _countof(entry->uid), __T("%7u"), player_uid);
	strnfmt(entry->sex, _countof(entry->sex), __T("%c"), (p_ptr->psex ? 'm' : 'f'));
	strnfmt(entry->p_r, _countof(entry->p_r), __T("%2d"), p_ptr->prace);
	strnfmt(entry->p_c, _countof(entry->p_c), __T("%2d"), p_ptr->pclass);

	/* Save the level and such */
	strnfmt(entry->cur_lev, _countof(entry->cur_lev), __T("%3d"), p_ptr->lev);
	strnfmt(entry->cur_dun, _countof(entry->cur_dun), __T("%3d"), p_ptr->depth);
	strnfmt(entry->max_lev, _countof(entry->max_lev), __T("%3d"), p_ptr->max_lev);
	strnfmt(entry->max_dun, _countof(entry->max_dun), __T("%3d"), p_ptr->max_depth);

	/* No cause of death */
	_tcscpy_s(entry->how, _countof(entry->how), died_from);
}

/*
 * Enters a players name on a hi-score table, if "legal".
 *
 * Assumes "signals_ignore_tstp()" has been called.
 */
void enter_score(time_t *death_time)
{
	int j;

	/* Cheaters are not scored */
	for (j = OPT_SCORE; j < OPT_MAX; ++j)
	{
		if (!op_ptr->opt[j]) continue;

		msg_print(__T("Score not registered for cheaters."));
		message_flush();
		return;
	}

	/* Wizard-mode pre-empts scoring */
	if (p_ptr->noscore & (NOSCORE_WIZARD | NOSCORE_DEBUG))
	{
		msg_print(__T("Score not registered for wizards."));
		message_flush();
	}

#ifndef SCORE_BORGS

	/* Borg-mode pre-empts scoring */
	else if (p_ptr->noscore & NOSCORE_BORG)
	{
		msg_print(__T("Score not registered for borgs."));
		message_flush();
	}

#endif /* SCORE_BORGS */

	/* Hack -- Interupted */
	else if (!p_ptr->total_winner && streq(p_ptr->died_from, __T("Interrupting")))
	{
		msg_print(__T("Score not registered due to interruption."));
		message_flush();
	}
	/* Hack -- Quitter */
	else if (!p_ptr->total_winner && streq(p_ptr->died_from, __T("Quitting")))
	{
		msg_print(__T("Score not registered due to quitting."));
		message_flush();
	}
	/* Add a new entry to the score list, see where it went */
	else
	{
		high_score entry;
		high_score scores[MAX_HISCORES];

		build_score(&entry, p_ptr->died_from, death_time);

		highscore_read(scores, N_ELEMENTS(scores));
		highscore_add(&entry, scores, N_ELEMENTS(scores));
		highscore_write(scores, N_ELEMENTS(scores));
	}

	/* Success */
	return;
}

/*
 * Predict the players location, and display it.
 */
void predict_score(void)
{
	int j;
	high_score the_score;

	high_score scores[MAX_HISCORES];

	/* Read scores, place current score */
	highscore_read(scores, N_ELEMENTS(scores));
	build_score(&the_score, __T("nobody (yet!)"), NULL);

	if (p_ptr->is_dead)
		j = highscore_where(&the_score, scores, N_ELEMENTS(scores));
	else
		j = highscore_add(&the_score, scores, N_ELEMENTS(scores));

	/* Hack -- Display the top fifteen scores */
	if (j < 10)
	{
		display_scores_aux(scores, 0, 15, j);
	}
	else /* Display some "useful" scores */
	{
		display_scores_aux(scores, 0, 5, -1);
		display_scores_aux(scores, j - 2, j + 7, j);
	}
}

/*
 * Show scores.
 */
void show_scores(void)
{
	screen_save();

	/* Display the scores */
	if (character_generated)
	{
		predict_score();
	}
	else
	{
		high_score scores[MAX_HISCORES];
		highscore_read(scores, N_ELEMENTS(scores));
		display_scores_aux(scores, 0, MAX_HISCORES, -1);
	}

	screen_load();

	/* Hack - Flush it */
	Term_fresh();
}
