﻿/* テスト */
/* 
 * Debug Status: Not checked.
 * - Fixed __T on this page.
 */
/*
 * File: cmd2.c
 * Purpose: Chest and door opening/closing, disarming, running, resting, &c.
 *
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke
 *
 * This work is free software; you can redistribute it and/or modify it
 * under the terms of either:
 *
 * a) the GNU General Public License as published by the Free Software
 *    Foundation, version 2, or
 *
 * b) the "Angband licence":
 *    This software may be copied and distributed for educational, research,
 *    and not for profit purposes provided that this copyright and statement
 *    are included in all such copies.  Other copyrights may also apply.
 */
extern "C" 
{
#include "angband.h"
#include "object/tvalsval.h"
#include "cmds.h"
#include "game-cmd.h"
}

extern errr cmd_insert(cmd_code c, ...);
extern void msg_print(const _TCHAR *msg);
extern bool get_check(const _TCHAR *prompt);
extern void prt(const _TCHAR *str, int row, int col);

/*
 * Go up one level
 */
void do_cmd_go_up(cmd_code code, cmd_arg args[])
{
	/* Verify stairs */
	if (cave_feat[p_ptr->py][p_ptr->px] != FEAT_LESS)
	{
		msg_print(__T("I see no up staircase here."));
		return;
	}
	/* Ironman */
	if (OPT(adult_ironman))
	{
		msg_print(__T("Nothing happens!"));
		return;
	}
	/* Hack -- take a turn */
	p_ptr->energy_use = 100;

	/* Success */
	message(MSG_STAIRS_UP, 0, L"You enter a maze of up staircases.");

	/* Create a way back */
	p_ptr->create_up_stair = FALSE;
	p_ptr->create_down_stair = TRUE;

	/* Change level */
	dungeon_change_level(p_ptr->depth - 1);
}

/*
 * Go down one level
 */
void do_cmd_go_down(cmd_code code, cmd_arg args[])
{
	/* Verify stairs */
	if (cave_feat[p_ptr->py][p_ptr->px] != FEAT_MORE)
	{
		msg_print(__T("I see no down staircase here."));
		return;
	}

	/* Hack -- take a turn */
	p_ptr->energy_use = 100;

	/* Success */
	message(MSG_STAIRS_DOWN, 0, __T("You enter a maze of down staircases."));

	/* Create a way back */
	p_ptr->create_up_stair = TRUE;
	p_ptr->create_down_stair = FALSE;

	/* Change level */
	dungeon_change_level(p_ptr->depth + 1);
}

/*
 * Simple command to "search" for one turn
 */
void do_cmd_search(cmd_code code, cmd_arg args[])
{
	/* Take a turn */
	p_ptr->energy_use = 100;

	/* Search */
	search();
}

/*
 * Hack -- toggle search mode
 */
void do_cmd_toggle_search(cmd_code code, cmd_arg args[])
{
	/* Stop searching */
	if (p_ptr->searching)
	{
		/* Clear the searching flag */
		p_ptr->searching = FALSE;

		/* Recalculate bonuses */
		p_ptr->update |= (PU_BONUS);

		/* Redraw the state */
		p_ptr->redraw |= (PR_STATE);
	}
	else /* Start searching */
	{
		/* Set the searching flag */
		p_ptr->searching = TRUE;

		/* Update stuff */
		p_ptr->update |= (PU_BONUS);

		/* Redraw stuff */
		p_ptr->redraw |= (PR_STATE | PR_SPEED);
	}
}

/*
 * Determine if a grid contains a chest
 */
static s16b chest_check(int y, int x)
{
	s16b this_o_idx, next_o_idx = 0;

	/* Scan all objects in the grid */
	for (this_o_idx = cave_o_idx[y][x]; this_o_idx; this_o_idx = next_o_idx)
	{
		object_type *o_ptr;

		/* Get the object */
		o_ptr = &o_list[this_o_idx];

		/* Get the next object */
		next_o_idx = o_ptr->next_o_idx;

		/* Skip unknown chests XXX XXX */
		/* if (!o_ptr->marked) continue; */

		/* Check for chest */
		if (o_ptr->tval == TV_CHEST) return (this_o_idx);
	}
	/* No chest */
	return (0);
}

/*
 * Allocate objects upon opening a chest
 *
 * Disperse treasures from the given chest, centered at (x,y).
 *
 * Small chests often contain "gold", while Large chests always contain
 * items.  Wooden chests contain 2 items, Iron chests contain 4 items,
 * and Steel chests contain 6 items.  The "value" of the items in a
 * chest is based on the "power" of the chest, which is in turn based
 * on the level on which the chest is generated.
 */
static void chest_death(int y, int x, s16b o_idx)
{
	int number, value;
	bool tiny;

	object_type *o_ptr;

	object_type *i_ptr;
	object_type object_type_body;

	/* Get the chest */
	o_ptr = &o_list[o_idx];

	/* Small chests often hold "gold" */
	tiny = (o_ptr->sval < SV_CHEST_MIN_LARGE);

	/* Determine how much to drop (see above) */
	number = (o_ptr->sval % SV_CHEST_MIN_LARGE) * 2;

	/* Zero pval means empty chest */
	if (!o_ptr->pval) 
		number = 0;

	/* Opening a chest */
	opening_chest = TRUE;

	/* Determine the "value" of the items */
	value = ABS(o_ptr->pval) + 10;

	/* Drop some objects (non-chests) */
	for (; number > 0; --number)
	{
		/* Get local object */
		i_ptr = &object_type_body;

		/* Wipe the object */
		object_wipe(i_ptr);

		/* Small chests often drop gold */
		if (tiny && (randint0(100) < 75))
			make_gold(i_ptr, value, SV_GOLD_ANY);

		/* Otherwise drop an item */
		else
		{
			if (!make_object(i_ptr, value, FALSE, FALSE))
				continue;
		}
		/* Drop it in the dungeon */
		drop_near(i_ptr, -1, y, x);
	}
	/* No longer opening a chest */
	opening_chest = FALSE;

	/* Empty */
	o_ptr->pval = 0;

	/* Known */
	object_known(o_ptr);
}

/*
 * Chests have traps too.
 *
 * Exploding chest destroys contents (and traps).
 * Note that the chest itself is never destroyed.
 */
static void chest_trap(int y, int x, s16b o_idx)
{
	int i, trap;

	object_type *o_ptr = &o_list[o_idx];

	/* Ignore disarmed chests */
	if (o_ptr->pval <= 0) 
		return;

	/* Obtain the traps */
	trap = chest_traps[o_ptr->pval];

	/* Lose strength */
	if (trap & (CHEST_LOSE_STR))
	{
		msg_print(__T("A small needle has pricked you!"));
		take_hit(damroll(1, 4), L"a poison needle");
		(void)do_dec_stat(A_STR, FALSE);
	}
	/* Lose constitution */
	if (trap & (CHEST_LOSE_CON))
	{
		msg_print(__T("A small needle has pricked you!"));
		take_hit(damroll(1, 4), L"a poison needle");
		(void)do_dec_stat(A_CON, FALSE);
	}
	/* Poison */
	if (trap & (CHEST_POISON))
	{
		msg_print(__T("A puff of green gas surrounds you!"));
		if (!(p_ptr->state.resist_pois || p_ptr->timed[TMD_OPP_POIS]))
			(void)inc_timed(TMD_POISONED, 10 + randint1(20), TRUE);
		else if (p_ptr->state.resist_pois)
			object_notice_flag(1, TR1_RES_POIS);
	}
	/* Paralyze */
	if (trap & (CHEST_PARALYZE))
	{
		msg_print(__T("A puff of yellow gas surrounds you!"));
		if (!p_ptr->state.free_act)
			(void)inc_timed(TMD_PARALYZED, 10 + randint1(20), TRUE);
		else
			object_notice_flag(2, TR2_FREE_ACT);
	}
	/* Summon monsters */
	if (trap & (CHEST_SUMMON))
	{
		int num = 2 + randint1(3);
		msg_print(__T("You are enveloped in a cloud of smoke!"));
		sound(MSG_SUM_MONSTER);
		for (i = 0; i < num; i++)
		{
			(void)summon_specific(y, x, p_ptr->depth, 0);
		}
	}
	/* Explode */
	if (trap & (CHEST_EXPLODE))
	{
		msg_print(__T("There is a sudden explosion!"));
		msg_print(__T("Everything inside the chest is destroyed!"));
		o_ptr->pval = 0;
		take_hit(damroll(5, 8), L"an exploding chest");
	}
}

/*
 * Attempt to open the given chest at the given location
 *
 * Assume there is no monster blocking the destination
 *
 * Returns TRUE if repeated commands may continue
 */
static bool do_cmd_open_chest(int y, int x, s16b o_idx)
{
	int i, j;
	bool flag = TRUE;
	bool more = FALSE;

	object_type *o_ptr = &o_list[o_idx];

	/* Attempt to unlock it */
	if (o_ptr->pval > 0)
	{
		/* Assume locked, and thus not open */
		flag = FALSE;

		/* Get the "disarm" factor */
		i = p_ptr->state.skills[SKILL_DISARM];

		/* Penalize some conditions */
		if (p_ptr->timed[TMD_BLIND] || no_lite()) 
			i = i / 10;
		if (p_ptr->timed[TMD_CONFUSED] || p_ptr->timed[TMD_IMAGE]) 
			i = i / 10;

		/* Extract the difficulty */
		j = i - o_ptr->pval;

		/* Always have a small chance of success */
		if (j < 2) 
			j = 2;

		/* Success -- May still have traps */
		if (randint0(100) < j)
		{
			message(MSG_LOCKPICK, 0, L"You have picked the lock.");
			gain_exp(1);
			flag = TRUE;
		}
		else /* Failure -- Keep trying */
		{
			/* We may continue repeating */
			more = TRUE;
			if (OPT(flush_failure)) flush();
			message(MSG_LOCKPICK_FAIL, 0, L"You failed to pick the lock.");
		}
	}
	/* Allowed to open */
	if (flag)
	{
		/* Apply chest traps, if any */
		chest_trap(y, x, o_idx);

		/* Let the Chest drop items */
		chest_death(y, x, o_idx);

		/* Squelch chest if autosquelch calls for it */
		p_ptr->notice |= PN_SQUELCH;

		/* Redraw chest, to be on the safe side (it may have been squelched) */
		lite_spot(y, x);
	}
	/* Result */
	return (more);
}

/*
 * Attempt to disarm the chest at the given location
 *
 * Assume there is no monster blocking the destination
 *
 * Returns TRUE if repeated commands may continue
 */
static bool do_cmd_disarm_chest(int y, int x, s16b o_idx)
{
	int i, j;
	bool more = FALSE;

	object_type *o_ptr = &o_list[o_idx];

	/* Get the "disarm" factor */
	i = p_ptr->state.skills[SKILL_DISARM];

	/* Penalize some conditions */
	if (p_ptr->timed[TMD_BLIND] || no_lite()) i = i / 10;
	if (p_ptr->timed[TMD_CONFUSED] || p_ptr->timed[TMD_IMAGE]) i = i / 10;

	/* Extract the difficulty */
	j = i - o_ptr->pval;

	/* Always have a small chance of success */
	if (j < 2) 
		j = 2;

	/* Must find the trap first. */
	if (!object_known_p(o_ptr))
	{
		msg_print(__T("I don't see any traps."));
	}
	/* Already disarmed/unlocked */
	else if (o_ptr->pval <= 0)
	{
		msg_print(__T("The chest is not trapped."));
	}
	/* No traps to find. */
	else if (!chest_traps[o_ptr->pval])
	{
		msg_print(__T("The chest is not trapped."));
	}
	/* Success (get a lot of experience) */
	else if (randint0(100) < j)
	{
		message(MSG_DISARM, 0, L"You have disarmed the chest.");
		gain_exp(o_ptr->pval);
		o_ptr->pval = (0 - o_ptr->pval);
	}
	/* Failure -- Keep trying */
	else if ((i > 5) && (randint1(i) > 5))
	{
		/* We may keep trying */
		more = TRUE;
		if (OPT(flush_failure)) flush();
		msg_print(__T("You failed to disarm the chest."));
	}
	else /* Failure -- Set off the trap */
	{
		msg_print(__T("You set off a trap!"));
		chest_trap(y, x, o_idx);
	}
	/* Result */
	return (more);
}

/*
 * Return TRUE if the given feature is an open door
 */
static bool is_open(int feat)
{
	return (feat == FEAT_OPEN);
}

/*
 * Return TRUE if the given feature is a closed door
 */
static bool is_closed(int feat)
{
	return ((feat >= FEAT_DOOR_HEAD) &&
	        (feat <= FEAT_DOOR_TAIL));
}

/*
 * Return TRUE if the given feature is a trap
 */
static bool is_trap(int feat)
{
	return ((feat >= FEAT_TRAP_HEAD) &&
	        (feat <= FEAT_TRAP_TAIL));
}

/*
 * Return the number of doors/traps around (or under) the character.
 */
static int count_feats(int *y, int *x, bool (*test)(int feat), bool under)
{
	int d;
	int xx, yy;
	int count = 0; /* Count how many matches */

	/* Check around (and under) the character */
	for (d = 0; d < 9; d++)
	{
		/* if not searching under player continue */
		if ((d == 8) && !under) continue;

		/* Extract adjacent (legal) location */
		yy = p_ptr->py + ddy_ddd[d];
		xx = p_ptr->px + ddx_ddd[d];

		/* Paranoia */
		if (!in_bounds_fully(yy, xx)) 
			continue;

		/* Must have knowledge */
		if (!(cave_info[yy][xx] & (CAVE_MARK))) 
			continue;

		/* Not looking for this feature */
		if (!((*test)(cave_feat[yy][xx]))) 
			continue;

		/* Count it */
		++count;

		/* Remember the location of the last door found */
		*y = yy;
		*x = xx;
	}
	/* All done */
	return count;
}

/*
 * Return the number of chests around (or under) the character.
 * If requested, count only trapped chests.
 */
static int count_chests(int *y, int *x, bool trapped)
{
	int d, count, o_idx;

	object_type *o_ptr;

	/* Count how many matches */
	count = 0;

	/* Check around (and under) the character */
	for (d = 0; d < 9; d++)
	{
		/* Extract adjacent (legal) location */
		int yy = p_ptr->py + ddy_ddd[d];
		int xx = p_ptr->px + ddx_ddd[d];

		/* No (visible) chest is there */
		if ((o_idx = chest_check(yy, xx)) == 0) 
			continue;

		/* Grab the object */
		o_ptr = &o_list[o_idx];

		/* Already open */
		if (o_ptr->pval == 0) 
			continue;

		/* No (known) traps here */
		if (trapped && (!object_known_p(o_ptr) || (o_ptr->pval < 0) ||
				!chest_traps[o_ptr->pval]))
		{
			continue;
		}
		/* Count it */
		++count;

		/* Remember the location of the last chest found */
		*y = yy;
		*x = xx;
	}
	/* All done */
	return count;
}

/*
 * Extract a "direction" which will move one step from the player location
 * towards the given "target" location (or "5" if no motion necessary).
 */
static int coords_to_dir(int y, int x)
{
	return (motion_dir(p_ptr->py, p_ptr->px, y, x));
}

/*
 * Determine if a given grid may be "opened"
 */
static bool do_cmd_open_test(int y, int x)
{
	/* Must have knowledge */
	if (!(cave_info[y][x] & (CAVE_MARK)))
	{
		/* Message */
		msg_print(__T("You see nothing there."));

		/* Nope */
		return (FALSE);
	}

	/* Must be a closed door */
	if (!((cave_feat[y][x] >= FEAT_DOOR_HEAD) &&
	      (cave_feat[y][x] <= FEAT_DOOR_TAIL)))
	{
		/* Message */
		message(MSG_NOTHING_TO_OPEN, 0, L"You see nothing there to open.");

		/* Nope */
		return (FALSE);
	}
	/* Okay */
	return (TRUE);
}

/*
 * Perform the basic "open" command on doors
 *
 * Assume there is no monster blocking the destination
 *
 * Returns TRUE if repeated commands may continue
 */
static bool do_cmd_open_aux(int y, int x)
{
	int i, j;
	bool more = FALSE;

	/* Verify legality */
	if (!do_cmd_open_test(y, x)) 
		return (FALSE);

	/* Jammed door */
	if (cave_feat[y][x] >= FEAT_DOOR_HEAD + 0x08)
	{
		/* Stuck */
		msg_print(__T("The door appears to be stuck."));
	}
	/* Locked door */
	else if (cave_feat[y][x] >= FEAT_DOOR_HEAD + 0x01)
	{
		/* Disarm factor */
		i = p_ptr->state.skills[SKILL_DISARM];

		/* Penalize some conditions */
		if (p_ptr->timed[TMD_BLIND] || no_lite()) 
			i = i / 10;
		if (p_ptr->timed[TMD_CONFUSED] || p_ptr->timed[TMD_IMAGE]) 
			i = i / 10;

		/* Extract the lock power */
		j = cave_feat[y][x] - FEAT_DOOR_HEAD;

		/* Extract the difficulty XXX XXX XXX */
		j = i - (j * 4);

		/* Always have a small chance of success */
		if (j < 2) 
			j = 2;

		/* Success */
		if (randint0(100) < j)
		{
			/* Message */
			message(MSG_LOCKPICK, 0, L"You have picked the lock.");

			/* Open the door */
			cave_set_feat(y, x, FEAT_OPEN);

			/* Update the visuals */
			p_ptr->update |= (PU_UPDATE_VIEW | PU_MONSTERS);

			/* Experience */
			gain_exp(1);
		}
		else /* Failure */
		{
			/* Failure */
			if (OPT(flush_failure)) flush();

			/* Message */
			message(MSG_LOCKPICK_FAIL, 0, L"You failed to pick the lock.");

			/* We may keep trying */
			more = TRUE;
		}
	}
	else /* Closed door */
	{
		/* Open the door */
		cave_set_feat(y, x, FEAT_OPEN);

		/* Update the visuals */
		p_ptr->update |= (PU_UPDATE_VIEW | PU_MONSTERS);

		/* Sound */
		sound(MSG_OPENDOOR);
	}
	/* Result */
	return (more);
}

/*
 * Open a closed/locked/jammed door or a closed/locked chest.
 *
 * Unlocking a locked door/chest is worth one experience point.
 */
void do_cmd_open(cmd_code code, cmd_arg args[])
{
	int y, x, dir;
	s16b o_idx;

	bool more = FALSE;

	dir = args[0].direction;

	/* Get location */
	y = p_ptr->py + ddy[dir];
	x = p_ptr->px + ddx[dir];

	/* Check for chests */
	o_idx = chest_check(y, x);

	/* Verify legality */
	if (!o_idx && !do_cmd_open_test(y, x)) 
		return;

	/* Take a turn */
	p_ptr->energy_use = 100;

	/* Apply confusion */
	if (confuse_dir(&dir))
	{
		/* Get location */
		y = p_ptr->py + ddy[dir];
		x = p_ptr->px + ddx[dir];

		/* Check for chest */
		o_idx = chest_check(y, x);
	}
	/* Monster */
	if (cave_m_idx[y][x] > 0)
	{
		/* Message */
		msg_print(__T("There is a monster in the way!"));

		/* Attack */
		py_attack(y, x);
	}
	/* Chest */
	else if (o_idx)
	{
		/* Open the chest */
		more = do_cmd_open_chest(y, x, o_idx);
	}
	else /* Door */
	{
		/* Open the door */
		more = do_cmd_open_aux(y, x);
	}
	/* Cancel repeat unless we may continue */
	if (!more) disturb(0, 0);
}

void textui_cmd_open(void) 
{ 
	int y, x, dir = DIR_UNKNOWN; 

	/* Easy Open */ 
	if (OPT(easy_open)) 
	{ 
		int num_doors, num_chests; 

		/* Count closed doors */ 
		num_doors = count_feats(&y, &x, is_closed, FALSE); 

		/* Count chests (locked) */ 
		num_chests = count_chests(&y, &x, FALSE); 

		/* See if only one target */ 
		if ((num_doors + num_chests) == 1) 
		{ 
			dir = coords_to_dir(y, x); 
		} 
	} 
	cmd_insert(CMD_OPEN, dir); 
} 

/*
 * Determine if a given grid may be "closed"
 */
static bool do_cmd_close_test(int y, int x)
{
	/* Must have knowledge */
	if (!(cave_info[y][x] & (CAVE_MARK)))
	{
		/* Message */
		msg_print(__T("You see nothing there."));

		/* Nope */
		return (FALSE);
	}
 	/* Require open/broken door */
	if ((cave_feat[y][x] != FEAT_OPEN) &&
	    (cave_feat[y][x] != FEAT_BROKEN))
	{
		/* Message */
		msg_print(__T("You see nothing there to close."));

		/* Nope */
		return (FALSE);
	}
	/* Okay */
	return (TRUE);
}

/*
 * Perform the basic "close" command
 *
 * Assume there is no monster blocking the destination
 *
 * Returns TRUE if repeated commands may continue
 */
static bool do_cmd_close_aux(int y, int x)
{
	bool more = FALSE;

	/* Verify legality */
	if (!do_cmd_close_test(y, x)) 
		return (FALSE);

	/* Broken door */
	if (cave_feat[y][x] == FEAT_BROKEN)
	{
		/* Message */
		msg_print(__T("The door appears to be broken."));
	}
	else /* Open door */
	{
		/* Close the door */
		cave_set_feat(y, x, FEAT_DOOR_HEAD + 0x00);

		/* Update the visuals */
		p_ptr->update |= (PU_UPDATE_VIEW | PU_MONSTERS);

		/* Sound */
		sound(MSG_SHUTDOOR);
	}
	/* Result */
	return (more);
}

/*
 * Close an open door.
 */
void do_cmd_close(cmd_code code, cmd_arg args[])
{
	int y, x, dir;
	bool more = FALSE;

	dir = args[0].direction;

	/* Get location */
	y = p_ptr->py + ddy[dir];
	x = p_ptr->px + ddx[dir];

	/* Verify legality */
	if (!do_cmd_close_test(y, x)) 
		return;

	/* Take a turn */
	p_ptr->energy_use = 100;

	/* Apply confusion */
	if (confuse_dir(&dir))
	{
		/* Get location */
		y = p_ptr->py + ddy[dir];
		x = p_ptr->px + ddx[dir];
	}
	/* Monster */
	if (cave_m_idx[y][x] > 0)
	{
		/* Message */
		msg_print(__T("There is a monster in the way!"));

		/* Attack */
		py_attack(y, x);
	}
	/* Door */
	else
	{
		/* Close door */
		more = do_cmd_close_aux(y, x);
	}
	/* Cancel repeat unless told not to */
	if (!more) disturb(0, 0);
}

void textui_cmd_close(void) 
{ 
	int y, x, dir = DIR_UNKNOWN; 

	/* Easy Close */ 
	if (OPT(easy_open)) 
	{ 
		/* Count open doors */ 
		if (count_feats(&y, &x, is_open, FALSE) == 1) 
		{ 
			dir = coords_to_dir(y, x); 
		} 
	} 
	else 
	{ 
		if (!get_rep_dir(&dir)) 
			return; 
	}
	cmd_insert(CMD_CLOSE, dir); 
} 

/*
 * Determine if a given grid may be "tunneled"
 */
static bool do_cmd_tunnel_test(int y, int x)
{
	/* Must have knowledge */
	if (!(cave_info[y][x] & (CAVE_MARK)))
	{
		/* Message */
		msg_print(__T("You see nothing there."));

		/* Nope */
		return (FALSE);
	}
	/* Must be a wall/door/etc */
	if (cave_floor_bold(y, x))
	{
		/* Message */
		msg_print(__T("You see nothing there to tunnel."));

		/* Nope */
		return (FALSE);
	}
	/* Okay */
	return (TRUE);
}

/*
 * Tunnel through wall.  Assumes valid location.
 *
 * Note that it is impossible to "extend" rooms past their
 * outer walls (which are actually part of the room).
 *
 * Attempting to do so will produce floor grids which are not part
 * of the room, and whose "illumination" status do not change with
 * the rest of the room.
 */
static bool twall(int y, int x)
{
	/* Paranoia -- Require a wall or door or some such */
	if (cave_floor_bold(y, x)) return (FALSE);

	/* Sound */
	sound(MSG_DIG);

	/* Forget the wall */
	cave_info[y][x] &= ~(CAVE_MARK);

	/* Remove the feature */
	cave_set_feat(y, x, FEAT_FLOOR);

	/* Update the visuals */
	p_ptr->update |= (PU_UPDATE_VIEW | PU_MONSTERS);

	/* Fully update the flow */
	p_ptr->update |= (PU_FORGET_FLOW | PU_UPDATE_FLOW);

	/* Result */
	return (TRUE);
}

/*
 * Perform the basic "tunnel" command
 *
 * Assumes that no monster is blocking the destination
 *
 * Uses "twall" (above) to do all "terrain feature changing".
 *
 * Returns TRUE if repeated commands may continue
 */
static bool do_cmd_tunnel_aux(int y, int x)
{
	bool more = FALSE;

	/* Verify legality */
	if (!do_cmd_tunnel_test(y, x)) 
		return (FALSE);

	/* Sound XXX XXX XXX */
	/* sound(MSG_DIG); */

	/* Titanium */
	if (cave_feat[y][x] >= FEAT_PERM_EXTRA)
	{
		msg_print(__T("This seems to be permanent rock."));
	}
	/* Granite */
	else if (cave_feat[y][x] >= FEAT_WALL_EXTRA)
	{
		/* Tunnel */
		if ((p_ptr->state.skills[SKILL_DIGGING] > 40 + randint0(1600)) && twall(y, x))
		{
			msg_print(__T("You have finished the tunnel."));
		}
		else /* Keep trying */
		{
			/* We may continue tunelling */
			msg_print(__T("You tunnel into the granite wall."));
			more = TRUE;
		}
	}
	/* Quartz / Magma */
	else if (cave_feat[y][x] >= FEAT_MAGMA)
	{
		bool okay = FALSE;
		bool gold = FALSE;
		bool hard = FALSE;

		/* Found gold */
		if (cave_feat[y][x] >= FEAT_MAGMA_H)
		{
			gold = TRUE;
		}
		/* Extract "quartz" flag XXX XXX XXX */
		if ((cave_feat[y][x] - FEAT_MAGMA) & 0x01)
		{
			hard = TRUE;
		}
		/* Quartz */
		if (hard)
		{
			okay = (p_ptr->state.skills[SKILL_DIGGING] > 20 + randint0(800));
		}
		else /* Magma */
		{
			okay = (p_ptr->state.skills[SKILL_DIGGING] > 10 + randint0(400));
		}
		/* Success */
		if (okay && twall(y, x))
		{
			/* Found treasure */
			if (gold)
			{
				/* Place some gold */
				place_gold(y, x, p_ptr->depth);

				/* Message */
				msg_print(__T("You have found something!"));
			}
			else /* Found nothing */
			{
				/* Message */
				msg_print(__T("You have finished the tunnel."));
			}
		}
		/* Failure (quartz) */
		else if (hard)
		{
			/* Message, continue digging */
			msg_print(__T("You tunnel into the quartz vein."));
			more = TRUE;
		}

		/* Failure (magma) */
		else
		{
			/* Message, continue digging */
			msg_print(__T("You tunnel into the magma vein."));
			more = TRUE;
		}
	}
	/* Rubble */
	else if (cave_feat[y][x] == FEAT_RUBBLE)
	{
		/* Remove the rubble */
		if ((p_ptr->state.skills[SKILL_DIGGING] > randint0(200)) && twall(y, x))
		{
			/* Message */
			msg_print(__T("You have removed the rubble."));

			/* Hack -- place an object */
			if (randint0(100) < 10)
			{
				/* Create a simple object */
				place_object(y, x, p_ptr->depth, FALSE, FALSE);

				/* Observe the new object */
				if (!squelch_hide_item(&o_list[cave_o_idx[y][x]]) &&
				    player_can_see_bold(y, x))
				{
					msg_print(__T("You have found something!"));
				}
			}
		}
		else
		{
			/* Message, keep digging */
			msg_print(__T("You dig in the rubble."));
			more = TRUE;
		}
	}
	/* Secret doors */
	else if (cave_feat[y][x] >= FEAT_SECRET)
	{
		/* Tunnel */
		if ((p_ptr->state.skills[SKILL_DIGGING] > 30 + randint0(1200)) && twall(y, x))
		{
			msg_print(__T("You have finished the tunnel."));
		}
		else /* Keep trying */
		{
			/* We may continue tunelling */
			msg_print(__T("You tunnel into the granite wall."));
			more = TRUE;

			/* Occasional Search XXX XXX */
			if (randint0(100) < 25) search();
		}
	}
	else /* Doors */
	{
		/* Tunnel */
		if ((p_ptr->state.skills[SKILL_DIGGING] > 30 + randint0(1200)) && twall(y, x))
		{
			msg_print(__T("You have finished the tunnel."));
		}
		else /* Keep trying */
		{
			/* We may continue tunelling */
			msg_print(__T("You tunnel into the door."));
			more = TRUE;
		}
	}
	/* Result */
	return (more);
}

/*
 * Tunnel through "walls" (including rubble and secret doors)
 *
 * Digging is very difficult without a "digger" weapon, but can be
 * accomplished by strong players using heavy weapons.
 */
void do_cmd_tunnel(cmd_code code, cmd_arg args[])
{
	int y, x, dir;
	bool more = FALSE;

	dir = args[0].direction;

	/* Get location */
	y = p_ptr->py + ddy[dir];
	x = p_ptr->px + ddx[dir];

	/* Oops */
	if (!do_cmd_tunnel_test(y, x)) 
		return;

	/* Take a turn */
	p_ptr->energy_use = 100;

	/* Apply confusion */
	if (confuse_dir(&dir))
	{
		/* Get location */
		y = p_ptr->py + ddy[dir];
		x = p_ptr->px + ddx[dir];
	}
	/* Monster */
	if (cave_m_idx[y][x] > 0)
	{
		/* Message */
		msg_print(__T("There is a monster in the way!"));

		/* Attack */
		py_attack(y, x);
	}
	else /* Walls */
	{
		/* Tunnel through walls */
		more = do_cmd_tunnel_aux(y, x);
	}
	/* Cancel repetition unless we can continue */
	if (!more) disturb(0, 0);
}

void textui_cmd_tunnel(void) 
{ 
	int dir; 
	if (!get_rep_dir(&dir)) 
		return; 

	cmd_insert(CMD_TUNNEL, dir);
}

/*
 * Determine if a given grid may be "disarmed"
 */
static bool do_cmd_disarm_test(int y, int x)
{
	/* Must have knowledge */
	if (!(cave_info[y][x] & (CAVE_MARK)))
	{
		/* Message */
		msg_print(__T("You see nothing there."));

		/* Nope */
		return (FALSE);
	}
	/* Require an actual trap */
	if (!((cave_feat[y][x] >= FEAT_TRAP_HEAD) &&
	      (cave_feat[y][x] <= FEAT_TRAP_TAIL)))
	{
		/* Message */
		msg_print(__T("You see nothing there to disarm."));

		/* Nope */
		return (FALSE);
	}
	/* Okay */
	return (TRUE);
}

/*
 * Perform the basic "disarm" command
 *
 * Assume there is no monster blocking the destination
 *
 * Returns TRUE if repeated commands may continue
 */
static bool do_cmd_disarm_aux(int y, int x)
{
	int i, j, power;

	_TCHAR *name;

	bool more = FALSE;

	/* Verify legality */
	if (!do_cmd_disarm_test(y, x)) return (FALSE);

	/* Get the trap name */
	name = (f_name + f_info[cave_feat[y][x]].name);

	/* Get the "disarm" factor */
	i = p_ptr->state.skills[SKILL_DISARM];

	/* Penalize some conditions */
	if (p_ptr->timed[TMD_BLIND] || no_lite()) i = i / 10;
	if (p_ptr->timed[TMD_CONFUSED] || p_ptr->timed[TMD_IMAGE]) i = i / 10;

	/* XXX XXX XXX Variable power? */

	/* Extract trap "power" */
	power = 5;

	/* Extract the difficulty */
	j = i - power;

	/* Always have a small chance of success */
	if (j < 2) 
		j = 2;

	/* Success */
	if (randint0(100) < j)
	{
		/* Message */
		message_format(MSG_DISARM, 0, L"You have disarmed the %s.", name);

		/* Reward */
		gain_exp(power);

		/* Forget the trap */
		cave_info[y][x] &= ~(CAVE_MARK);

		/* Remove the trap */
		cave_set_feat(y, x, FEAT_FLOOR);
	}
	/* Failure -- Keep trying */
	else if ((i > 5) && (randint1(i) > 5))
	{
		/* Failure */
		if (OPT(flush_failure)) flush();

		/* Message */
		msg_format(__T("You failed to disarm the %s."), name);

		/* We may keep trying */
		more = TRUE;
	}
	else /* Failure -- Set off the trap */
	{
		/* Message */
		msg_format(__T("You set off the %s!"), name);

		/* Hit the trap */
		hit_trap(y, x);
	}
	/* Result */
	return (more);
}

/*
 * Disarms a trap, or a chest
 */
void do_cmd_disarm(cmd_code code, cmd_arg args[])
{
	int y, x, dir;
	s16b o_idx;
	bool more = FALSE;

	dir = args[0].direction; 

	/* Get location */
	y = p_ptr->py + ddy[dir];
	x = p_ptr->px + ddx[dir];

	/* Check for chests */
	o_idx = chest_check(y, x);

	/* Verify legality */
	if (!o_idx && !do_cmd_disarm_test(y, x)) 
		return;

	/* Take a turn */
	p_ptr->energy_use = 100;

	/* Apply confusion */
	if (confuse_dir(&dir))
	{
		/* Get location */
		y = p_ptr->py + ddy[dir];
		x = p_ptr->px + ddx[dir];

		/* Check for chests */
		o_idx = chest_check(y, x);
	}
	/* Monster */
	if (cave_m_idx[y][x] > 0)
	{
		/* Message */
		msg_print(__T("There is a monster in the way!"));

		/* Attack */
		py_attack(y, x);
	}
	/* Chest */
	else if (o_idx)
	{
		/* Disarm the chest */
		more = do_cmd_disarm_chest(y, x, o_idx);
	}
	/* Disarm trap */
	else
	{
		/* Disarm the trap */
		more = do_cmd_disarm_aux(y, x);
	}
	/* Cancel repeat unless told not to */
	if (!more) disturb(0, 0);
}

void textui_cmd_disarm(void) 
{ 
	int y, x, dir; 

	dir = DIR_UNKNOWN; 

	/* Easy Disarm */ 
	if (OPT(easy_open)) 
	{ 
		int num_traps, num_chests; 

		/* Count visible traps */ 
		num_traps = count_feats(&y, &x, is_trap, TRUE); 

		/* Count chests (trapped) */ 
		num_chests = count_chests(&y, &x, TRUE); 

		/* See if only one target */ 
		if (num_traps || num_chests) 
		{ 
			if (num_traps + num_chests <= 1) 
				dir = coords_to_dir(y, x); 
		} 
	}
	else 
	{ 
		if (!get_rep_dir(&dir)) 
			return; 
	}	
	cmd_insert(CMD_DISARM, dir); 
} 

/*
 * Determine if a given grid may be "bashed"
 */
static bool do_cmd_bash_test(int y, int x)
{
	/* Must have knowledge */
	if (!(cave_info[y][x] & (CAVE_MARK)))
	{
		/* Message */
		msg_print(__T("You see nothing there."));

		/* Nope */
		return (FALSE);
	}
	/* Require a door */
	if (!((cave_feat[y][x] >= FEAT_DOOR_HEAD) &&
	      (cave_feat[y][x] <= FEAT_DOOR_TAIL)))
	{
		/* Message */
		msg_print(__T("You see nothing there to bash."));

		/* Nope */
		return (FALSE);
	}
	/* Okay */
	return (TRUE);
}

/*
 * Perform the basic "bash" command
 *
 * Assume there is no monster blocking the destination
 *
 * Returns TRUE if repeated commands may continue
 */
static bool do_cmd_bash_aux(int y, int x)
{
	int bash, temp;
	bool more = FALSE;

	/* Verify legality */
	if (!do_cmd_bash_test(y, x)) 
		return (FALSE);

	/* Message */
	msg_print(__T("You smash into the door!"));

	/* Hack -- Bash power based on strength */
	/* (Ranges from 3 to 20 to 100 to 200) */
	bash = adj_str_blow[p_ptr->state.stat_ind[A_STR]];

	/* Extract door power */
	temp = ((cave_feat[y][x] - FEAT_DOOR_HEAD) & 0x07);

	/* Compare bash power to door power XXX XXX XXX */
	temp = (bash - (temp * 10));

	/* Hack -- always have a chance */
	if (temp < 1) 
		temp = 1;

	/* Hack -- attempt to bash down the door */
	if (randint0(100) < temp)
	{
		/* Break down the door */
		if (randint0(100) < 50)
		{
			cave_set_feat(y, x, FEAT_BROKEN);
		}
		else /* Open the door */
		{
			cave_set_feat(y, x, FEAT_OPEN);
		}
		/* Message */
		message(MSG_OPENDOOR, 0, L"The door crashes open!");

		/* Update the visuals */
		p_ptr->update |= (PU_UPDATE_VIEW | PU_MONSTERS);
	}
	/* Saving throw against stun */
	else if (randint0(100) < adj_dex_safe[p_ptr->state.stat_ind[A_DEX]] +
	         p_ptr->lev)
	{
		/* Message */
		msg_print(__T("The door holds firm."));

		/* Allow repeated bashing */
		more = TRUE;
	}
	else /* High dexterity yields coolness */
	{
		/* Message */
		msg_print(__T("You are off-balance."));

		/* Hack -- Lose balance ala paralysis */
		(void)inc_timed(TMD_PARALYZED, 2 + randint0(2), TRUE);
	}

	/* Result */
	return (more);
}

/*
 * Bash open a door, success based on character strength
 *
 * For a closed door, pval is positive if locked; negative if stuck.
 *
 * For an open door, pval is positive for a broken door.
 *
 * A closed door can be opened - harder if locked. Any door might be
 * bashed open (and thereby broken). Bashing a door is (potentially)
 * faster! You move into the door way. To open a stuck door, it must
 * be bashed. A closed door can be jammed (see do_cmd_spike()).
 *
 * Creatures can also open or bash doors, see elsewhere.
 */
void do_cmd_bash(cmd_code code, cmd_arg args[])
{
	int y, x, dir;

	dir = args[0].direction;

	/* Get location */
	y = p_ptr->py + ddy[dir];
	x = p_ptr->px + ddx[dir];

	/* Verify legality */
	if (!do_cmd_bash_test(y, x)) 
		return;

	/* Take a turn */
	p_ptr->energy_use = 100;

	/* Apply confusion */
	if (confuse_dir(&dir))
	{
		/* Get location */
		y = p_ptr->py + ddy[dir];
		x = p_ptr->px + ddx[dir];
	}
	/* Monster */
	if (cave_m_idx[y][x] > 0)
	{
		/* Message */
		msg_print(__T("There is a monster in the way!"));

		/* Attack */
		py_attack(y, x);
	}
	else /* Door */
	{
		/* Bash the door */
		if (!do_cmd_bash_aux(y, x))
		{
			/* Cancel repeat */
			disturb(0, 0);
		}
	}
}

void textui_cmd_bash(void) 
{ 
	int dir; 
	if (!get_rep_dir(&dir)) 
        return; 

	cmd_insert(CMD_BASH, dir); 
} 

/*
 * Manipulate an adjacent grid in some way
 *
 * Attack monsters, tunnel through walls, disarm traps, open doors.
 *
 * This command must always take energy, to prevent free detection
 * of invisible monsters.
 *
 * The "semantics" of this command must be chosen before the player
 * is confused, and it must be verified against the new grid.
 */
void do_cmd_alter_aux(int dir)
{
	int y, x;

	int feat;

	bool more = FALSE;

	/* Get location */
	y = p_ptr->py + ddy[dir];
	x = p_ptr->px + ddx[dir];

	/* Original feature */
	feat = cave_feat[y][x];

	/* Must have knowledge to know feature XXX XXX */
	if (!(cave_info[y][x] & (CAVE_MARK))) 
		feat = FEAT_NONE;

	/* Take a turn */
	p_ptr->energy_use = 100;

	/* Apply confusion */
	if (confuse_dir(&dir))
	{
		/* Get location */
		y = p_ptr->py + ddy[dir];
		x = p_ptr->px + ddx[dir];
	}
	/* Attack monsters */
	if (cave_m_idx[y][x] > 0)
	{
		/* Attack */
		py_attack(y, x);
	}
	/* Tunnel through walls */
	else if (feat >= FEAT_SECRET)
	{
		/* Tunnel */
		more = do_cmd_tunnel_aux(y, x);
	}
#if 0
	/* Bash jammed doors */
	else if (feat >= FEAT_DOOR_HEAD + 0x08)
	{
		/* Tunnel */
		more = do_cmd_bash_aux(y, x);
	}
#endif /* 0 */
	/* Open closed doors */
	else if (feat >= FEAT_DOOR_HEAD)
	{
		/* Tunnel */
		more = do_cmd_open_aux(y, x);
	}
	/* Disarm traps */
	else if (feat >= FEAT_TRAP_HEAD)
	{
		/* Tunnel */
		more = do_cmd_disarm_aux(y, x);
	}
#if 0
	/* Close open doors */
	else if (feat == FEAT_OPEN)
	{
		/* Close */
		more = do_cmd_close_aux(y, x);
	}
#endif
	else /* Oops */
	{
		/* Oops */
		msg_print(__T("You spin around."));
	}
	/* Cancel repetition unless we can continue */
	if (!more) 
		disturb(0, 0);
}

void do_cmd_alter(cmd_code code, cmd_arg args[]) 
{ 
	do_cmd_alter_aux(args[0].direction); 
} 

void textui_cmd_alter(void) 
{ 
	int dir; 

	if (!get_rep_dir(&dir)) 
        return; 

	cmd_insert(CMD_ALTER, dir); 
} 

/*
 * Find the index of some "spikes", if possible.
 *
 * XXX XXX XXX Let user choose a pile of spikes, perhaps?
 */
static bool get_spike(int *ip)
{
	int i;

	/* Check every item in the pack */
	for (i = 0; i < INVEN_PACK; i++)
	{
		object_type *o_ptr = &inventory[i];

		/* Skip non-objects */
		if (!o_ptr->k_idx) 
			continue;

		/* Check the "tval" code */
		if (o_ptr->tval == TV_SPIKE)
		{
			/* Save the spike index */
			(*ip) = i;

			/* Success */
			return (TRUE);
		}
	}
	/* Oops */
	return (FALSE);
}

/*
 * Determine if a given grid may be "spiked"
 */
static bool do_cmd_spike_test(int y, int x)
{
	/* Must have knowledge */
	if (!(cave_info[y][x] & (CAVE_MARK)))
	{
		/* Message */
		msg_print(__T("You see nothing there."));

		/* Nope */
		return (FALSE);
	}
	/* Require a door */
	if (!((cave_feat[y][x] >= FEAT_DOOR_HEAD) &&
	      (cave_feat[y][x] <= FEAT_DOOR_TAIL)))
	{
		/* Message */
		msg_print(__T("You see nothing there to spike."));

		/* Nope */
		return (FALSE);
	}
	/* Okay */
	return (TRUE);
}

/*
 * Jam a closed door with a spike
 *
 * This command may NOT be repeated
 */
void do_cmd_spike(cmd_code code, cmd_arg args[])
{
	int y, x, dir, item = 0;

	dir = args[0].direction;

	/* Get a spike */
	if (!get_spike(&item))
	{
		/* Message */
		msg_print(__T("You have no spikes!"));

		/* Done */
		return;
	}
	/* Get location */
	y = p_ptr->py + ddy[dir];
	x = p_ptr->px + ddx[dir];

	/* Verify legality */
	if (!do_cmd_spike_test(y, x)) return;

	/* Take a turn */
	p_ptr->energy_use = 100;

	/* Confuse direction */
	if (confuse_dir(&dir))
	{
		/* Get location */
		y = p_ptr->py + ddy[dir];
		x = p_ptr->px + ddx[dir];
	}
	/* Monster */
	if (cave_m_idx[y][x] > 0)
	{
		/* Message */
		msg_print(__T("There is a monster in the way!"));

		/* Attack */
		py_attack(y, x);
	}
	else /* Go for it */
	{
		/* Verify legality */
		if (!do_cmd_spike_test(y, x)) return;

		/* Successful jamming */
		msg_print(__T("You jam the door with a spike."));

		/* Convert "locked" to "stuck" XXX XXX XXX */
		if (cave_feat[y][x] < FEAT_DOOR_HEAD + 0x08)
		{
			cave_feat[y][x] += 0x08;
		}
		/* Add one spike to the door */
		if (cave_feat[y][x] < FEAT_DOOR_TAIL)
		{
			cave_feat[y][x] += 0x01;
		}
		/* Use up, and describe, a single spike, from the bottom */
		inven_item_increase(item, -1);
		inven_item_describe(item);
		inven_item_optimize(item);
	}
}

void textui_cmd_spike(void) 
{ 
	int dir; 
 	if (!get_rep_dir(&dir)) 
 		return; 

	cmd_insert(CMD_JAM, dir); 
} 

/*
 * Determine if a given grid may be "walked"
 */
static bool do_cmd_walk_test(int y, int x)
{
	/* Allow attack on visible monsters if unafraid */
	if ((cave_m_idx[y][x] > 0) && (mon_list[cave_m_idx[y][x]].ml))
	{
		/* Handle player fear */
		if(p_ptr->state.afraid)
		{
			/* Extract monster name (or "it") */
			_TCHAR m_name[80];
			monster_type *m_ptr;

			m_ptr = &mon_list[cave_m_idx[y][x]];
			monster_desc(m_name, _countof(m_name), m_ptr, 0);

			/* Message */
			message_format(MSG_AFRAID, 0,
				L"You are too afraid to attack %s!", m_name);

			/* Nope */
			return (FALSE);
		}
		return (TRUE);
	}
	/* Hack -- walking obtains knowledge XXX XXX */
	if (!(cave_info[y][x] & (CAVE_MARK))) 
		return (TRUE);

	/* Require open space */
	if (!cave_floor_bold(y, x))
	{
		/* Rubble */
		if (cave_feat[y][x] == FEAT_RUBBLE)
		{
			/* Message */
			message(MSG_HITWALL, 0, L"There is a pile of rubble in the way!");
		}
		/* Door */
		else if (cave_feat[y][x] < FEAT_SECRET)
		{
			/* Hack -- Handle "OPT(easy_alter)" */
			if (OPT(easy_alter)) return (TRUE);

			/* Message */
			message(MSG_HITWALL, 0, L"There is a door in the way!");
		}
		else /* Wall */
		{
			/* Message */
			message(MSG_HITWALL, 0, L"There is a wall in the way!");
		}
		/* Nope */
		return (FALSE);
	}
	/* Okay */
	return (TRUE);
}

/*
 * Walk in the given direction. 
 */ 
void do_cmd_walk(cmd_code code, cmd_arg args[])
{
	int y, x, dir;

	dir = args[0].direction;

	/* Get location */
	y = p_ptr->py + ddy[dir];
	x = p_ptr->px + ddx[dir];

	/* Verify legality */
	if (!do_cmd_walk_test(y, x)) 
		return;

	/* Take a turn */
	p_ptr->energy_use = 100;

	/* Confuse direction */
	if (confuse_dir(&dir))
	{
		/* Get location */
		y = p_ptr->py + ddy[dir];
		x = p_ptr->px + ddx[dir];
	}
	/* Verify legality */
	if (!do_cmd_walk_test(y, x)) 
		return;

	/* Move the player */
	move_player(dir);
}

/*
 * Tell the game we want to walk - in future we might want to supply 
 * directions here rather than rely on keymap/macro things. 
 */ 
void textui_cmd_walk(void) 
{ 
	int dir; 
 	if (!get_rep_dir(&dir)) 
 		return; 
 	
 	cmd_insert(CMD_WALK, dir);
}

/*
 * Jump into a trap, turn off pickup (does not work).
 *
 * What a horrible concept.
 */
void do_cmd_jump(cmd_code code, cmd_arg args[])
{
	bool old_easy_alter;

	/* OPT(easy_alter) can be turned off (don't disarm traps) */
	old_easy_alter = OPT(easy_alter);
	OPT(easy_alter) = FALSE;

	do_cmd_walk(code, args);

	/* Restore OPT(easy_alter) */
	OPT(easy_alter) = old_easy_alter;
}

void textui_cmd_jump(void) 
{ 
	int dir; 
	if (!get_rep_dir(&dir)) 
        return; 

	cmd_insert(CMD_JUMP, dir);
} 

/*
 * Start running.
 *
 * Note that running while confused is not allowed.
 */
void do_cmd_run(cmd_code code, cmd_arg args[])
{
	int y, x, dir;

	dir = args[0].direction;

	/* Hack XXX XXX XXX */
	if (p_ptr->timed[TMD_CONFUSED])
	{
		msg_print(__T("You are too confused!"));
		return;
	}

	/* Get location */
	y = p_ptr->py + ddy[dir];
	x = p_ptr->px + ddx[dir];

	/* Verify legality */
	if (!do_cmd_walk_test(y, x)) return;

	/* Start run */
	run_step(dir);
}

void textui_cmd_run(void) 
{ 
	int dir; 
	if (!get_rep_dir(&dir)) 
        return; 

	cmd_insert(CMD_RUN, dir);
} 

/*
 * Start running with pathfinder.
 *
 * Note that running while confused is not allowed.
 */
void do_cmd_pathfind(cmd_code code, cmd_arg args[])
{
	/* Hack XXX XXX XXX */
	if (p_ptr->timed[TMD_CONFUSED])
	{
		msg_print(__T("You are too confused!"));
		return;
	}

	if (findpath(args[0].point.y, args[0].point.x))
	{
		p_ptr->running = 1000;
		/* Calculate torch radius */
		p_ptr->update |= (PU_TORCH);
		p_ptr->running_withpathfind = TRUE;
		run_step(0);
	}
}

/*
 * Stay still.  Search.  Enter stores.
 * Pick up treasure if "pickup" is true.
 */
void do_cmd_hold(cmd_code code, cmd_arg args[])
{
	/* Take a turn */
	p_ptr->energy_use = 100;

	/* Spontaneous Searching */
	if ((p_ptr->state.skills[SKILL_SEARCH_FREQUENCY] >= 50) ||
	    one_in_(50 - p_ptr->state.skills[SKILL_SEARCH_FREQUENCY]))
	{
		search();
	}

	/* Continuous Searching */
	if (p_ptr->searching)
	{
		search();
	}

	/* Handle objects now.  XXX XXX XXX */
	p_ptr->energy_use += py_pickup(0) * 10;

	/* Hack -- enter a store if we are on one */
	if ((cave_feat[p_ptr->py][p_ptr->px] >= FEAT_SHOP_HEAD) &&
	    (cave_feat[p_ptr->py][p_ptr->px] <= FEAT_SHOP_TAIL))
	{
		/* Disturb */
		disturb(0, 0);

		cmd_insert(CMD_ENTER_STORE);

		/* Free turn XXX XXX XXX */
		p_ptr->energy_use = 0;
	}
}

/*
 * Pick up objects on the floor beneath you.  -LM-
 */
void do_cmd_pickup(cmd_code code, cmd_arg args[])
{
	int energy_cost;

	/* Pick up floor objects, forcing a menu for multiple objects. */
	energy_cost = py_pickup(1) * 10;

	/* Maximum time expenditure is a full turn. */
	if (energy_cost > 100) energy_cost = 100;

	/* Charge this amount of energy. */
	p_ptr->energy_use = energy_cost;
}

/*
 * Rest (restores hit points and mana and such)
 */
void do_cmd_rest(cmd_code code, cmd_arg args[]) 
{ 
	/* Save the rest code */ 
	switch (args[0].choice) 
	{ 
		case REST_ALL: 
		{ 
			p_ptr->resting = -2; 
			break; 
		} 
		case REST_ALL_POINTS: 
		{ 
			p_ptr->resting = -1; 
			break; 
		} 
		case REST_SOME_POINTS: 
		{ 
			p_ptr->resting = -3; 
			break; 
		} 
		default: 
		{ 
			p_ptr->resting = p_ptr->command_arg; 
		} 
	} 

	/* Take a turn XXX XXX XXX (?) */ 
	p_ptr->energy_use = 100; 

	/* Cancel the arg */ 
	p_ptr->command_arg = 0; 

	/* Cancel searching */ 
	p_ptr->searching = FALSE; 
	
	/* Recalculate bonuses */ 
	p_ptr->update |= (PU_BONUS); 

	/* Redraw the state */ 
	p_ptr->redraw |= (PR_STATE); 

	/* Handle stuff */ 
	handle_stuff(); 

	/* Refresh XXX XXX XXX */ 
	Term_fresh(); 
} 

void textui_cmd_rest(void) 
{ 
	/* Prompt for time if needed */ 

	if (p_ptr->command_arg <= 0)
	{
		const _TCHAR *p = __T("Rest (0-9999, '!' for HP or SP, '*' for HP and SP, '&' as needed): ");

		_TCHAR out_val[5] = __T("& ");

		/* Ask for duration */
		if (!get_string(p, out_val, _countof(out_val))) return;

		/* Rest until done */
		if (out_val[0] == '&')
		{
			cmd_insert(CMD_REST, REST_ALL);
		}
		/* Rest a lot */
		else if (out_val[0] == '*')
		{
			cmd_insert(CMD_REST, REST_ALL_POINTS);
		}
		/* Rest until HP or SP filled */
		else if (out_val[0] == '!')
		{
			cmd_insert(CMD_REST, REST_SOME_POINTS);
		}
		else /* Rest some */
		{
			p_ptr->command_arg = _tstoi(out_val);
			if (p_ptr->command_arg <= 0) return;
			if (p_ptr->command_arg > 9999) p_ptr->command_arg = 9999;
			cmd_insert(CMD_REST, REST_TURNS);
		}
	}
}

/*
 * Hack -- commit suicide
 */
void do_cmd_suicide(cmd_code code, cmd_arg args[])
{
	/* Commit suicide */
	p_ptr->is_dead = TRUE;

	/* Stop playing */
	p_ptr->playing = FALSE;

	/* Leaving */
	p_ptr->leaving = TRUE;

	/* Cause of death */
	_tcscpy_s(p_ptr->died_from, _countof(p_ptr->died_from), __T("Quitting"));
}

void textui_cmd_suicide(void)
{
	/* Flush input */
	flush();

	/* Verify Retirement */
	if (p_ptr->total_winner)
	{
		/* Verify */
		if (!get_check(__T("Do you want to retire? "))) return;
	}
	else /* Verify Suicide */
	{
		_TCHAR ch;

		/* Verify */
		if (!get_check(__T("Do you really want to commit suicide? "))) return;

		/* Special Verification for suicide */
		prt(__T("Please verify SUICIDE by typing the '@' sign: "), 0, 0);
		flush();
		ch = inkey();
		prt(__T(""), 0, 0);
		if (ch != '@') return;
	}

	cmd_insert(CMD_SUICIDE);
}

void do_cmd_save_game(cmd_code code, cmd_arg args[])
{
	save_game();
}
