﻿/* テスト */
/*
 * File: monster1.c
 * Purpose: Monster description code.
 *
 * Copyright (c) 1997-2007 Ben Harrison, James E. Wilson, Robert A. Koeneke
 *
 * This work is free software; you can redistribute it and/or modify it
 * under the terms of either:
 *
 * a) the GNU General Public License as published by the Free Software
 *    Foundation, version 2, or
 *
 * b) the "Angband licence":
 *    This software may be copied and distributed for educational, research,
 *    and not for profit purposes provided that this copyright and statement
 *    are included in all such copies.  Other copyrights may also apply.
 */
extern "C"
{
#include "angband.h"
#include "object/tvalsval.h"
}
extern void message_flush(void);
extern errr Term_erase(int x, int y, int n);
extern size_t strnfmt(_TCHAR *buf, size_t max, const _TCHAR *fmt, ...);

/*
 * Pronoun arrays, by gender.
 */
static const _TCHAR *wd_he[3] = { L"it", L"he", L"she" };
static const _TCHAR *wd_his[3] = { L"its", L"his", L"her" };

/*
 * Pluralizer.  Args(count, singular, plural)
 */
#define plural(c, s, p)    (((c) == 1) ? (s) : (p))


static void output_list(const _TCHAR *list[], int num, byte attr)
{
	int i;
	const _TCHAR *conjunction = L"and ";

	if (num < 0)
	{
		num = -num;
		conjunction = L"or ";
	}

	for (i = 0; i < num; i++)
	{
        if (i)
		{
			if (num > 2)
				text_out(L", ");
			else
				text_out(L" ");

			if (i == num - 1)
				text_out(conjunction);
		}

		text_out_c(attr, list[i]);
	}
}


static void output_desc_list(int msex, const _TCHAR *intro, const _TCHAR *list[], int n, byte attr)
{
	if (n != 0)
	{
		/* Output intro */
		text_out(format(__T("%^s %s "), wd_he[msex], intro));

		/* Output list */
		output_list(list, n, attr);

		/* Output end */
		text_out(L".  ");
	}
}


/*
 * Determine if the "armor" is known
 * The higher the level, the fewer kills needed.
 */
static bool know_armour(int r_idx, const monster_lore *l_ptr)
{
	const monster_race *r_ptr = &r_info[r_idx];

	s32b level = r_ptr->level;

	s32b kills = l_ptr->tkills;

	/* Normal monsters */
	if (kills > 304 / (4 + level)) return (TRUE);

	/* Skip non-uniques */
	if (!(r_ptr->flags[0] & RF0_UNIQUE)) return (FALSE);

	/* Unique monsters */
	if (kills > 304 / (38 + (5 * level) / 4)) return (TRUE);

	/* Assume false */
	return (FALSE);
}


/*
 * Determine if the "damage" of the given attack is known
 * the higher the level of the monster, the fewer the attacks you need,
 * the more damage an attack does, the more attacks you need
 */
static bool know_damage(int r_idx, const monster_lore *l_ptr, int i)
{
	const monster_race *r_ptr = &r_info[r_idx];

	s32b level = r_ptr->level;

	s32b a = l_ptr->blows[i];

	s32b d1 = r_ptr->blow[i].d_dice;
	s32b d2 = r_ptr->blow[i].d_side;

	s32b d = d1 * d2;

	/* Normal monsters */
	if ((4 + level) * a >= 80 * d) return (TRUE);

	/* Skip non-uniques */
	if (!(r_ptr->flags[0] & RF0_UNIQUE)) return (FALSE);

	/* Unique monsters */
	if ((4 + level) * (2 * a) > 80 * d) return (TRUE);

	/* Assume false */
	return (FALSE);
}

/*
 * Dump flavour text
 */
static void describe_monster_desc(int r_idx)
{
	const monster_race *r_ptr = &r_info[r_idx];
	text_out(L"%s\n", r_text + r_ptr->text);
}


static void describe_monster_spells(int r_idx, const monster_lore *l_ptr)
{
	const monster_race *r_ptr = &r_info[r_idx];
	int m, n;
	int msex = 0;
	bool breath = FALSE;
	bool magic = FALSE;
	int vn;
	const _TCHAR *vp[64];


	/* Extract a gender (if applicable) */
	if (r_ptr->flags[0] & RF0_FEMALE) msex = 2;
	else if (r_ptr->flags[0] & RF0_MALE) msex = 1;

	/* Collect innate attacks */
	vn = 0;
	if (l_ptr->spell_flags[0] & RSF0_SHRIEK)  vp[vn++] = L"shriek for help";
	if (l_ptr->spell_flags[0] & RSF0_XXX2)    vp[vn++] = L"do something";
	if (l_ptr->spell_flags[0] & RSF0_XXX3)    vp[vn++] = L"do something";
	if (l_ptr->spell_flags[0] & RSF0_XXX4)    vp[vn++] = L"do something";
	if (l_ptr->spell_flags[0] & RSF0_ARROW_1) vp[vn++] = L"fire an arrow";
	if (l_ptr->spell_flags[0] & RSF0_ARROW_2) vp[vn++] = L"fire arrows";
	if (l_ptr->spell_flags[0] & RSF0_ARROW_3) vp[vn++] = L"fire a missile";
	if (l_ptr->spell_flags[0] & RSF0_ARROW_4) vp[vn++] = L"fire missiles";
	if (l_ptr->spell_flags[0] & RSF0_BOULDER) vp[vn++] = L"throw boulders";

	/* Describe innate attacks */
	output_desc_list(msex, L"may", vp, -vn, TERM_WHITE);


	/* Collect breaths */
	vn = 0;
	if (l_ptr->spell_flags[0] & RSF0_BR_ACID) vp[vn++] = L"acid";
	if (l_ptr->spell_flags[0] & RSF0_BR_ELEC) vp[vn++] = L"lightning";
	if (l_ptr->spell_flags[0] & RSF0_BR_FIRE) vp[vn++] = L"fire";
	if (l_ptr->spell_flags[0] & RSF0_BR_COLD) vp[vn++] = L"frost";
	if (l_ptr->spell_flags[0] & RSF0_BR_POIS) vp[vn++] = L"poison";
	if (l_ptr->spell_flags[0] & RSF0_BR_NETH) vp[vn++] = L"nether";
	if (l_ptr->spell_flags[0] & RSF0_BR_LITE) vp[vn++] = L"light";
	if (l_ptr->spell_flags[0] & RSF0_BR_DARK) vp[vn++] = L"darkness";
	if (l_ptr->spell_flags[0] & RSF0_BR_CONF) vp[vn++] = L"confusion";
	if (l_ptr->spell_flags[0] & RSF0_BR_SOUN) vp[vn++] = L"sound";
	if (l_ptr->spell_flags[0] & RSF0_BR_CHAO) vp[vn++] = L"chaos";
	if (l_ptr->spell_flags[0] & RSF0_BR_DISE) vp[vn++] = L"disenchantment";
	if (l_ptr->spell_flags[0] & RSF0_BR_NEXU) vp[vn++] = L"nexus";
	if (l_ptr->spell_flags[0] & RSF0_BR_TIME) vp[vn++] = L"time";
	if (l_ptr->spell_flags[0] & RSF0_BR_INER) vp[vn++] = L"inertia";
	if (l_ptr->spell_flags[0] & RSF0_BR_GRAV) vp[vn++] = L"gravity";
	if (l_ptr->spell_flags[0] & RSF0_BR_SHAR) vp[vn++] = L"shards";
	if (l_ptr->spell_flags[0] & RSF0_BR_PLAS) vp[vn++] = L"plasma";
	if (l_ptr->spell_flags[0] & RSF0_BR_WALL) vp[vn++] = L"force";
	if (l_ptr->spell_flags[0] & RSF0_BR_MANA) vp[vn++] = L"mana";
	if (l_ptr->spell_flags[0] & RSF0_XXX5)    vp[vn++] = L"something";
	if (l_ptr->spell_flags[0] & RSF0_XXX6)    vp[vn++] = L"something";
	if (l_ptr->spell_flags[0] & RSF0_XXX7)    vp[vn++] = L"something";

	/* Describe breaths */
	if (vn)
	{
		/* Note breath */
		breath = TRUE;

		/* Display */
		text_out(L"%^s may ", wd_he[msex]);
		text_out_c(TERM_L_RED, L"breathe ");
		output_list(vp, -vn, TERM_WHITE);
	}


	/* Collect spells */
	vn = 0;
	if (l_ptr->spell_flags[1] & RSF1_BA_ACID)     vp[vn++] = L"produce acid balls";
	if (l_ptr->spell_flags[1] & RSF1_BA_ELEC)     vp[vn++] = L"produce lightning balls";
	if (l_ptr->spell_flags[1] & RSF1_BA_FIRE)     vp[vn++] = L"produce fire balls";
	if (l_ptr->spell_flags[1] & RSF1_BA_COLD)     vp[vn++] = L"produce frost balls";
	if (l_ptr->spell_flags[1] & RSF1_BA_POIS)     vp[vn++] = L"produce poison balls";
	if (l_ptr->spell_flags[1] & RSF1_BA_NETH)     vp[vn++] = L"produce nether balls";
	if (l_ptr->spell_flags[1] & RSF1_BA_WATE)     vp[vn++] = L"produce water balls";
	if (l_ptr->spell_flags[1] & RSF1_BA_MANA)     vp[vn++] = L"invoke mana storms";
	if (l_ptr->spell_flags[1] & RSF1_BA_DARK)     vp[vn++] = L"invoke darkness storms";
	if (l_ptr->spell_flags[1] & RSF1_DRAIN_MANA)  vp[vn++] = L"drain mana";
	if (l_ptr->spell_flags[1] & RSF1_MIND_BLAST)  vp[vn++] = L"cause mind blasting";
	if (l_ptr->spell_flags[1] & RSF1_BRAIN_SMASH) vp[vn++] = L"cause brain smashing";
	if (l_ptr->spell_flags[1] & RSF1_CAUSE_1)     vp[vn++] = L"cause light wounds";
	if (l_ptr->spell_flags[1] & RSF1_CAUSE_2)     vp[vn++] = L"cause serious wounds";
	if (l_ptr->spell_flags[1] & RSF1_CAUSE_3)     vp[vn++] = L"cause critical wounds";
	if (l_ptr->spell_flags[1] & RSF1_CAUSE_4)     vp[vn++] = L"cause mortal wounds";
	if (l_ptr->spell_flags[1] & RSF1_BO_ACID)     vp[vn++] = L"produce acid bolts";
	if (l_ptr->spell_flags[1] & RSF1_BO_ELEC)     vp[vn++] = L"produce lightning bolts";
	if (l_ptr->spell_flags[1] & RSF1_BO_FIRE)     vp[vn++] = L"produce fire bolts";
	if (l_ptr->spell_flags[1] & RSF1_BO_COLD)     vp[vn++] = L"produce frost bolts";
	if (l_ptr->spell_flags[1] & RSF1_BO_POIS)     vp[vn++] = L"produce poison bolts";
	if (l_ptr->spell_flags[1] & RSF1_BO_NETH)     vp[vn++] = L"produce nether bolts";
	if (l_ptr->spell_flags[1] & RSF1_BO_WATE)     vp[vn++] = L"produce water bolts";
	if (l_ptr->spell_flags[1] & RSF1_BO_MANA)     vp[vn++] = L"produce mana bolts";
	if (l_ptr->spell_flags[1] & RSF1_BO_PLAS)     vp[vn++] = L"produce plasma bolts";
	if (l_ptr->spell_flags[1] & RSF1_BO_ICEE)     vp[vn++] = L"produce ice bolts";
	if (l_ptr->spell_flags[1] & RSF1_MISSILE)     vp[vn++] = L"produce magic missiles";
	if (l_ptr->spell_flags[1] & RSF1_SCARE)       vp[vn++] = L"terrify";
	if (l_ptr->spell_flags[1] & RSF1_BLIND)       vp[vn++] = L"blind";
	if (l_ptr->spell_flags[1] & RSF1_CONF)        vp[vn++] = L"confuse";
	if (l_ptr->spell_flags[1] & RSF1_SLOW)        vp[vn++] = L"slow";
	if (l_ptr->spell_flags[1] & RSF1_HOLD)        vp[vn++] = L"paralyze";
	if (l_ptr->spell_flags[2] & RSF2_HASTE)       vp[vn++] = L"haste-self";
	if (l_ptr->spell_flags[2] & RSF2_XXX1)        vp[vn++] = L"do something";
	if (l_ptr->spell_flags[2] & RSF2_HEAL)        vp[vn++] = L"heal-self";
	if (l_ptr->spell_flags[2] & RSF2_XXX2)        vp[vn++] = L"do something";
	if (l_ptr->spell_flags[2] & RSF2_BLINK)       vp[vn++] = L"blink-self";
	if (l_ptr->spell_flags[2] & RSF2_TPORT)       vp[vn++] = L"teleport-self";
	if (l_ptr->spell_flags[2] & RSF2_XXX3)        vp[vn++] = L"do something";
	if (l_ptr->spell_flags[2] & RSF2_XXX4)        vp[vn++] = L"do something";
	if (l_ptr->spell_flags[2] & RSF2_TELE_TO)     vp[vn++] = L"teleport to";
	if (l_ptr->spell_flags[2] & RSF2_TELE_AWAY)   vp[vn++] = L"teleport away";
	if (l_ptr->spell_flags[2] & RSF2_TELE_LEVEL)  vp[vn++] = L"teleport level";
	if (l_ptr->spell_flags[2] & RSF2_XXX5)        vp[vn++] = L"do something";
	if (l_ptr->spell_flags[2] & RSF2_DARKNESS)    vp[vn++] = L"create darkness";
	if (l_ptr->spell_flags[2] & RSF2_TRAPS)       vp[vn++] = L"create traps";
	if (l_ptr->spell_flags[2] & RSF2_FORGET)      vp[vn++] = L"cause amnesia";
	if (l_ptr->spell_flags[2] & RSF2_XXX6)        vp[vn++] = L"do something";
	if (l_ptr->spell_flags[2] & RSF2_S_KIN)       vp[vn++] = L"summon similar monsters";
	if (l_ptr->spell_flags[2] & RSF2_S_MONSTER)   vp[vn++] = L"summon a monster";
	if (l_ptr->spell_flags[2] & RSF2_S_MONSTERS)  vp[vn++] = L"summon monsters";
	if (l_ptr->spell_flags[2] & RSF2_S_ANIMAL)    vp[vn++] = L"summon animals";
	if (l_ptr->spell_flags[2] & RSF2_S_SPIDER)    vp[vn++] = L"summon spiders";
	if (l_ptr->spell_flags[2] & RSF2_S_HOUND)     vp[vn++] = L"summon hounds";
	if (l_ptr->spell_flags[2] & RSF2_S_HYDRA)     vp[vn++] = L"summon hydras";
	if (l_ptr->spell_flags[2] & RSF2_S_ANGEL)     vp[vn++] = L"summon an angel";
	if (l_ptr->spell_flags[2] & RSF2_S_DEMON)     vp[vn++] = L"summon a demon";
	if (l_ptr->spell_flags[2] & RSF2_S_UNDEAD)    vp[vn++] = L"summon an undead";
	if (l_ptr->spell_flags[2] & RSF2_S_DRAGON)    vp[vn++] = L"summon a dragon";
	if (l_ptr->spell_flags[2] & RSF2_S_HI_UNDEAD) vp[vn++] = L"summon greater undead";
	if (l_ptr->spell_flags[2] & RSF2_S_HI_DRAGON) vp[vn++] = L"summon ancient dragons";
	if (l_ptr->spell_flags[2] & RSF2_S_HI_DEMON)  vp[vn++] = L"summon greater demons";
	if (l_ptr->spell_flags[2] & RSF2_S_WRAITH)    vp[vn++] = L"summon ringwraiths";
	if (l_ptr->spell_flags[2] & RSF2_S_UNIQUE)    vp[vn++] = L"summon uniques";

	/* Describe spells */
	if (vn)
	{
		/* Note magic */
		magic = TRUE;

		/* Intro */
		if (breath)
			text_out(L", and is also ");
		else
			text_out(L"%^s is ", wd_he[msex]);

		/* Verb Phrase */
		text_out_c(TERM_L_RED, L"magical");
		text_out(L", casting spells");

		/* Adverb */
		if (l_ptr->flags[1] & RF1_SMART) text_out(L" intelligently");

		/* List */
		text_out(L" which ");
		output_list(vp, -vn, TERM_WHITE);
	}


	/* End the sentence about innate/other spells */
	if (breath || magic)
	{
		/* Total casting */
		m = l_ptr->cast_innate + l_ptr->cast_spell;

		/* Average frequency */
		n = (r_ptr->freq_innate + r_ptr->freq_spell) / 2;

		/* Describe the spell frequency */
		if (m > 100)
		{
			text_out(L"; ");
			text_out_c(TERM_L_GREEN, L"1");
			text_out(L" time in ");
			text_out_c(TERM_L_GREEN, L"%d", 100 / n);
		}

		/* Guess at the frequency */
		else if (m)
		{
			n = ((n + 9) / 10) * 10;
			text_out(L"; about ");
			text_out_c(TERM_L_GREEN, L"1");
			text_out(L" time in ");
			text_out_c(TERM_L_GREEN, L"%d", 100 / n);
		}

		/* End this sentence */
		text_out(L".  ");
	}
}

/*
 * Describe a monster's drop.
 */
static void describe_monster_drop(int r_idx, const monster_lore *l_ptr)
{
	const monster_race *r_ptr = &r_info[r_idx];

	int n;
	int msex = 0;


	/* Extract a gender (if applicable) */
	if (r_ptr->flags[0] & RF0_FEMALE) msex = 2;
	else if (r_ptr->flags[0] & RF0_MALE) msex = 1;

	/* Drops gold and/or items */
	if (l_ptr->drop_gold || l_ptr->drop_item)
	{
		/* Intro */
		text_out(L"%^s may carry", wd_he[msex]);

		/* Count maximum drop */
		n = MAX(l_ptr->drop_gold, l_ptr->drop_item);

		/* Count drops */
		if (n == 1) text_out(L" a single ");
		else if (n == 2) text_out(L" one or two ");
		else text_out(format(__T(" up to %d "), n));


		/* Quality */
		if (l_ptr->flags[0] & RF0_DROP_GREAT) text_out(L"exceptional ");
		else if (l_ptr->flags[0] & RF0_DROP_GOOD) text_out(L"good ");


		/* Objects */
		if (l_ptr->drop_item)
		{
			/* Dump "object(s)" */
			text_out(L"object%s", PLURAL(n));

			/* Add conjunction if also dropping gold */
			if (l_ptr->drop_gold) text_out(L" or ");
		}

		/* Treasures */
		if (l_ptr->drop_gold)
		{
			/* Dump "treasure(s)" */
			text_out(L"treasure%s", PLURAL(n));
		}

		/* End this sentence */
		text_out(L".  ");
	}
}

/*
 * Describe all of a monster's attacks.
 */
static void describe_monster_attack(int r_idx, const monster_lore *l_ptr)
{
	const monster_race *r_ptr = &r_info[r_idx];
	int m, n, r;
	u32b f[OBJ_FLAG_N];
	bool known;

	int msex = 0;

	int color_special[RBE_MAX+1];
	player_state st;

	calc_bonuses(inventory, &st, TRUE);

	/* Color-code special attacks.  Green is not special or resisted,
	 * yellow is unresisted attacks that make a monster more dangerous,
	 * red cannot be ignored regardless of power difference under
	 * threat of expensive and lasting character harm.
	 */

	/* All special attacks are assumed ineffective */
	for (m = 0; m <= RBE_MAX; m++)
		color_special[m] = TERM_L_GREEN;

	/* If you have an item that can be affected, item affecting attacks
	 * are bad. */
	for (m = 0; m < INVEN_TOTAL; m++)
	{
		object_type *o_ptr = &inventory[m];

		/* Only occupied slots */
		if (!o_ptr->k_idx) continue;

		object_flags_known(o_ptr, f);

		/* We need to be careful not to reveal the nature of the object
		 * here.  Assume the player is conservative with unknown items.
		 */
		known = object_known_p(o_ptr);

		/* Can only be hurt by disenchantment with an enchanted item */
		if (m >= INVEN_WIELD && (!known || (o_ptr->to_a >= 0) ||
				(o_ptr->to_h >= 0) || (o_ptr->to_d >= 0)) &&
				!st.resist_disen)
			color_special[RBE_UN_BONUS] = TERM_L_RED;

		/* A charged item is needed for drain charges */
		if (m < INVEN_PACK && (!known || o_ptr->pval > 0) &&
				(o_ptr->tval == TV_STAFF || o_ptr->tval == TV_WAND))
			color_special[RBE_UN_POWER] = TERM_L_RED;

		/* Non-artifacts are needed for theft */
		if (m < INVEN_PACK && (!known || !artifact_p(o_ptr)))
			color_special[RBE_EAT_ITEM] = TERM_L_RED;

		/* Characters without food do not suffer from eat food */
		if (m < INVEN_PACK && o_ptr->tval == TV_FOOD)
			color_special[RBE_EAT_FOOD] = TERM_YELLOW;

		/* A fueled lite is needed for eat light */
		if (m == INVEN_LITE && !(f[2] & TR2_NO_FUEL) &&
				(o_ptr->timeout > 0))
			color_special[RBE_EAT_LITE] = TERM_YELLOW;

		/* With corrodable equipment, acid is much worse */
		if (m >= INVEN_BODY && m <= INVEN_FEET &&
				(!known || o_ptr->ac + o_ptr->to_a > 0)
				&& !(f[2] & TR2_IGNORE_ACID) && !st.immune_acid)
			color_special[RBE_ACID] = TERM_L_RED;
	}

	/* If they can hurt the character, they're also bad, but this is in general
	 * less of a problem than long-term equipment damage.
	 */

	if (!st.resist_pois && !p_ptr->timed[TMD_OPP_POIS])
		color_special[RBE_POISON] = TERM_YELLOW;
	if (p_ptr->au)
		color_special[RBE_EAT_GOLD] = TERM_YELLOW;

	/* Theft has a general resistance */
	if (p_ptr->lev + adj_dex_safe[st.stat_ind[A_DEX]] >= 100)
	{
		color_special[RBE_EAT_GOLD] = TERM_L_GREEN;
		color_special[RBE_EAT_ITEM] = TERM_L_GREEN;
	}

	/* Acid can cause both wounding and equipment damage - always display
	 * the worse.
	 */

	if (!st.resist_acid && !st.immune_acid && !p_ptr->timed[TMD_OPP_ACID] &&
			(color_special[RBE_ACID] != TERM_L_RED))
		color_special[RBE_ACID] = TERM_YELLOW;
	if (!st.resist_fire && !st.immune_fire && !p_ptr->timed[TMD_OPP_FIRE])
		color_special[RBE_FIRE] = TERM_YELLOW;
	if (!st.resist_elec && !st.immune_elec && !p_ptr->timed[TMD_OPP_ELEC])
		color_special[RBE_ELEC] = TERM_YELLOW;
	if (!st.resist_cold && !st.immune_cold && !p_ptr->timed[TMD_OPP_COLD])
		color_special[RBE_COLD] = TERM_YELLOW;

	if (!p_ptr->state.resist_blind)
		color_special[RBE_BLIND] = TERM_YELLOW;
	if (!p_ptr->state.resist_confu)
		color_special[RBE_CONFUSE] = TERM_YELLOW;
	if (!p_ptr->state.resist_fear && p_ptr->state.skills[SKILL_SAVE] < 100)
		color_special[RBE_TERRIFY] = TERM_YELLOW;
	if (!p_ptr->state.resist_chaos)
		color_special[RBE_HALLU] = TERM_YELLOW;

	/* Paralysis gets an honorary permanent damage because it's so often
	 * an instakill.
	 */

	if (!st.free_act && st.skills[SKILL_SAVE] < 100)
		color_special[RBE_PARALYZE] = TERM_L_RED;

	/* These types of wounding are expensive to fix */

	if (!st.sustain_str)
		color_special[RBE_LOSE_STR] = TERM_L_RED;
	if (!st.sustain_int)
		color_special[RBE_LOSE_INT] = TERM_L_RED;
	if (!st.sustain_wis)
		color_special[RBE_LOSE_WIS] = TERM_L_RED;
	if (!st.sustain_dex)
		color_special[RBE_LOSE_DEX] = TERM_L_RED;
	if (!st.sustain_con)
		color_special[RBE_LOSE_CON] = TERM_L_RED;
	if (!st.sustain_chr)
		color_special[RBE_LOSE_CHR] = TERM_L_RED;
	if (!st.sustain_str || !st.sustain_int || !st.sustain_wis ||
			!st.sustain_dex || !st.sustain_con || !st.sustain_chr)
		color_special[RBE_LOSE_ALL] = TERM_L_RED;

	/* Hold life isn't 100% effective */
	color_special[RBE_EXP_10] = color_special[RBE_EXP_20] =
		color_special[RBE_EXP_40] = color_special[RBE_EXP_80] =
		st.hold_life ? TERM_YELLOW : TERM_L_RED;

	/* Shatter is always dangerous */
	color_special[RBE_SHATTER] = TERM_YELLOW;

	/* Extract a gender (if applicable) */
	if (r_ptr->flags[0] & RF0_FEMALE) msex = 2;
	else if (r_ptr->flags[0] & RF0_MALE) msex = 1;


	/* Count the number of "known" attacks */
	for (n = 0, m = 0; m < MONSTER_BLOW_MAX; m++)
	{
		/* Skip non-attacks */
		if (!r_ptr->blow[m].method) continue;

		/* Count known attacks */
		if (l_ptr->blows[m]) n++;
	}

	/* Examine (and count) the actual attacks */
	for (r = 0, m = 0; m < MONSTER_BLOW_MAX; m++)
	{
		int method, effect, d1, d2;
		const _TCHAR *p = NULL;
		const _TCHAR *q = NULL;

		/* Skip unknown and undefined attacks */
		if (!r_ptr->blow[m].method || !l_ptr->blows[m]) continue;


		/* Extract the attack info */
		method = r_ptr->blow[m].method;
		effect = r_ptr->blow[m].effect;
		d1 = r_ptr->blow[m].d_dice;
		d2 = r_ptr->blow[m].d_side;


		/* Get the method */
		switch (method)
		{
			case RBM_HIT:    p = L"hit"; break;
			case RBM_TOUCH:  p = L"touch"; break;
			case RBM_PUNCH:  p = L"punch"; break;
			case RBM_KICK:   p = L"kick"; break;
			case RBM_CLAW:   p = L"claw"; break;
			case RBM_BITE:   p = L"bite"; break;
			case RBM_STING:  p = L"sting"; break;
			case RBM_BUTT:   p = L"butt"; break;
			case RBM_CRUSH:  p = L"crush"; break;
			case RBM_ENGULF: p = L"engulf"; break;
			case RBM_CRAWL:  p = L"crawl on you"; break;
			case RBM_DROOL:  p = L"drool on you"; break;
			case RBM_SPIT:   p = L"spit"; break;
			case RBM_GAZE:   p = L"gaze"; break;
			case RBM_WAIL:   p = L"wail"; break;
			case RBM_SPORE:  p = L"release spores"; break;
			case RBM_BEG:    p = L"beg"; break;
			case RBM_INSULT: p = L"insult"; break;
			case RBM_MOAN:   p = L"moan"; break;
			case RBM_XXX1:
			case RBM_XXX2:
			case RBM_XXX3:
			case RBM_XXX4:
			case RBM_XXX5:
			default:		p = L"do something weird";
		}


		/* Get the effect */
		switch (effect)
		{
			case RBE_HURT:      q = __T("attack"); break;
			case RBE_POISON:    q = __T("poison"); break;
			case RBE_UN_BONUS:  q = __T("disenchant"); break;
			case RBE_UN_POWER:  q = __T("drain charges"); break;
			case RBE_EAT_GOLD:  q = __T("steal gold"); break;
			case RBE_EAT_ITEM:  q = __T("steal items"); break;
			case RBE_EAT_FOOD:  q = __T("eat your food"); break;
			case RBE_EAT_LITE:  q = __T("absorb light"); break;
			case RBE_ACID:      q = __T("shoot acid"); break;
			case RBE_ELEC:      q = __T("electrify"); break;
			case RBE_FIRE:      q = __T("burn"); break;
			case RBE_COLD:      q = __T("freeze"); break;
			case RBE_BLIND:     q = __T("blind"); break;
			case RBE_CONFUSE:   q = __T("confuse"); break;
			case RBE_TERRIFY:   q = __T("terrify"); break;
			case RBE_PARALYZE:  q = __T("paralyze"); break;
			case RBE_LOSE_STR:  q = __T("reduce strength"); break;
			case RBE_LOSE_INT:  q = __T("reduce intelligence"); break;
			case RBE_LOSE_WIS:  q = __T("reduce wisdom"); break;
			case RBE_LOSE_DEX:  q = __T("reduce dexterity"); break;
			case RBE_LOSE_CON:  q = __T("reduce constitution"); break;
			case RBE_LOSE_CHR:  q = __T("reduce charisma"); break;
			case RBE_LOSE_ALL:  q = __T("reduce all stats"); break;
			case RBE_SHATTER:   q = __T("shatter"); break;
			case RBE_EXP_10:    q = __T("lower experience"); break;
			case RBE_EXP_20:    q = __T("lower experience"); break;
			case RBE_EXP_40:    q = __T("lower experience"); break;
			case RBE_EXP_80:    q = __T("lower experience"); break;
			case RBE_HALLU:     q = __T("cause hallucinations"); break;
		}

		/* Introduce the attack description */
		if (!r)
			text_out(L"%^s can ", wd_he[msex]);
		else if (r < n - 1)
			text_out(L", ");
		else
			text_out(L", and ");

		/* Describe the method */
		text_out(p);

		/* Describe the effect (if any) */
		if (q)
		{
			/* Describe the attack type */
			text_out(L" to ");
			text_out_c(color_special[effect], L"%s", q);

			/* Describe damage (if known) */
			if (d1 && d2 && know_damage(r_idx, l_ptr, m))
			{
				text_out(L" with damage ");
				text_out_c(TERM_L_GREEN, L"%dd%d", d1, d2);
			}
		}

		/* Count the attacks as printed */
		r++;
	}

	/* Finish sentence above */
	if (r)
		text_out(L".  ");

	/* Notice lack of attacks */
	else if (l_ptr->flags[0] & RF0_NEVER_BLOW)
		text_out(L"%^s has no physical attacks.  ", wd_he[msex]);

	/* Or describe the lack of knowledge */
	else
		text_out(L"Nothing is known about %s attack.  ", wd_his[msex]);
}


/*
 * Describe special abilities of monsters.
 */
static void describe_monster_abilities(int r_idx, const monster_lore *l_ptr)
{
	const monster_race *r_ptr = &r_info[r_idx];

	int vn;
	const _TCHAR *vp[64];
	bool prev = FALSE;
	
	int msex = 0;


	/* Extract a gender (if applicable) */
	if (r_ptr->flags[0] & RF0_FEMALE) msex = 2;
	else if (r_ptr->flags[0] & RF0_MALE) msex = 1;


	/* Collect special abilities. */
	vn = 0;
	if (l_ptr->flags[1] & RF1_OPEN_DOOR) vp[vn++] = L"open doors";
	if (l_ptr->flags[1] & RF1_BASH_DOOR) vp[vn++] = L"bash down doors";
	if (l_ptr->flags[1] & RF1_PASS_WALL) vp[vn++] = L"pass through walls";
	if (l_ptr->flags[1] & RF1_KILL_WALL) vp[vn++] = L"bore through walls";
	if (l_ptr->flags[1] & RF1_MOVE_BODY) vp[vn++] = L"push past weaker monsters";
	if (l_ptr->flags[1] & RF1_KILL_BODY) vp[vn++] = L"destroy weaker monsters";
	if (l_ptr->flags[1] & RF1_TAKE_ITEM) vp[vn++] = L"pick up objects";
	if (l_ptr->flags[1] & RF1_KILL_ITEM) vp[vn++] = L"destroy objects";

	/* Describe special abilities. */
	output_desc_list(msex, L"can", vp, vn, TERM_WHITE);


	/* Describe detection traits */
	vn = 0;
	if (l_ptr->flags[1] & RF1_INVISIBLE)  vp[vn++] = L"invisible";
	if (l_ptr->flags[1] & RF1_COLD_BLOOD) vp[vn++] = L"cold blooded";
	if (l_ptr->flags[1] & RF1_EMPTY_MIND) vp[vn++] = L"not detected by telepathy";
	if (l_ptr->flags[1] & RF1_WEIRD_MIND) vp[vn++] = L"rarely detected by telepathy";

	output_desc_list(msex, L"is", vp, vn, TERM_WHITE);


	/* Describe special things */
	if (l_ptr->flags[1] & RF1_MULTIPLY)
		text_out(L"%^s breeds explosively.  ", wd_he[msex]);
	if (l_ptr->flags[1] & RF1_REGENERATE)
		text_out(L"%^s regenerates quickly.  ", wd_he[msex]);


	/* Collect susceptibilities */
	vn = 0;
	if (l_ptr->flags[2] & RF2_HURT_ROCK) vp[vn++] = L"rock remover";
	if (l_ptr->flags[2] & RF2_HURT_LITE) vp[vn++] = L"bright light";
	if (l_ptr->flags[2] & RF2_HURT_FIRE) vp[vn++] = L"fire";
	if (l_ptr->flags[2] & RF2_HURT_COLD) vp[vn++] = L"cold";

	if (vn)
	{
		/* Output connecting text */
		text_out(L"%^s is hurt by ", wd_he[msex]);
		output_list(vp, vn, TERM_YELLOW);
		prev = TRUE;
	}

	/* Collect immunities and resistances */
	vn = 0;
	if (l_ptr->flags[2] & RF2_IM_ACID)   vp[vn++] = L"acid";
	if (l_ptr->flags[2] & RF2_IM_ELEC)   vp[vn++] = L"lightning";
	if (l_ptr->flags[2] & RF2_IM_FIRE)   vp[vn++] = L"fire";
	if (l_ptr->flags[2] & RF2_IM_COLD)   vp[vn++] = L"cold";
	if (l_ptr->flags[2] & RF2_IM_POIS)   vp[vn++] = L"poison";
	if (l_ptr->flags[2] & RF2_IM_WATER)  vp[vn++] = L"water";
	if (l_ptr->flags[2] & RF2_RES_NETH)  vp[vn++] = L"nether";
	if (l_ptr->flags[2] & RF2_RES_PLAS)  vp[vn++] = L"plasma";
	if (l_ptr->flags[2] & RF2_RES_NEXUS) vp[vn++] = L"nexus";
	if (l_ptr->flags[2] & RF2_RES_DISE)  vp[vn++] = L"disenchantment";

	if (vn)
	{
		/* Output connecting text */
		if (prev)
			text_out(L", but resists ");
		else
			text_out(L"%^s resists ", wd_he[msex]);

		/* Write the text */
		output_list(vp, vn, TERM_ORANGE);
		prev = TRUE;
	}

	/* Collect non-effects */
	vn = 0;
	if (l_ptr->flags[2] & RF2_NO_STUN) vp[vn++] = L"stunned";
	if (l_ptr->flags[2] & RF2_NO_FEAR) vp[vn++] = L"frightened";
	if (l_ptr->flags[2] & RF2_NO_CONF) vp[vn++] = L"confused";
	if (l_ptr->flags[2] & RF2_NO_SLEEP) vp[vn++] = L"slept";

	if (vn)
	{
		/* Output connecting text */
		if (prev)
			text_out(L", and cannot be ");
		else
			text_out(L"%^s cannot be ", wd_he[msex]);

		output_list(vp, -vn, TERM_ORANGE);
		prev = TRUE;
	}


	/* Full stop. */
	if (prev) text_out(L".  ");


	/* Do we know how aware it is? */
	if ((((int)l_ptr->wake * (int)l_ptr->wake) > r_ptr->sleep) ||
	    (l_ptr->ignore == MAX_UCHAR) ||
	    ((r_ptr->sleep == 0) && (l_ptr->tkills >= 10)))
	{
		const _TCHAR *act;

		if (r_ptr->sleep > 200)     act = __T("prefers to ignore");
		else if (r_ptr->sleep > 95) act = __T("pays very little attention to");
		else if (r_ptr->sleep > 75) act = __T("pays little attention to");
		else if (r_ptr->sleep > 45) act = __T("tends to overlook");
		else if (r_ptr->sleep > 25) act = __T("takes quite a while to see");
		else if (r_ptr->sleep > 10) act = __T("takes a while to see");
		else if (r_ptr->sleep > 5)  act = __T("is fairly observant of");
		else if (r_ptr->sleep > 3)  act = __T("is observant of");
		else if (r_ptr->sleep > 1)  act = __T("is very observant of");
		else if (r_ptr->sleep > 0)  act = __T("is vigilant for");
		else                        act = __T("is ever vigilant for");

		text_out(L"%^s %s intruders, which %s may notice from ", wd_he[msex], act, wd_he[msex]);
		text_out_c(TERM_L_GREEN, L"%d", 10 * r_ptr->aaf);
		text_out(L" feet.  ");
	}

	/* Describe escorts */
	if ((l_ptr->flags[0] & (RF0_ESCORT | RF0_ESCORTS)))
	{
		text_out(L"%^s usually appears with escorts.  ", wd_he[msex]);
	}

	/* Describe friends */
	else if ((l_ptr->flags[0] & (RF0_FRIEND | RF0_FRIENDS)))
	{
		text_out(L"%^s usually appears in groups.  ", wd_he[msex]);
	}
}


/*
 * Describe how often the monster has killed/been killed.
 */
static void describe_monster_kills(int r_idx, const monster_lore *l_ptr)
{
	const monster_race *r_ptr = &r_info[r_idx];

	int msex = 0;

	bool out = TRUE;


	/* Extract a gender (if applicable) */
	if (r_ptr->flags[0] & RF0_FEMALE) msex = 2;
	else if (r_ptr->flags[0] & RF0_MALE) msex = 1;


	/* Treat uniques differently */
	if (l_ptr->flags[0] & RF0_UNIQUE)
	{
		/* Hack -- Determine if the unique is "dead" */
		bool dead = (r_ptr->max_num == 0) ? TRUE : FALSE;

		/* We've been killed... */
		if (l_ptr->deaths)
		{
			/* Killed ancestors */
			text_out(L"%^s has slain %d of your ancestors", wd_he[msex], l_ptr->deaths);


			/* But we've also killed it */
			if (dead)
				text_out(L", but you have taken revenge!  ");

			/* Unavenged (ever) */
			else
				text_out(L", who remain%s unavenged.  ", PLURAL(l_ptr->deaths));
		}

		/* Dead unique who never hurt us */
		else if (dead)
		{
			text_out(L"You have slain this foe.  ");
		}
		else
		{
			/* Alive and never killed us */
			out = FALSE;
		}
	}

	/* Not unique, but killed us */
	else if (l_ptr->deaths)
	{
		/* Dead ancestors */
		text_out(L"%d of your ancestors %s been killed by this creature, ",
		            l_ptr->deaths, plural(l_ptr->deaths, L"has", L"have"));

		/* Some kills this life */
		if (l_ptr->pkills)
		{
			text_out(L"and you have exterminated at least %d of the creatures.  ",
			            l_ptr->pkills);
		}

		/* Some kills past lives */
		else if (l_ptr->tkills)
		{
			text_out(L"and %s have exterminated at least %d of the creatures.  ",
			            L"your ancestors", l_ptr->tkills);
		}

		/* No kills */
		else
		{
			text_out_c(TERM_RED, L"and %s is not ever known to have been defeated.  ",
			            wd_he[msex]);
		}
	}

	/* Normal monsters */
	else
	{
		/* Killed some this life */
		if (l_ptr->pkills)
		{
			text_out(L"You have killed at least %d of these creatures.  ",
			            l_ptr->pkills);
		}

		/* Killed some last life */
		else if (l_ptr->tkills)
		{
			text_out(L"Your ancestors have killed at least %d of these creatures.  ",
			            l_ptr->tkills);
		}

		/* Killed none */
		else
		{
			text_out(L"No battles to the death are recalled.  ");
		}
	}

	/* Separate */
	if (out) text_out(L"\n");
}


/*
 * Note how tough a monster is.
 */
static void describe_monster_toughness(int r_idx, const monster_lore *l_ptr)
{
	const monster_race *r_ptr = &r_info[r_idx];

	int msex = 0;

	/* Extract a gender (if applicable) */
	if (r_ptr->flags[0] & RF0_FEMALE) msex = 2;
	else if (r_ptr->flags[0] & RF0_MALE) msex = 1;
	

	/* Describe monster "toughness" */
	if (know_armour(r_idx, l_ptr))
	{
		/* Armor */
		text_out(L"%^s has an armor rating of ", wd_he[msex]);
		text_out_c(TERM_L_GREEN, L"%d", r_ptr->ac);
		text_out(L", and a");

		if (!(l_ptr->flags[0] & RF0_UNIQUE))
			text_out(L"n average");

		text_out(L" life rating of ");
		text_out_c(TERM_L_GREEN, L"%d", r_ptr->avg_hp);
		text_out(L".  ");
	}
}


static void describe_monster_exp(int r_idx, const monster_lore *l_ptr)
{
	const monster_race *r_ptr = &r_info[r_idx];

	const _TCHAR *p;
	const _TCHAR *q;

	long i, j;

	_TCHAR buf[20] = __T("");

	/* Introduction */
	if (l_ptr->flags[0] & RF0_UNIQUE)
		text_out(L"Killing");
	else
		text_out(L"A kill of");

	text_out(L" this creature");

	/* calculate the integer exp part */
	i = (long)r_ptr->mexp * r_ptr->level / p_ptr->lev;

	/* calculate the fractional exp part scaled by 100, */
	/* must use long arithmetic to avoid overflow */
	j = ((((long)r_ptr->mexp * r_ptr->level % p_ptr->lev) *
		  (long)1000 / p_ptr->lev + 5) / 10);

	/* Calculate textual representation */
	strnfmt(buf, _countof(buf), L"%ld", (long)i);
	if (j)
		_tcscat_s(buf, _countof(buf), format(__T(".%02ld"), (long)j));

	/* Mention the experience */
	text_out(L" is worth ");
	text_out_c(TERM_ORANGE, buf);
	text_out(L" point%s", PLURAL((i == 1) && (j == 0)));

	/* Take account of annoying English */
	p = L"th";
	i = p_ptr->lev % 10;
	if ((p_ptr->lev / 10) == 1) /* nothing */;
	else if (i == 1) p = L"st";
	else if (i == 2) p = L"nd";
	else if (i == 3) p = L"rd";

	/* Take account of "leading vowels" in numbers */
	q = __T("");
	i = p_ptr->lev;
	if ((i == 8) || (i == 11) || (i == 18)) 
		q = __T("n");

	/* Mention the dependance on the player's level */
	text_out(L" for a%s %lu%s level character.  ", q, (long)i, p);
}


static void describe_monster_movement(int r_idx, const monster_lore *l_ptr)
{
	const monster_race *r_ptr = &r_info[r_idx];

	bool old = FALSE;


	text_out(L"This");

	if (r_ptr->flags[2] & RF2_ANIMAL) text_out(L" natural");
	if (r_ptr->flags[2] & RF2_EVIL) text_out(L" evil");
	if (r_ptr->flags[2] & RF2_UNDEAD) text_out(L" undead");
	if (r_ptr->flags[2] & RF2_METAL) text_out(L" metal");

	if (r_ptr->flags[2] & RF2_DRAGON) text_out(L" dragon");
	else if (r_ptr->flags[2] & RF2_DEMON) text_out(L" demon");
	else if (r_ptr->flags[2] & RF2_GIANT) text_out(L" giant");
	else if (r_ptr->flags[2] & RF2_TROLL) text_out(L" troll");
	else if (r_ptr->flags[2] & RF2_ORC) text_out(L" orc");
	else text_out(L" creature");

	/* Describe location */
	if (r_ptr->level == 0)
	{
		text_out(L" lives in the town");
		old = TRUE;
	}
	else
	{
		byte colour = (r_ptr->level > p_ptr->max_depth) ? TERM_RED : TERM_L_GREEN;

		if (l_ptr->flags[0] & RF0_FORCE_DEPTH)
			text_out(L" is found ");
		else
			text_out(L" is normally found ");

		text_out(L"at depths of ");
		text_out_c(colour, L"%d", r_ptr->level * 50);
		text_out(L" feet (level ");
		text_out_c(colour, L"%d", r_ptr->level);
		text_out(L")");

		old = TRUE;
	}

	if (old) text_out(L", and");

	text_out(L" moves");

	/* Random-ness */
	if ((l_ptr->flags[0] & (RF0_RAND_50 | RF0_RAND_25)))
	{
		/* Adverb */
		if ((l_ptr->flags[0] & RF0_RAND_50) && (l_ptr->flags[0] & RF0_RAND_25))
			text_out(L" extremely");
		else if (l_ptr->flags[0] & RF0_RAND_50)
			text_out(L" somewhat");
		else if (l_ptr->flags[0] & RF0_RAND_25)
			text_out(L" a bit");

		/* Adjective */
		text_out(L" erratically");

		/* Hack -- Occasional conjunction */
		if (r_ptr->speed != 110) text_out(L", and");
	}

	/* Speed */
	if (r_ptr->speed > 110)
	{
		if (r_ptr->speed > 130) text_out_c(TERM_GREEN, L" incredibly");
		else if (r_ptr->speed > 120) text_out_c(TERM_GREEN, L" very");
		text_out_c(TERM_GREEN, L" quickly");
	}
	else if (r_ptr->speed < 110)
	{
		if (r_ptr->speed < 90) text_out_c(TERM_GREEN, L" incredibly");
		else if (r_ptr->speed < 100) text_out_c(TERM_GREEN, L" very");
		text_out_c(TERM_GREEN, L" slowly");
	}
	else
	{
		text_out(L" at ");
		text_out_c(TERM_GREEN, L"normal speed");
	}

	/* The code above includes "attack speed" */
	if (l_ptr->flags[0] & RF0_NEVER_MOVE)
		text_out(L", but does not deign to chase intruders");

	/* End this sentence */
	text_out(L".  ");
}


/*
 * Learn everything about a monster (by cheating)
 */
static void cheat_monster_lore(int r_idx, monster_lore *l_ptr)
{
	const monster_race *r_ptr = &r_info[r_idx];

	int i;


	/* Hack -- Maximal kills */
	l_ptr->tkills = MAX_SHORT;

	/* Hack -- Maximal info */
	l_ptr->wake = l_ptr->ignore = MAX_UCHAR;

	/* Observe "maximal" attacks */
	for (i = 0; i < MONSTER_BLOW_MAX; i++)
	{
		/* Examine "actual" blows */
		if (r_ptr->blow[i].effect || r_ptr->blow[i].method)
		{
			/* Hack -- maximal observations */
			l_ptr->blows[i] = MAX_UCHAR;
		}
	}

	/* Hack -- maximal drops */
	if (r_ptr->flags[0] & RF0_DROP_4)
		l_ptr->drop_item = 6;
	else if (r_ptr->flags[0] & RF0_DROP_3)
		l_ptr->drop_item = 4;
	else if (r_ptr->flags[0] & RF0_DROP_2)
		l_ptr->drop_item = 3;
	else if (r_ptr->flags[0] & RF0_DROP_1)
		l_ptr->drop_item = 3;

	if (r_ptr->flags[0] & RF0_DROP_40)
		l_ptr->drop_item++;
	if (r_ptr->flags[0] & RF0_DROP_60)
		l_ptr->drop_item++;

	l_ptr->drop_gold = l_ptr->drop_item;

	/* Hack -- but only "valid" drops */
	if (r_ptr->flags[0] & RF0_ONLY_GOLD) l_ptr->drop_item = 0;
	if (r_ptr->flags[0] & RF0_ONLY_ITEM) l_ptr->drop_gold = 0;

	/* Hack -- observe many spells */
	l_ptr->cast_innate = MAX_UCHAR;
	l_ptr->cast_spell = MAX_UCHAR;

	/* Hack -- know all the flags */
	race_flags_assign(l_ptr->flags, r_ptr->flags);
	race_flags_assign_spell(l_ptr->spell_flags, r_ptr->spell_flags);
}


/*
 * Hack -- display monster information using "roff()"
 *
 * Note that there is now a compiler option to only read the monster
 * descriptions from the raw file when they are actually needed, which
 * saves about 60K of memory at the cost of disk access during monster
 * recall, which is optional to the user.
 *
 * This function should only be called with the cursor placed at the
 * left edge of the screen, on a cleared line, in which the recall is
 * to take place.  One extra blank line is left after the recall.
 */
void describe_monster(int r_idx, bool spoilers)
{
	monster_lore lore;

	/* Get the race and lore */
	const monster_race *r_ptr = &r_info[r_idx];
	monster_lore *l_ptr = &l_list[r_idx];


	/* Hack -- create a copy of the monster-memory */
	COPY(&lore, l_ptr, monster_lore);

	/* Assume some "obvious" flags */
	lore.flags[0] |= (r_ptr->flags[0] & RF0_OBVIOUS_MASK);


	/* Killing a monster reveals some properties */
	if (lore.tkills)
	{
		/* Know "race" flags */
		lore.flags[2] |= (r_ptr->flags[2] & RF2_RACE_MASK);

		/* Know "forced" flags */
		lore.flags[0] |= (r_ptr->flags[0] & (RF0_FORCE_DEPTH));
	}

	/* Cheat -- know everything */
	if (OPT(cheat_know) || spoilers) cheat_monster_lore(r_idx, &lore);

	/* Show kills of monster vs. player(s) */
	if (!spoilers) describe_monster_kills(r_idx, &lore);

	/* Monster description */
	describe_monster_desc(r_idx);

	/* Describe the movement and level of the monster */
	describe_monster_movement(r_idx, &lore);

	if (!spoilers) describe_monster_exp(r_idx, &lore);
	describe_monster_spells(r_idx, &lore);
	if (!spoilers) describe_monster_toughness(r_idx, &lore);
	/* Describe the abilities of the monster */
	describe_monster_abilities(r_idx, &lore);
	describe_monster_drop(r_idx, &lore);
	describe_monster_attack(r_idx, &lore);

	/* Notice "Quest" monsters */
	if (lore.flags[0] & RF0_QUESTOR)
		text_out(L"You feel an intense desire to kill this monster...  ");

	/* All done */
	text_out(L"\n");
}


/*
 * Hack -- Display the "name" and "attr/chars" of a monster race
 */
void roff_top(int r_idx)
{
	monster_race *r_ptr = &r_info[r_idx];

	byte a1, a2;
	_TCHAR c1, c2;


	/* Get the chars */
	c1 = r_ptr->d_char;
	c2 = r_ptr->x_char;

	/* Get the attrs */
	a1 = r_ptr->d_attr;
	a2 = r_ptr->x_attr;


	/* Clear the top line */
	Term_erase(0, 0, 255);

	/* Reset the cursor */
	Term_gotoxy(0, 0);

	/* A title (use "The" for non-uniques) */
	if (!(r_ptr->flags[0] & RF0_UNIQUE))
	{
		Term_addstr(-1, TERM_WHITE, L"The ");
	}

	/* Dump the name */
	Term_addstr(-1, TERM_WHITE, (r_name + r_ptr->name));

	/* Append the "standard" attr/_TCHAR info */
	Term_addstr(-1, TERM_WHITE, L" ('");
	Term_addch(a1, c1);
	Term_addstr(-1, TERM_WHITE, L"')");

	/* Append the "optional" attr/_TCHAR info */
	Term_addstr(-1, TERM_WHITE, L"/('");
	Term_addch(a2, c2);
	if (use_bigtile && (a2 & 0x80)) Term_addch(255, -1);
	Term_addstr(-1, TERM_WHITE, L"'):");
}


/*
 * Hack -- describe the given monster race at the top of the screen
 */
void screen_roff(int r_idx)
{
	/* Flush messages */
	message_flush();

	/* Begin recall */
	Term_erase(0, 1, 255);

	/* Output to the screen */
	text_out_hook = text_out_to_screen;

	/* Recall monster */
	describe_monster(r_idx, FALSE);

	/* Describe monster */
	roff_top(r_idx);
}


/*
 * Hack -- describe the given monster race in the current "term" window
 */
void display_roff(int r_idx)
{
	int y;

	/* Erase the window */
	for (y = 0; y < Term->hgt; y++)
	{
		/* Erase the line */
		Term_erase(0, y, 255);
	}
	/* Begin recall */
	Term_gotoxy(0, 1);

	/* Output to the screen */
	text_out_hook = text_out_to_screen;

	/* Recall monster */
	describe_monster(r_idx, FALSE);

	/* Describe monster */
	roff_top(r_idx);
}

/*
 * Return the r_idx of the monster with the given name.
 *
 * Note: on the rare occasions where two different entries have
 * the same name (such as novice mages), we return the index of the
 * first entry that matches. It would probably be a good idea in the
 * future to make sure all monsters have distinct names.
 */
int lookup_monster(const _TCHAR *name)
{
	int i;
	
	/* Look for it */
	for (i = 1; i < z_info->r_max; i++)
	{
		monster_race *r_ptr = &r_info[i];
		const _TCHAR *nm = r_name + r_ptr->name;
		
		/* Found a match */
		if (streq(name, nm))
			return i;
		
	} 
	return -1;
}
