//
// FTS client library
// Copyright (C) 2001 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation; either version 2.1
// of the License, or (at your option) any later version.
// 
// See file COPYING.LIB for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

using System;
using System.Text;

namespace ircam.fts.client
{
    /// <summary> A buffer of FtsAtom, i.e. int, double, strings and FtsObject
    /// 
    /// This class does not use standard Collection implementations in order
    /// to avoid allocations during message reception.
    /// 
    /// </summary>

    [Serializable]
    public class FtsArgs
    {
        private const int INITIAL_SIZE = 8;

        public FtsArgs()
        {
            size = INITIAL_SIZE;
            array = new FtsAtom[size];
            for (int i = 0; i < size; i++)
                array[i] = new FtsAtom();
            current = 0;
        }

        private void EnsureCapacity(int wanted)
        {
            if (current + wanted < size)
                return;

            while (size <= current + wanted)
                size *= 2;

            FtsAtom[] newArray = new FtsAtom[size];

            Array.Copy(array, 0, newArray, 0, current);

            for (int i = current; i < size; i++)
                newArray[i] = new FtsAtom();

            array = newArray;
        }

        public void AddBoolean(bool val)
        {
            AddInt(val ? 1 : 0);
        }

        public void AddInt(int val)
        {
            EnsureCapacity(1);
            array[current++].SetInt(val);
        }

        public void AddDouble(double val)
        {
            EnsureCapacity(1);
            array[current++].SetDouble(val);
        }

        /// <summary> replaced by AddDouble(double value) </summary>
        public void AddFloat(float val)
        {
            AddDouble((double)val);
        }

        public void AddSymbol(FtsSymbol val)
        {
            EnsureCapacity(1);
            array[current++].SetSymbol(val);
        }

        public void AddString(string val)
        {
            EnsureCapacity(1);
            array[current++].SetString(val);
        }

        public void AddRawString(string val)
        {
            EnsureCapacity(1);
            array[current++].SetRawString(val);
        }

        public void AddObject(FtsObject val)
        {
            EnsureCapacity(1);
            array[current++].SetObject(val);
        }

        public void Add(object val)
        {
            if (val is int)
                AddInt((int)val);
            else if (val is double)
                AddDouble((double)val);
            else if (val is float)
                AddDouble((double)val);
            else if (val is FtsSymbol)
                AddSymbol((FtsSymbol)val);
            else if (val is string)
                AddString((string)val);
            else if (val is FtsObject)
                AddObject((FtsObject)val);
        }



        public bool IsInt(int index)
        {
            return array[index].IsInt();
        }

        public bool IsDouble(int index)
        {
            return array[index].IsDouble();
        }

        /// <summary> replaced by IsDouble(int index) </summary>
        public bool IsFloat(int index)
        {
            return IsDouble(index);
        }

        public bool IsSymbol(int index)
        {
            return array[index].IsSymbol();
        }

        public bool IsString(int index)
        {
            return array[index].IsString();
        }

        public bool IsRawString(int index)
        {
            return array[index].IsRawString();
        }

        public bool IsObject(int index)
        {
            return array[index].IsObject();
        }



        public int GetInt(int index)
        {
            return array[index].intValue;
        }

        public double GetDouble(int index)
        {
            return array[index].doubleValue;
        }

        /// <summary> replaced by GetDouble(int index) </summary>
        public float GetFloat(int index)
        {
            return (float)array[index].doubleValue;
        }

        public FtsSymbol GetSymbol(int index)
        {
            return array[index].symbolValue;
        }

        public string GetString(int index)
        {
            return array[index].stringValue;
        }

        public string GetRawString(int index)
        {
            return array[index].stringValue;
        }

        public FtsObject GetObject(int index)
        {
            return array[index].objectValue;
        }


        public void Clear()
        {
            current = 0;
        }

        public FtsAtom[] Atoms
        {
            get
            {
                return array;
            }
        }

        public int Length
        {
            get
            {
                return current;
            }
        }

        public override string ToString()
        {
            StringBuilder b = new StringBuilder();
            FtsAtom[] atoms = this.Atoms;

            for (int i = 0; i < this.Length; i++)
            {
                b.Append(atoms[i] + " ");
            }
            return b.ToString();
        }

        private FtsAtom[] array;
        private int size;
        private int current;
    }
}