//
// FTS client library
// Copyright (C) 2001 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation; either version 2.1
// of the License, or (at your option) any later version.
// 
// See file COPYING.LIB for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

using System;

namespace ircam.fts.client
{
    /// <summary> A FtsAtom is a typed 'union', 
    /// holding the values transmitted other the client protocol</summary>

    [Serializable]
    public class FtsAtom
    {
        /// <summary> If the atom is a integer, holds the integer value </summary>
        public int intValue;

        /// <summary> If the atom is a double, holds the double value </summary>
        public double doubleValue;

        /// <summary> If the atom is a Symbol, holds the Symbol value </summary>
        public FtsSymbol symbolValue;

        /// <summary> If the atom is a String or a RawString, holds the String value </summary>
        public string stringValue;

        /// <summary> If the atom is a object, holds the object value </summary>
        public FtsObject objectValue;

        /// <summary> Tests if atom contains a 'void' value (i.e. no valid value) </summary>
        /// <returns> true if atom type is void </returns>
        public bool IsVoid()
        {
            return type == typeof(void);
        }

        /// <summary> Tests if atom contains an integer </summary>
        /// <returns> true if atom type is integer </returns>
        public bool IsInt()
        {
            return type == typeof(int);
        }

        /// <summary> Tests if atom contains a double </summary>
        /// <returns> true if atom type is double </returns>
        public bool IsDouble()
        {
            return type == typeof(double);
        }

        /// <summary> Tests if atom contains a FtsSymbol </summary>
        /// <returns> true if atom type is FtsSymbol </returns>
        public bool IsSymbol()
        {
            return type == typeof(FtsSymbol);
        }

        /// <summary> Tests if atom contains a String </summary>
        /// <returns> true if atom type is String </returns>
        public bool IsString()
        {
            return type == typeof(string);
        }

        /// <summary> Tests if atom contains a raw String </summary>
        /// <returns> true if atom type is RawString </returns>
        public bool IsRawString()
        {
            return type == typeof(IRawString);
        }

        /// <summary> Tests if atom contains a FtsObject or an instance of a derived class </summary>
        /// <returns> true if atom type is FtsObject or a derived class of FtsObject </returns>
        public bool IsObject()
        {
            // return true if the class of the contained object is a derived class of FtsObject
            return typeof(FtsObject).IsAssignableFrom(type);
        }

        /// <summary> Set the integer value </summary>
        /// <param name="i">the value </param>
        public void SetInt(int i)
        {
            type = typeof(int);
            intValue = i;
        }

        /// <summary> Set the double value </summary>
        /// <param name="f">the value </param>
        public void SetDouble(double f)
        {
            type = typeof(double);
            doubleValue = f;
        }

        /// <summary> Set the FtsSymbol value </summary>
        /// <param name="s">the value </param>
        public void SetSymbol(FtsSymbol s)
        {
            type = typeof(FtsSymbol);
            symbolValue = s;
        }

        /// <summary> Set the String value </summary>
        /// <param name="s">the value </param>
        public void SetString(string s)
        {
            type = typeof(string);
            stringValue = s;
        }

        /// <summary> Set the raw String value </summary>
        /// <param name="s">the value </param>
        public void SetRawString(string s)
        {
            type = typeof(IRawString);
            stringValue = s;
        }

        /// <summary> Set the FtsObject value </summary>
        /// <param name="o">the value </param>
        public void SetObject(FtsObject o)
        {
            if (o != null)
                type = o.GetType();
            else
                type = typeof(FtsObject);

            objectValue = o;
        }

        public object GetValue()
        {
            if (IsVoid())
                return null;
            else if (IsInt())
                return intValue;
            else if (IsDouble())
                return doubleValue;
            else if (IsString())
                return stringValue;
            else if (IsRawString())
                return stringValue;
            else if (IsSymbol())
                return symbolValue;
            else if (IsObject())
                return objectValue;

            return null;
        }

        public void SetValue(object val)
        {
            if (val is int)
                SetInt((int)val);
            else if (val is double)
                SetDouble((double)val);
            else if (val is FtsSymbol)
                SetSymbol((FtsSymbol)val);
            else if (val is string)
                SetString((string)val);
            else if (val is FtsObject)
                SetObject((FtsObject)val);
        }

        public override string ToString()
        {
            if (IsVoid())
                return "void";
            else if (IsInt())
                return Convert.ToString(intValue);
            else if (IsDouble())
                return Convert.ToString(doubleValue);
            else if (IsSymbol())
                return (symbolValue != null) ? symbolValue.ToString() : "null";
            else if (IsString() || IsRawString())
                return stringValue;
            else if (IsObject())
                return (objectValue != null) ? objectValue.ToString() : "null";

            return "unknown";
        }

        private Type type;
    }
}