//
// FTS client library
// Copyright (C) 2001 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation; either version 2.1
// of the License, or (at your option) any later version.
// 
// See file COPYING.LIB for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

using System;
using System.Collections;
using System.Threading;

namespace ircam.fts.client
{
    public class FtsServer
    {
        public FtsServer(FtsServerConnection connection)
        {
            this.connection = connection;

            newObjectID = 16; // Ids 0 to 15 are reserved for pre-defined system objects

            objectTable = new Hashtable();
            encoder = new BinaryProtocolEncoder(this.connection);
        }

        internal static int ROOT_OBJECT_ID = 0;
        internal static int CLIENT_OBJECT_ID = 1;

        public FtsObject RootObject
        {
            set
            {
                value.ID = ROOT_OBJECT_ID;
                AddObject(ROOT_OBJECT_ID, value);
            }
        }

        public FtsObject ClientObject
        {
            set
            {
                value.ID = CLIENT_OBJECT_ID;
                AddObject(CLIENT_OBJECT_ID, value);
            }
        }

        /// <summary> Start the receive thread
        /// 
        /// As soon as this method is called, message handlers will be called.
        /// This implies that the objects that may receive messages must be 
        /// already installed with their message handlers.
        /// </summary>
        public void Start()
        {
            thread = new Thread(
                delegate()
                {
                    BinaryProtocolDecoder decoder = new BinaryProtocolDecoder(this);
                    byte[] inputBuffer = new byte[0x10000];

                    try
                    {
                        while (true)
                        {
                            int len = connection.Read(inputBuffer, 0, inputBuffer.Length);
                            decoder.Decode(inputBuffer, 0, len);
                        }
                    }
                    catch (Exception e)
                    {
                        Console.Error.WriteLine("ReceiveThread: got exception {0}", e.ToString());
                        Console.Error.WriteLine(e.StackTrace);
                        return;
                    }
                });
            thread.Start();
        }

        /// <summary> Close the connection.
        /// 
        /// Closes the connection socket and wait for end of receive thread execution.
        /// Closing the socket will make the receive thread exit after a short while.
        /// </summary>
        public void Close()
        {
            connection.Close();
        }

        internal int NewObjectID
        {
            get
            {
                int id = newObjectID;

                newObjectID += 2;

                return id;
            }
        }

        internal FtsObject this[int id]
        {
            get
            {
                return (FtsObject)objectTable[id];
            }
            set
            {
                objectTable[id] = value;
            }
        }

        internal void AddObject(int id, FtsObject obj)
        {
            objectTable.Add(id, obj);
        }

        internal void RemoveObject(int id)
        {
            objectTable.Remove(id);
        }

        internal BinaryProtocolEncoder Encoder
        {
            get
            {
                return encoder;
            }
        }

        internal FtsServerConnection Connection
        {
            get
            {
                return connection;
            }
        }

        // Connection to FTS
        private FtsServerConnection connection;

        // Output to FTS
        private BinaryProtocolEncoder encoder;

        // Objects ID handling
        private int newObjectID;
        private Hashtable objectTable;

        private Thread thread;
    }
}