//
// FTS client library
// Copyright (C) 2001 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation; either version 2.1
// of the License, or (at your option) any later version.
// 
// See file COPYING.LIB for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

using System;
using System.IO;
using System.Threading;
using System.Net;
using System.Net.Sockets;

namespace ircam.fts.client
{
    public class FtsSocketConnection : FtsServerConnection
    {
        private const int DEFAULT_PORT = 2023;
        private const int DEFAULT_CONNECT_TIMEOUT = 30;

        /// <summary> 
        /// Constructs a new FtsSocketConnection to the specified host and port.
        /// Tries to connect to FTS. Retries until connection is made or timed'out.
        /// </summary>
        /// <param name="hostname">the host name </param>
        /// <param name="port">the port number </param>
        /// <param name="connectTimeout">the timeout value in seconds </param>
        public FtsSocketConnection(string hostname, int port, int connectTimeout)
        {
            do
            {
                try
                {
                    tcpClient = new TcpClient(hostname, port);
                }
                catch (SocketException e)
                {
                    Console.Error.WriteLine("SocketException: {0}", e);
                }

                if (tcpClient != null)
                    break;

                try
                {
                    Thread.Sleep(1000);
                }
                catch (ThreadInterruptedException e)
                {
                    Console.Error.WriteLine("ThreadInterruptedException: {0}", e);
                }

                connectTimeout--;
            } while (connectTimeout > 0);

            if (connectTimeout <= 0)
                throw new FtsClientException("Cannot connect");

            networkStream = tcpClient.GetStream();
        }

        public FtsSocketConnection(string hostname, int port)
            : this(hostname, port, DEFAULT_CONNECT_TIMEOUT) { }

        public FtsSocketConnection(string hostname)
            : this(hostname, DEFAULT_PORT, DEFAULT_CONNECT_TIMEOUT) { }

        public FtsSocketConnection()
            : this("127.0.0.1", DEFAULT_PORT, DEFAULT_CONNECT_TIMEOUT) { }

        public override void Close()
        {
            tcpClient.Close();
        }

        public override int Read(byte[] b, int off, int len)
        {
            return networkStream.Read(b, off, len);
        }

        public override void Write(byte[] b, int off, int len)
        {
            networkStream.Write(b, off, len);
        }

        private TcpClient tcpClient;
        private NetworkStream networkStream;
    }
}