//
// FTS client library
// Copyright (C) 2001 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation; either version 2.1
// of the License, or (at your option) any later version.
// 
// See file COPYING.LIB for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

using System;
using System.Threading;

using ircam.fts.client;

namespace TestClient
{
    public class FtsGraphicObject : FtsObject
    {
        public FtsGraphicObject(FtsServer server, FtsPatcher parent, int id, string className, FtsArgs args)
            : base(server, parent, id)
        {
            this.description = args.GetString(11);

            Console.WriteLine("FtsGraphicObject: className: {0} description: {1}", className, description);
        }

        public string Description
        {
            get
            {
                return description;
            }
        }

        private string description;
    }

    public class ConsoleStream : FtsObject
    {
        public ConsoleStream(FtsServer server, FtsObject rootPatcher)
            : base(server, rootPatcher, FtsSymbol.Get("console_stream"))
        {
            FtsObject.RegisterMessageHandler(this.GetType(), FtsSymbol.Get("print_line"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    Console.WriteLine("Console Stream: {0}", args.GetString(0));
                });
        }
    }

    public class FtsConnection : FtsObject
    {
        public FtsConnection(FtsServer server, FtsPatcher patcher, int id, FtsGraphicObject source,
                             int outlet, FtsGraphicObject destination, int inlet, int type)
            : base(server, patcher, id)
        {
            Console.WriteLine("Created a FtsConnection between: {0} outlet: {1} and {2} inlet: {3}",
                              source.Description, outlet, destination.Description, inlet);
        }
    }

    public class FtsPatcher : FtsObject
    {
        public FtsPatcher(FtsServer server, FtsObject parent, int id)
            : base(server, parent, id)
        {
            FtsObject.RegisterMessageHandler(this.GetType(), FtsSymbol.Get("startUpload"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    Console.WriteLine("FtsPatcher: startUpload");
                });
            FtsObject.RegisterMessageHandler(this.GetType(), FtsSymbol.Get("setPatcherBounds"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    int x = args.GetInt(0);
                    int y = args.GetInt(1);
                    int width = args.GetInt(2);
                    int height = args.GetInt(3);

                    Console.WriteLine("FtsPatcher: setPatcherBounds {0} {1} {2} {3}", x, y, width, height);
                });
            FtsObject.RegisterMessageHandler(this.GetType(), FtsSymbol.Get("addObject"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    int _id = args.GetInt(0);
                    string className = args.GetSymbol(9).ToString();

                    FtsGraphicObject graphicObject = new FtsGraphicObject(this.Server, this, _id, className, args);
                });
            FtsObject.RegisterMessageHandler(this.GetType(), FtsSymbol.Get("addConnection"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    if (args.Length == 6)
                    {
                        FtsConnection connection = new FtsConnection(this.Server,
                                                                     this,
                                                                     args.GetInt(0),
                                                                     (FtsGraphicObject)args.GetObject(1),
                                                                     args.GetInt(2),
                                                                     (FtsGraphicObject)args.GetObject(3),
                                                                     args.GetInt(4),
                                                                     args.GetInt(5));
                    }
                });
            FtsObject.RegisterMessageHandler(this.GetType(), FtsSymbol.Get("endUpload"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    Console.WriteLine("FtsPatcher: endUpload");
                });
            FtsObject.RegisterMessageHandler(this.GetType(), FtsSymbol.Get("openEditor"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    Console.WriteLine("FtsPatcher: openEditor");
                });
        }
    }

    public class ClientPatcher : FtsObject
    {
        public ClientPatcher(FtsServer server, FtsObject rootPatcher)
            : base(server, rootPatcher, 1)
        {
            this.server = server;
            this.rootPatcher = rootPatcher;

            FtsObject.RegisterMessageHandler(this.GetType(), FtsSymbol.Get("patcher_loaded"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    int id = args.GetInt(0);
                    string name = args.GetSymbol(1).ToString();
                    int type = args.GetInt(2);

                    Console.WriteLine("Client Patcher: patcher_loaded, id:{0} name: {1} type: {2}", id, name, type);
                    FtsPatcher patcher = new FtsPatcher(server, rootPatcher, id);
                });
        }

        private FtsServer server;
        private FtsObject rootPatcher;
    }


    public class TestClient
    {
        public TestClient(string[] args)
        {
            try
            {
                // etablish connection
                FtsServerConnection connection = new FtsSocketConnection();
                FtsServer server = new FtsServer(connection);

                // start the receive thread
                server.Start();

                // create root patcher
                FtsObject rootPatcher = new FtsObject(server, null, 0);

                // create console stream
                ConsoleStream consoleStream = new ConsoleStream(server, rootPatcher);
                consoleStream.Send(FtsSymbol.Get("set_default"));

                ClientPatcher clientPatcher = new ClientPatcher(server, rootPatcher);

                FtsArgs ftsArgs = new FtsArgs();

                // add path to patch to load
                ftsArgs.Clear();
                ftsArgs.Add(args[0]);

                // send load message to fts
                Console.WriteLine("Send message load: " + args[0]);
                clientPatcher.Send(FtsSymbol.Get("load"), ftsArgs);

                // sleep 5 second
                Thread.Sleep(5000);

                // try to shutdown fts
                Console.WriteLine("Shutdown FTS");
                clientPatcher.Send(FtsSymbol.Get("shutdown"));

                // stop thread and close connection
                Console.WriteLine("stop thread and close connection");
                server.Close();
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        [STAThread]
        public static void Main(string[] args)
        {
            if (args.Length == 1)
            {
                new TestClient(args);
            }
            else
            {
                Console.WriteLine("usage: TestClient [[drive:][path]patchfile]");
            }
        }
    }
}