//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation; either version 2.1
// of the License, or (at your option) any later version.
// 
// See file COPYING.LIB for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

using System;
using System.IO;
using System.Drawing;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;

using ircam.fts.client;
using ircam.jmax.fts;
using ircam.jmax.dialogs;
using ircam.jmax.editors.console;
using ircam.jmax.editors.patcher;

namespace ircam.jmax
{
    internal class RootPatcher : FtsPatcherObject
    {
        internal RootPatcher(FtsServer server)
            : base(server, null, FtsObject.NO_ID, "jpatcher", null, 0, 0) { }

        public override void ReleaseObject(FtsObject obj)
        {
            obj.Dispose();
        }
    }

    internal class JMaxClient : FtsObject
    {
        internal JMaxClient(FtsServer server, FtsPatcherObject rootPatcher)
            : base(server, rootPatcher, FtsObject.NO_ID) { }

        internal void Load(string name)
        {
            FtsArgs args = new FtsArgs();

            args.AddString(name);

            Send(FtsSymbol.Get("load"), args);
        }

        internal void LoadProject(string name)
        {
            FtsArgs args = new FtsArgs();

            args.AddString(name);

            Send(FtsSymbol.Get("load_project"), args);
        }

        internal void LoadPackage(string name, string fileName)
        {
            FtsArgs args = new FtsArgs();

            args.AddString(name);
            args.AddString(fileName);

            Send(FtsSymbol.Get("load_package"), args);
        }

        internal void LoadSummary(string name)
        {
            FtsArgs args = new FtsArgs();

            args.AddSymbol(FtsSymbol.Get(name));

            Send(FtsSymbol.Get("load_summary"), args);
        }

        static JMaxClient()
        {
            FtsObject.RegisterMessageHandler(typeof(JMaxClient), FtsSymbol.Get("patcher_loaded"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    if (args.IsInt(0))
                    {
                        FtsPatcherObject patcher = new FtsPatcherObject(JMaxLibrary.FtsServer,
                                                                        JMaxLibrary.RootPatcher,
                                                                        args.GetInt(0), "jpatcher", null, 0, 0);
                        ErmesSketchWindow win = new ErmesSketchWindow(patcher);
                        string name = args.GetSymbol(1).ToString();
                        win.Text = name;
                        patcher.EditorForm = win;
                        patcher.Name = name;
                        patcher.Type = args.GetInt(2);
                    }
                });
            FtsObject.RegisterMessageHandler(typeof(JMaxClient), FtsSymbol.Get("project"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    if (args.IsInt(0))
                    {
                        JMaxLibrary.Project = new FtsProject(JMaxLibrary.FtsServer,
                                                             JMaxLibrary.RootPatcher,
                                                             args.GetInt(0));
                    }
                });
            FtsObject.RegisterMessageHandler(typeof(JMaxClient), FtsSymbol.Get("config"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    if (args.IsInt(0))
                    {
                        JMaxLibrary.Config = new FtsConfig(JMaxLibrary.FtsServer,
                                                           JMaxLibrary.RootPatcher,
                                                           args.GetInt(0));
                    }
                });
            FtsObject.RegisterMessageHandler(typeof(JMaxClient), FtsSymbol.Get("package"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    if (args.IsInt(0))
                    {
                        new FtsPackage(JMaxLibrary.FtsServer, JMaxLibrary.RootPatcher, args.GetInt(0));
                    }
                });
            FtsObject.RegisterMessageHandler(typeof(JMaxClient), FtsSymbol.Get("showMessage"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    if (args.IsSymbol(0))
                    {
                        DialogResult res = MessageBox.Show(args.GetSymbol(0).ToString(), "Warning",
                                                           MessageBoxButtons.OK, MessageBoxIcon.Information);
                    }
                });
        }
    }

    // JMaxLibrary.getProperty should disappear, and properties stored in the system
    // properties, that can *also* be loaded from a file.

    /// <summary> The main application class in jMax. Contains the global parameters 
    /// and/or the global functionalities of the system. 
    /// It is also the entry point for several TCL commands. It handles, for example:
    /// - the startup process, the TCL initialization
    /// - the FTS connection
    /// - the resource definition loading
    /// - a set of global functions related to the window list (stack, tile, close)
    /// - file format versions handling
    /// - system properties
    /// - creation of new windows (new, double-click on a subpatcher, etc.)
    /// - recent file history
    /// - quit of the application
    /// </summary>

    public sealed class JMaxLibrary
    {
        // **********************************************************************
        // Still ugly
        // **********************************************************************

        /// <summary> Quit verify if there is anything to save
        /// Currently, this is done in the wrong way: we get the list
        /// of windows from the MaxWindowManager, and for those that 
        /// are MaxEditor, we do the check;
        /// </summary>

        public static void Quit()
        {
            bool someOneNeedSave = false;
            bool doTheSave = false;
            FtsPatcherObject patcher;

            // First, search if there is anything to save
            // Loop in all the documents in all the  types.

            List<Form> windows = MaxWindowManager.WindowManager.WindowList;
            
            search: for (int i = 0; i < windows.Count; i++)
            {
                Form win = (Form)windows[i];
                if (win is ErmesSketchWindow)
                {
                    patcher = ((ErmesSketchWindow)win).SketchPad.FtsPatcher;

                    if (patcher.IsDirty)
                    {
                        someOneNeedSave = true;
                        goto search;
                    }
                }
            }

            // in such case, should give the offer to cancel the quit.
            if (someOneNeedSave)
            {
                DialogResult result = MessageBox.Show("There are unsaved documents; you really want to Quit ?",
                                                      "Quit",
                                                      MessageBoxButtons.YesNoCancel,
                                                      MessageBoxIcon.Question,
                                                      MessageBoxDefaultButton.Button3);
                if (result == DialogResult.Yes)
                {
                    doTheSave = false;
                }
                if (result == DialogResult.No)
                {
                    doTheSave = true;
                }
                if (result == DialogResult.Cancel)
                {
                    return;
                }
            }

            // Dispose (and optionally save) all the documents
            CloseAllWindows(doTheSave);

            //Look if current project needs to be saved
            if (Project.IsDirty)
            {
                DialogResult result = MessageBox.Show("Project File is not saved.\nDo you want to save it now?",
                                                      "Project Not Saved",
                                                      MessageBoxButtons.YesNoCancel,
                                                      MessageBoxIcon.Question,
                                                      MessageBoxDefaultButton.Button3);
                if (result == DialogResult.Yes)
                {
                    Project.Save(Project.FileName);
                }
            }

            //Look if current audio/midi configuration needs to be saved
            if (Config.IsDirty && (Config.FileName != null))
            {
                DialogResult result = MessageBox.Show("Configuration File is not saved.\nDo you want to save it now?",
                                                      "Config Not Saved",
                                                      MessageBoxButtons.YesNoCancel,
                                                      MessageBoxIcon.Question,
                                                      MessageBoxDefaultButton.Button3);
                if (result == DialogResult.Yes)
                {
                    Config.Save(Config.FileName);
                }
            }
            /////////////////////////////////////////////////////////////////////////

            singleInstance.recentFileHistory.Save();

            if (MaxWindowManager.TopForm != null)
            {
                MaxWindowManager.TopForm.Visible = false;
                MaxWindowManager.TopForm.Dispose();
            }

            if (singleInstance.killFtsOnQuit)
                if (singleInstance.server != null)
                    try
                    {
                        singleInstance.clientObject.Send(FtsSymbol.Get("shutdown"));
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("JMaxLibrary: IOEception quitting application \n{0}", e.ToString());
                    }

            if (!singleInstance.isAttached)
            {
                FtsErrorStreamer.StopFtsErrorStreamer();
            }

            Application.Exit();
        }

        public static void CloseAllWindows(bool save)
        {
            Form win;
            FtsPatcherObject patcher;
            List<FtsPatcherObject> patches = new List<FtsPatcherObject>();
            object[] windows = MaxWindowManager.WindowManager.WindowListArrayCopy;

            for (int i = 0; i < windows.Length; i++)
            {
                win = (Form)windows[i];
                if (win is ErmesSketchWindow)
                {
                    patcher = ((ErmesSketchWindow)win).SketchPad.FtsPatcher;

                    if (patcher.IsARootPatcher)
                    {
                        if (save && patcher.IsDirty)
                        {
                            if (patcher.CanSave)
                                patcher.Save();
                            else
                            {
                                string fileName = MaxFileChooser.ChooseFileToSave(null, null, "Save As", MaxFileChooser.JMAX_FILE_TYPE);

                                if (fileName != null)
                                {
                                    patcher.Save(MaxFileChooser.SaveType, fileName);
                                }
                            }
                        }
                        if (patcher.Name != null)
                            patches.Add(patcher);

                        patcher.StopUpdates();
                        patcher.RequestDestroyEditor();
                        RootPatcher.RequestDeleteObject(patcher);
                        ((ErmesSketchWindow)win).Dispose();
                    }
                }
            }
            /* save the open windows in the project */
            if (patches.Count > 0)
                Project.SaveWindows(patches.GetEnumerator());
        }

        private static bool shown = false;

        public static void FtsQuitted()
        {
            if (!shown)
            {
                shown = true;
                MessageBox.Show("jMax for .NET server connection lost! \n Quit user interface.",
                                "Fatal Error",
                                MessageBoxButtons.OK,
                                MessageBoxIcon.Error);

                FtsErrorStreamer.StopFtsErrorStreamer();
                Application.Exit();
            }
        }

        public static void Load(string name)
        {
            singleInstance.clientObject.Load(name);
        }

        public static void LoadProject(string name)
        {
            singleInstance.clientObject.LoadProject(name);
        }

        public static void LoadPackage(string name, string fileName)
        {
            singleInstance.clientObject.LoadPackage(name, fileName);
        }

        public static void LoadSummary(string name)
        {
            singleInstance.clientObject.LoadSummary(name);
        }

        // **********************************************************************


        private static readonly JMaxLibrary singleInstance = new JMaxLibrary();

        // Static accessors
        public static JMaxLibrary SingleInstance
        {
            get
            {
                return singleInstance;
            }
        }

        public static FtsPatcherObject RootPatcher
        {
            get
            {
                return singleInstance.rootPatcher;
            }
        }

        public static RecentFileHistory RecentFileHistory
        {
            get
            {
                return singleInstance.recentFileHistory;
            }
        }

        public static FtsServer FtsServer
        {
            get
            {
                return singleInstance.server;
            }
        }

        public static FtsProject Project
        {
            get
            {
                return singleInstance.project;
            }
            set
            {
                /* discard all classes and helpPatch from old project */
                JMaxClassMap.Clear();
                FtsHelpPatchTable.Clear();

                /* set the current project */
                singleInstance.project = value;
            }
        }

        public static FtsConfig Config
        {
            get
            {
                return singleInstance.config;
            }
            set
            {
                singleInstance.config = value;
            }
        }

        public static FtsDspControl DspControl
        {
            get
            {
                return singleInstance.consoleWindow.DspControl;
            }
        }

        public static ConsoleWindow ConsoleWindow
        {
            get
            {
                return singleInstance.consoleWindow;
            }
        }

        public static IDictionary EnvironmentVariables
        {
            get
            {
                return singleInstance.environmentVariables;
            }
        }

        public static void ReportException(Exception e)
        {
            singleInstance._ReportException(e);
        }

        private JMaxLibrary()
        {
            environmentVariables = Environment.GetEnvironmentVariables();
            toOpen = new MaxVector();
        }

        public void Start(string[] args)
        {
            recentFileHistory = new RecentFileHistory(5);

            environmentVariables.Add("jmaxVersion", JMaxVersion.Version);

            ParseCommandLineOptions(args);

            GuessDirectories();

            ShowSplashScreen();

            OpenConsole();

            OpenConnection();

            CreatePredefinedObjects();

            InitConsole();

            StartReceive();

            // This should be really here, right before we eventually open command line documents
            RecentFileHistory.Load();

            OpenCommandLineFiles();

            Application.Run(consoleWindow);
        }


        private void ParseCommandLineOptions(string[] args)
        {
            //start parsing arguments
            // don't check for valid options, so the user can set
            // command line arguments that can be accessed from tcl scripts
            // and we don't know yet
            // The parsing is done with a finite state automata; any option
            // that don't have a value (i.e. followed by nothing or another value)
            // get "true" as value, any value that is *not* a option value
            // is interpreted as a file name and opened by jMax.

            bool inOpt = false;
            string option = null;

            for (int i = 0; i < args.Length; i++)
            {
                if (inOpt)
                {
                    // Waiting for an option's value

                    environmentVariables.Add(option.Substring(1), args[i]);
                    inOpt = false;
                }
                else
                {
                    // Waiting for a option
                    if (args[i].StartsWith("-"))
                    {
                        option = args[i];
                        inOpt = true;
                    }
                    else
                    {
                        toOpen.AddElement(args[i]);
                    }
                }
            }
        }

        private void GuessDirectories()
        {
            if (environmentVariables["jmaxRoot"] == null)
            {
                string root, currentDir;

                currentDir = Environment.CurrentDirectory;
                root = currentDir.Substring(0, currentDir.LastIndexOf("\\bin"));

                environmentVariables.Add("jmaxRoot", root);
                if (environmentVariables["jmaxServerDir"] == null)
                    environmentVariables.Add("jmaxServerDir", root + "\\bin");
            }
        }

        private void ShowSplashScreen()
        {
            if ((environmentVariables["jmaxSplashScreen"] == null) ||
                (!(environmentVariables["jmaxSplashScreen"]).Equals("hide")))
            {
                SplashDialog splashDialog = new SplashDialog();
                splashDialog.Show();
            }
        }

        private void OpenConsole()
        {
            consoleWindow = new ConsoleWindow("jMax Console");
            Console.SetOut(new ConsoleOutputStream(consoleWindow));
        }

        private void OpenConnection()
        {
            if (environmentVariables["jmaxConnection"] == null)
                environmentVariables.Add("jmaxConnection", "pipe");

            try
            {
                string cmd;
                string[] args = new string[10];
                int argc = 0;
                string connectionType = ((string)environmentVariables["jmaxConnection"]);
                string hostName = (string)environmentVariables["jmaxHost"];

                if (hostName == null)
                    hostName = "localhost";

                Console.Out.WriteLine("jMax starting server on " +
                                      hostName + " via " + connectionType + " connection");

                string ftsDir = (string)environmentVariables["jmaxServerDir"];
                string ftsName = (string)environmentVariables["jmaxServerName"];

                if (ftsName == null)
                    ftsName = "fts";

                cmd = ftsDir + "\\" + ftsName;

                if (connectionType.Equals("pipe"))
                    args[argc++] = "--stdio";

                // Support for project and configuration file given in command line
                if (environmentVariables["jmaxProject"] != null)
                    args[argc++] = "--project=" + ((string)environmentVariables["jmaxProject"]);

                if (environmentVariables["jmaxConfig"] != null)
                    args[argc++] = "--config=" + ((string)environmentVariables["jmaxConfig"]);

                object o = environmentVariables["attach"];

                FtsProcess fts = null;

                /* if(true) DEBUG ATTACH */
                if (o != null && ((string)o).Equals("true"))
                {
                    Console.Out.WriteLine("Attaching to FTS on host " + hostName);
                    killFtsOnQuit = true;
                    isAttached = true;
                }
                else
                {
                    fts = new FtsProcess(cmd, args);
                    //FtsErrorStreamer.StartFtsErrorStreamer(fts.StandardError);

                    killFtsOnQuit = true;
                }

                FtsServerConnection connection;

                /*if(false) DEBUG ATTACH*/
                if (connectionType.Equals("pipe"))
                {
                    if (fts == null)
                    {
                        Console.Error.WriteLine("Cannot attach to a pipe connection...");
                        Application.Exit();
                    }
                    connection = new FtsPipeConnection(fts);
                }
                else
                {
                    if (hostName == null)
                        connection = new FtsSocketConnection();
                    else if (environmentVariables["jmaxPort"] == null)
                        connection = new FtsSocketConnection(hostName);
                    else
                    {
                        int port = int.Parse((string)environmentVariables["jmaxPort"]);
                        connection = new FtsSocketConnection(hostName, port);
                    }
                }

                server = new FtsServer(connection);
            }
            catch (Exception e)
            {
                JMaxLibrary.ReportException(e);
            }
        }

        private void CreatePredefinedObjects()
        {
            rootPatcher = new RootPatcher(server);
            server.RootObject = rootPatcher;
            clientObject = new JMaxClient(server, rootPatcher);
            server.ClientObject = clientObject;
        }

        private void InitConsole()
        {
            FtsObject ftsConsole;

            try
            {
                ftsConsole = new FtsConsole(consoleWindow);
                ftsConsole.Send(FtsSymbol.Get("set_default"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("ConsoleWindow: Error in FtsConsole creation!");
                Console.Error.WriteLine(e.StackTrace);
            }

            consoleWindow.Init();
        }

        private void StartReceive()
        {
            server.Start();

            try
            {
                clientObject.Send(FtsSymbol.Get("get_project"));
            }
            catch (IOException e)
            {
                JMaxLibrary.ReportException(e);
            }

            try
            {
                clientObject.Send(FtsSymbol.Get("config"));
            }
            catch (IOException e)
            {
                JMaxLibrary.ReportException(e);
            }
        }

        private static void OpenCommandLineFiles()
        {
            string fileName = null;
            try
            {
                for (IEnumerator e = singleInstance.toOpen.Elements(); e.MoveNext(); )
                {
                    fileName = ((string)e.Current);
                    JMaxLibrary.Load(fileName);
                }
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("[JMaxLibrary]: I/O error loading file " + fileName);
            }
        }

        private void _ReportException(Exception e)
        {
            Console.Error.WriteLine("**************************************************");
            Console.Error.WriteLine(e.StackTrace);

            DialogResult res = MessageBox.Show("Caught .NET exception:" + e.StackTrace,
                                               ".NET exception",
                                               MessageBoxButtons.OK,
                                               MessageBoxIcon.Error);
        }

        private RecentFileHistory recentFileHistory;
        private FtsServer server;
        private IDictionary environmentVariables;
        private ConsoleWindow consoleWindow;
        private MaxVector toOpen;
        private bool killFtsOnQuit;
        private bool isAttached;
        private FtsPatcherObject rootPatcher;
        private JMaxClient clientObject;
        private FtsProject project;
        private FtsConfig config;
    }
}
