//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public License
// as published by the Free Software Foundation; either version 2.1
// of the License, or (at your option) any later version.
// 
// See file COPYING.LIB for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

using System;
using System.Text;
using System.Collections;

namespace ircam.jmax
{
    [Serializable]
    public class MaxVector
    {
        protected object[] objects;
        protected int fillPointer; // point to the next not used element

        public MaxVector()
        {
            this.objects = new object[8];
        }

        public void ensureCapacity(int requestedSize)
        {
            if (requestedSize >= objects.Length)
            {
                int newSize = objects.Length;

                while (newSize < requestedSize)
                    newSize = (3 * newSize) / 2;

                object[] oldObjects = objects;

                objects = new object[newSize];
                Array.Copy(oldObjects, 0, objects, 0, fillPointer);
            }
        }

        public int Size
        {
            get
            {
                return fillPointer;
            }
            set
            {
                if (value > objects.Length)
                    ensureCapacity(value);
                else
                {
                    for (int i = value; i < fillPointer; i++)
                        objects[i] = null;
                }

                fillPointer = value;
            }
        }

        public bool IsEmpty
        {
            get
            {
                return fillPointer == 0;
            }
        }

        [Serializable]
        internal sealed class MaxVectorEnumerator : IEnumerator
        {
            internal MaxVector maxVector;
            internal int count;

            internal MaxVectorEnumerator()
            {
                maxVector = new MaxVector();
                count = 0;
            }

            public bool MoveNext()
            {
                return count < maxVector.fillPointer;
            }

            public void Reset()
            {
                count = 0;
            }

            public object Current
            {
                get
                {
                    if (count < maxVector.fillPointer)
                    {
                        return maxVector.objects[count++];
                    }

                    throw new ArgumentOutOfRangeException("VectorEnumerator");
                }
            }
        }

        // Safe access

        public IEnumerator Elements()
        {
            return new MaxVectorEnumerator();
        }

        // Unsafe but fast access

        public object[] ObjectArray
        {
            get
            {
                return objects;
            }
        }

        public bool Contains(object elem)
        {
            if (elem == null)
            { //look for a null element:
                for (int i = 0; i < fillPointer; i++)
                    if (objects[i] == null)
                        return true;
            }
            else
            {
                for (int i = 0; i < fillPointer; i++)
                    if (elem.Equals(objects[i]))
                        return true;
            }
            return false;
        }

        public object this[int index]
        {
            get
            {
                if (index >= fillPointer)
                {
                    Console.Error.WriteLine("MaxVector: index {0} out of bounds ", index);
                    Console.Out.WriteLine(Environment.StackTrace);
                }

                return objects[index];
            }
            set
            {
                objects[index] = value;
            }
        }

        public void RemoveElementAt(int index)
        {
            int l = fillPointer - index - 1;

            if (l > 0)
                Array.Copy(objects, index + 1, objects, index, l);

            fillPointer--;
            objects[fillPointer] = null; /* to let gc do its work */
        }

        public void AddElement(object obj)
        {
            ensureCapacity(fillPointer + 1);
            objects[fillPointer++] = obj;
        }

        public void InsertElementAt(object obj, int index)
        {
            if (index < fillPointer + 1)
            {
                ensureCapacity(fillPointer + 1);
                Array.Copy(objects, index, objects, index + 1, fillPointer - index);
                objects[index] = obj;
                fillPointer++;
            }
        }

        public bool RemoveElement(object obj)
        {
            if (obj == null)
                return false;
            for (int i = 0; i < fillPointer; i++)
                if (obj.Equals(objects[i]))
                {
                    RemoveElementAt(i);
                    return true;
                }

            return false;
        }

        public void RemoveAllElements()
        {
            for (int i = 0; i < fillPointer; i++)
                objects[i] = null;

            fillPointer = 0;
        }

        public override string ToString()
        {
            int max = Size - 1;
            StringBuilder buf = new StringBuilder();
            buf.Append("MaxVector[" + Size + "]{");

            for (int i = 0; i < fillPointer; i++)
            {
                if (objects[i] != null)
                {
                    buf.Append(objects[i].ToString());
                }
                else
                    buf.Append("(null)");

                if (i < max)
                    buf.Append(", ");
            }

            buf.Append("}");

            return buf.ToString();
        }
    }
}
