using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using ircam.fts.client;
using ircam.jmax.fts;
using ircam.jmax.dialogs;
using ircam.jmax.editors.patcher;
using ircam.jmax.editors.project;
using ircam.jmax.editors.configuration;

namespace ircam.jmax.editors.console
{
	public partial class ConsoleWindow: Form
	{
        delegate void SetAppendTextCallback(string text);
        delegate void SetDspOnCallback(bool val);
        delegate bool GetDspOnCallback();

        private string helpFile = "jmax.chm";
        private FtsDspControl control;

        // (fd) This lock prevents a "setSelected" ont the dspOnButton to cause
        // "itemStateChanged" to repropagates the value to FTS
        private bool lockMode = false;

        public ConsoleWindow(string title)
		{
			InitializeComponent();

            this.Text = title;

            if ((JMaxLibrary.EnvironmentVariables["debug"] == null) ||
                ((JMaxLibrary.EnvironmentVariables["debug"]).Equals("true")))
            {
                this.Text += " - Debug Mode";
                this.printDsptoolBarButton.Visible = true;
            }
		}

        internal void NewPatcher()
        {
            FtsPatcherObject patcher = null;

            try
            {
                patcher = new FtsPatcherObject();
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("[NewAction]: Error in FtsPatcherObject creation!");
                Console.Error.WriteLine(e.StackTrace);
            }

            if (patcher != null)
            {
                patcher.RequestOpenEditor();
                patcher.RequestStopWaiting();
            }
        }

        private void Open()
        {
            string fileName;

            fileName = MaxFileChooser.ChooseFileToOpen(this);

            if (fileName != null)
            {
                RecentFileHistory recentFileHistory = JMaxLibrary.RecentFileHistory;
                recentFileHistory.AddFile(new FileInfo(fileName));

                try
                {
                    JMaxLibrary.Load(fileName);
                }
                catch (IOException e)
                {
                    Console.Error.WriteLine("[OpenAction]: I/O error loading file " + fileName);
                    Console.Error.WriteLine(e.StackTrace);
                }
            }
        }

        private void About()
        {
            AboutDialog ad = new AboutDialog();
            ad.ShowDialog(this);
            ad.Dispose();
        }

        public string AppendText
        {
            set
            {
                this.areaTextBox.AppendText(value);
            }
        }

        public bool DspOn
        {
            get
            {
                return this.GetDspOn();
            }
            set
            {
                this.SetDspOn(value);
            }
        }

        private bool GetDspOn()
        {
            if (this.controlPanelToolBar.InvokeRequired)
            {
                return (bool)this.Invoke(new GetDspOnCallback(GetDspOn));
            }
            else
            {
                return this.dspToolBarButton.Pushed;
            }
        }

        private void SetDspOn(bool val)
        {
            if (this.controlPanelToolBar.InvokeRequired)
            {
                SetDspOnCallback d = new SetDspOnCallback(SetDspOn);
                this.Invoke(d, new object[] { val });
            }
            else
            {
                dspToolBarButton.Pushed = val;
            }
        }

        public void Init()
        {
            try
            {
                control = new FtsDspControl();
                control.Send(FtsSymbol.Get("upload"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("ControlPanel: Error in FtsDspControl creation!");
                Console.Error.WriteLine(e.StackTrace);
            }

            if (JMaxLibrary.EnvironmentVariables.Contains("debug")) { }

            if (!lockMode)
            {
                if (!DspOn)
                {
                    control.RequestSetDspOn(false);
                }
                else
                {
                    control.RequestSetDspOn(true);
                }
            }
        }

        public bool LockMode
        {
            get
            {
                return lockMode;
            }
            set
            {
                lockMode = value;
            }
        }

        public FtsDspControl DspControl
        {
            get
            {
                return control;
            }
        }

        private void controlPanelToolBar_ButtonClick(object sender, System.Windows.Forms.ToolBarButtonClickEventArgs e)
        {
            if (e.Button == this.newToolBarButton)
            {
                NewPatcher();
            }
            else if (e.Button == this.openToolBarButton)
            {
                Open();
            }
            else if (e.Button == this.printDsptoolBarButton)
            {
                control.DspPrint();
            }
        }

        private void newPatcherMenuItem_Click(object sender, System.EventArgs e)
        {
            NewPatcher();
        }

        private void openMenuItem_Click(object sender, System.EventArgs e)
        {
            Open();
        }

        private void exitMenuItem_Click(object sender, System.EventArgs e)
        {
            JMaxLibrary.Quit();
        }

        private void newProjectMenuItem_Click(object sender, System.EventArgs e)
        {
            ProjectEditor.NewProject();
        }

        private void openProjectMenuItem_Click(object sender, System.EventArgs e)
        {
            ProjectEditor.OpenProject();
        }

        private void editProjectMenuItem_Click(object sender, System.EventArgs e)
        {
            JMaxLibrary.Project.RequestOpenEditor();
        }

        private void alwaysOnTopMenuItem_Click(object sender, System.EventArgs e)
        {
            this.TopMost = !this.TopMost;
        }

        private void contentsMenuItem_Click(object sender, System.EventArgs e)
        {
            Help.ShowHelp(this, helpFile);
        }

        private void aboutMenuItem_Click(object sender, System.EventArgs e)
        {
            About();
        }
	}
}